//using System;
//using System.Collections.Generic;
//using System.Text;
//using System.Net;
//using System.IO;
//using System.Text.RegularExpressions;
//using System.Runtime.Serialization.Formatters.Binary;
//using System.Runtime.Serialization;

package slothLib.web.webLoader;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.Proxy;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.net.*;

import slothLib.portUtil.PortUtil;

    /**
     * URLの内容を取得するクラス
     */
    public class WebGet
    {
        private Proxy proxy;
        private String userAgent;

        /**
         * コンストラクタ
         */
        public WebGet()
        {
            this.proxy = null;
            this.userAgent = "";
        }


        /**
         * URLのデータを取得する
         * @param url データを取得するURL
         * @param outputName 保存ファイル名
         * @return WebGetResult型の取得結果
         */
        public WebGetResult doFetch(String url, String outputName)
        {
            return this.doFetch(url, outputName, null, null);
        }

       
        /**
         * URLのデータを取得する
         * @param url データを取得するURL
         * @return WebGetResult型の取得結果
         */
        public WebGetResult doFetch(String url)
        {
            return this.doFetch(url,null,null);
        }
        
        /**
         * URLのデータを取得する
         * @param url データを取得するURL
         * @param allowContentTypes 取得するURLのCotentTypeのリスト
         * @param excludeContentTypes 取得しないURLのCotentTypeのリスト
         * @return WebGetResult型の取得結果
         */
        public WebGetResult doFetch(String url, String[] allowContentTypes, String[] excludeContentTypes)
        {
            return  this.doFetch(url,getTempFileName(url), allowContentTypes, excludeContentTypes);
        }

        /**
         * 複数URLのデータを取得する
         * @param urlList データを取得するURLのリスト
         * @param allowContentTypes 取得するURLのCotentTypeのリスト
         * @param excludeContentTypes 取得しないURLのCotentTypeのリスト
         * @return 
         */
        public MultiWebGetResults doFetch(String[] urlList, String[] allowContentTypes, String[] excludeContentTypes)
        {
            String[] tmpFileNameList = new String[urlList.length];

            for (int i = 0; i < urlList.length; i++)
            {
                tmpFileNameList[i] = getTempFileName(urlList[i]);
            }

            return this.doFetch(urlList, tmpFileNameList, allowContentTypes, excludeContentTypes);
        }

        /**
         * 複数URLのデータを取得する
         * @param urlList データを取得するURLのリスト
         * @return MultiWebGetResults型の結果
         */
        public MultiWebGetResults doFetch(String[] urlList)
        {
            return this.doFetch(urlList, null, null);
        }


        /**
         * URLのデータを取得する
         * @param url 取得するURL
         * @param outputName 出力ファイル名
         * @param allowContentTypes 取得するURLのContentType
         * @param excludeContentTypes 除外するURLのContentTypeのリスト
         * @return WebGetResult型の取得結果
         */
        public WebGetResult doFetch(String url, String outputName, 
        		String[] allowContentTypes, String[] excludeContentTypes)
        {
            // ヘッダ情報
            HttpHeader header = new HttpHeader();

            // 読込開始します。
            HttpURLConnection con;
        	String status = "";

            try {
            	if (proxy != null) 
            		con = (HttpURLConnection)(new URL(url)).openConnection(proxy);
            	else
            		con = (HttpURLConnection)(new URL(url)).openConnection();
            	
            	if (!PortUtil.isNullOrEmpty(this.userAgent))
            	{
            		//UserAgentの設定
            		con.setRequestProperty("User-Agent", this.userAgent);
            	}
            	
          	//using (HttpWebResponse res = (HttpWebResponse)req.GetResponse())
//                using (Stream st = res.GetResponseStream())
                //using (BinaryReader br = new BinaryReader(st))
//                using (FileStream fs = new FileStream(outputName, FileMode.Create))
                //using (BinaryWriter bw = new BinaryWriter(fs))
                {

                	status = con.getResponseMessage();
                	
                    // コンテンツタイプだけは必要ですので。
                    String contentType = con.getContentType();
                    String mainType = contentType.substring(0, contentType.indexOf("/"));
                    // エンコード情報が含まれて入れば、それを取り除く。
                    int iEncoding = contentType.indexOf(";");
                    if (iEncoding >= 0)
                    {
                        contentType = contentType.substring(0, iEncoding);
                    }

                    // ヘッダ情報を獲得
                    header = new HttpHeader(
                        PortUtil.getCharacterSet(con),
                        con.getContentEncoding(), 
                        con.getContentLength(), 
						//res.ContentType,
						contentType,
                        PortUtil.getCookies(con), 
                        new Date(con.getLastModified()), 
                        con.getRequestMethod(), 
                        PortUtil.getHttpVersion(con), 
                        con.getURL().toURI(),
                        con.getURL().getHost(), 
                        con.getResponseCode(), 
                        con.getResponseMessage()
                        );

                    // コンテンツタイプが対象のものかどうか調べる。

                    if (allowContentTypes != null)
                    {
                        List<String> allowList = new ArrayList<String>();
                        for (String type: allowContentTypes)
                        	allowList.add(type);
                        
                        	// 対象じゃなかったら，nullを返す
                        if (!allowList.contains(contentType) && !allowList.contains(mainType))
                        {
                            return null;
                        }
                    }

                    // コンテンツタイプが対象外かどうかを調べる。
                    if (excludeContentTypes != null)
                    {
                        List<String> excludeList = new ArrayList<String>();
                        for (String type: excludeContentTypes)
                        	excludeList.add(type);
                        	
                        //除外対象でもnullを返す
                        if (excludeList.contains(contentType) || excludeList.contains(mainType))
                        {
                            return null;
                        }
                    }

                    // バイナリでファイルに書き出し
                    InputStream is  = con.getInputStream();
                    FileOutputStream os = new FileOutputStream(outputName);
                    byte [] buf = new byte[8192];
                    while (true)
                    {
                    	int readCount = is.read(buf);
                        if (readCount < 0)
                        {
                            break;
                        }
                        System.out.println(readCount);
                        os.write(buf, 0, readCount);
                    }
                }
            } 
            catch (IOException ex){
            	ex.printStackTrace(System.err);
            	status += ":  " + ex.getMessage();
            }
            catch (Exception ex)
            {
                //何もできません
                ex.printStackTrace(System.err);
            }

            // 結果を作成
            WebGetResult re = new WebGetResult(url, outputName, header, status);

            //Debug.WriteLine("Response: "+status);
            return re;
        }

        /**
         * 複数URLのデータを取得する
         * @param urlList データを取得するURLのリスト
         * @param outputFileList 取得するURLのリストに対応した出力ファイル名のリスト
         * @param allowContentTypes 取得するURLのCotentTypeのリスト
         * @param excludeContentTypes 取得しないURLのCotentTypeのリスト
         */
        public MultiWebGetResults doFetch(String[] urlList, String[] outputFileList, String[] allowContentTypes, String[] excludeContentTypes)
        {
            List<WebGetResult> resultElements = new ArrayList<WebGetResult>();

            if (urlList.length < outputFileList.length)
            {
                System.err.println("WebGet:取得するURLリストと対応する出力ファイル名リストの要素数が対応していません");
                return null;
            }
            
            for (int index = 0; index < urlList.length; index++)
            {
                String url = urlList[index];
                String outputFile = outputFileList[index];

                WebGetResult result = this.doFetch(url, outputFile, allowContentTypes, excludeContentTypes);
                resultElements.add(result);
            }
            return new MultiWebGetResults(resultElements.toArray(new WebGetResult[0]));
        }


        /**
         * URLから一意のファイル名を得る
         * @param url 
         * @return 
         */
        private String getHashName(String url)
        {
            return url.hashCode() + "." +System.currentTimeMillis() + ".tmp";
        }

        /**
         * URLから一時ファイル名を得る
         * @param url 
         * @return 
         */
        private String getTempFileName(String url)
        {
            return System.getProperty("java.io.tmpdir") + "/" + getHashName(url);
        }

        /**
         * HTTP通信時に用いるproxyサーバ
         */
        public void setProxy(String value)
        {
        	if (value != null)
        	{
        		this.proxy = PortUtil.createProxy(value);
        	}
        	else
        	{
        		this.proxy = null;
        	}
        }
        public String getProxy()
        {
        	if (this.proxy != null)
        	{
        		return PortUtil.getProxyURL(this.proxy);
        	}
        	else
        	{
        		return "";
        	}
        }

        /**
         * HTTP通信時に用いるUserAgent
         */
        public void setUserAgent(String value)
        {
        	this.userAgent = value;
        }
        public String getUserAgent()
        {
        	return this.userAgent;
        }

    
    }

