package slothLib.web.search;

import java.io.UnsupportedEncodingException;
import java.net.Proxy;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import slothLib.portUtil.PortUtil;


/**
 * Yahoo!ウェブ検索Webサービス
 * http://developer.yahoo.co.jp/search/web/V1/webSearch.html
 * <remarks>
 * 

 * <newpara>[2007-03-04][ohshima]作成</newpara>
 * <newpara>[2007-05-09][ohshima]修正</newpara>
 * </remarks>
 */
public class YahooJpWebSearch implements IWebSearch
{

//		#region プライベートフィールド

	// アプリケーションID
	private String applicationID;

	// query と results(resultNumber) 以外の引数
	private SearchType type;
	private int start;
	private SearchFormat format;
	private boolean adultOk;
	private boolean similarOk;
	private String language;
	private String country;
	private String[] site;

	//結果が繰り返さないようにするために必要
	Map<String, Boolean> alreadyGotURL = new HashMap<String, Boolean>();

	// proxy
	private Proxy proxy;
	

	//	#endregion

		
//		#region コンストラクタ

	/**
	 * コンストラクタ
	 * アプリケーションIDを指定する
	 * @param applicationID プログラムで用いるアプリケーションID
	 */
	
	public YahooJpWebSearch(String applicationID)
	{
		this.applicationID = applicationID;
		
		// query と resultNumber 以外の引数
		this.type= SearchType.all;
		this.start= 1;
		this.format = SearchFormat.any;
		this.adultOk = false;
		this.similarOk = false;
		this.language = "ja";
		this.country = null;
		this.site = null;
	}
	

	
	
//	#region public メソッド
	
	/**
	 * Yahoo!ウェブ検索を行う
	 * @param query 検索クエリーです。このクエリーはYahoo!検索の全言語をサポートし、またメタキーワードも含みます。
	 * @param maxNumber 返却結果の数です。
	 * @return YahooJapanWebSearchResult型の検索結果
	 */
	public YahooJpWebSearchResult doSearch(String query, int maxNumber) throws WebSearchException
	{
		alreadyGotURL.clear();
		return doSearchOver(query, maxNumber);
	}


	/**
	 * 検索HIT数のみを取得するメソッド
	 * @param query 検索クエリ
	 * @return HIT数
	 */
	public long getTotalNumber(String query) throws WebSearchException
	{
		return doSearch(query, 1).getTotalResultsAvailable();
	}


	
	
	// プロパティ
	
	/**
	 * 指定検索の種類 デフォルトはall
	 */
	public SearchType getType()
	{
		return this.type;
	}
	
	public void setType(SearchType value){
		this.type = value;
	}
	
	/**
	 * 検索するファイルの種類　デフォルトはany
	 */
	public SearchFormat getFormat()
	{
		return this.format;
	}

	public	void setFormat(SearchFormat value)
	{
		this.format = value;
	}
	
	/**
	 * アダルトコンテンツの検索結果を含めるかどうかを指定します。デフォルトはfalse
	 */
	public boolean getAdultOk()
	{
		return this.adultOk;
	}
	public void setAdultOk(boolean value)
	{
		this.adultOk = value;
	}
	
	/**
	 * 同じコンテンツを別の検索結果とするかどうかを指定します。デフォルトはfalse
	 */
	public boolean getSimilarOk()
	{
		return this.similarOk;
	}
	public void setSimilarOk(boolean value) 
	{
		this.similarOk = value;
	}
	
	/**
	 * languageで書かれた結果になります。「サポートしている言語」をご参照ください。デフォルトは"ja"
	 */
	public String getLanguage()
	{
		return this.language;
	}
	public void setLanguage(String value) 
	{
		this.language = value;
	}
	
	/**
	 * ウェブサイトが位置する国の国コードです。デフォルトは空文字列
	 */
	public String getCountry()
	{
		return this.country;
	}
	public void setCountry(String value)
	{
		this.country = value;
	}
	
	/**
	 * >検索するドメイン（例えば www.yahoo.co.jp）を制限します。30ドメインまで指定することができます。デフォルトはnull
	 */
	public String[] getSite()
	{
		return this.site;
	}
	public void setSite(String[] value)
	{
		this.site = value;
	}
	

	
	
	// Proxy設定
	
	/**
	 * プロクシを取得・設定する。
	 */
	public void setProxy(String value)
	{
		if ( PortUtil.isNullOrEmpty(value))
		{
			this.proxy = null;
		}
		else
		{
			this.proxy = PortUtil.createProxy(value);
		}
	}
	public String getProxy()
	{
		return PortUtil.getProxyURL(this.proxy);
	}
	



//		#region フル機能の検索メソッド

        /**
		 * 50件までのYahoo!ウェブ検索を行う
		 * @param query 検索クエリーです。このクエリーはYahoo!検索の全言語をサポートし、またメタキーワードも含みます。
		 * @param results 返却結果の数です。
		 * @param start 返却結果の先頭位置です。最終位置（start + results - 1）は、1000を超えられません。
		 * @return YahooWebSearchResult型の検索結果
         */
		private YahooJpWebSearchResult doSearchOriginal(String query, int results, int start) throws WebSearchException
		{
			String requestURL = makeRequestURL(query, this.type, results, start, this.format, this.adultOk, this.similarOk, this.language, this.country, this.site);
			//System.Diagnostics.Debug.WriteLine(requestURL);
			System.err.println(requestURL);
			Document xmlDoc = PortUtil.getDocumentFromURL(requestURL, this.proxy);

			// ルートの要素を取得
			Element xmlRoot = xmlDoc.getDocumentElement();

			//<Result>要素のtotalResultsAvailableの属性値を取得 
			long totalResultsAvailable = Long.parseLong(xmlRoot.getAttribute("totalResultsAvailable"));
			int totalResultsReturned = Integer.parseInt(xmlRoot.getAttribute("totalResultsReturned"));
			int firstResultPosition = Integer.parseInt(xmlRoot.getAttribute("firstResultPosition"));

			// 検索結果要素を入れていく
			List<YahooJpWebElement> ResultElementList = new ArrayList<YahooJpWebElement>();

			NodeList xmlResultList = xmlRoot.getElementsByTagName("Result");

			int rank = start - 1;
            
            String firstURL = getElementString(((Element)xmlResultList.item(0)).getElementsByTagName("Url"));
            if(alreadyGotURL.containsKey(firstURL)){
                return new YahooJpWebSearchResult(query, 0, 0, -1,new YahooJpWebElement[0]);
            }else{
               alreadyGotURL.put(firstURL,true);
            }

			for (int i = 0; i < xmlResultList.getLength(); i++)
			{
				Element xmlResult = (Element)xmlResultList.item(i);
				rank++;
				String title = getElementString(xmlResult.getElementsByTagName("Title"));
				String summary = getElementString(xmlResult.getElementsByTagName("Summary"));
				String url = getElementString(xmlResult.getElementsByTagName("Url"));
				String date = getElementString(xmlResult.getElementsByTagName("ModificationDate"));
				String clickurl = getElementString(xmlResult.getElementsByTagName("ClickUrl"));
				String mimetype = getElementString(xmlResult.getElementsByTagName("MimeType"));
				String cacheUrl = null;
				String cacheSize = null;
				NodeList xmlCacheNode = xmlResult.getElementsByTagName("Cache");
				if (xmlCacheNode != null)
				{
					for (int j = 0; j < xmlCacheNode.getLength(); j++)
					{
						Element xmlCacheElement = (Element)xmlCacheNode.item(j);
						cacheUrl = getElementString(xmlCacheElement.getElementsByTagName("Url"));
						cacheSize = getElementString(xmlCacheElement.getElementsByTagName("Size"));
					}
				}
				YahooJpWebElement result = new YahooJpWebElement(rank, title, summary, url, clickurl, mimetype, date, cacheUrl, cacheSize);
				ResultElementList.add(result);
			}

			return new YahooJpWebSearchResult(query, totalResultsAvailable, totalResultsReturned, firstResultPosition, 
					(YahooJpWebElement[])ResultElementList.toArray(new YahooJpWebElement[0]));

		}


		/**
		 * Yahoo!ウェブ検索を行う
		 * @param query 検索クエリーです。このクエリーはYahoo!検索の全言語をサポートし、またメタキーワードも含みます。
		 * @param results 返却結果の数です。
		 * @return YahooWebSearchResult型の検索結果
		 */
		private YahooJpWebSearchResult doSearchOver(String query, int results) throws WebSearchException 
		{
			int i;
			int loop = (results - 1) / 50;
            //List<ISearchResultElement> resultElementList = new List<ISearchResultElement>();
            List<YahooJpWebElement> resultElementList = new ArrayList<YahooJpWebElement>();
            long totalResultsAvailable = 0;
			YahooJpWebSearchResult r;

			for (i = 0; i < loop; i++)
			{
                
				r = doSearchOriginal(query,  50, ((i * 50) + start));
                if (r.getFirstResultPosition() == -1) { break; }
                //resultElementList.AddRange(r);
                totalResultsAvailable = r.getTotalResultsAvailable();
                resultElementList.addAll(PortUtil.toList(r.getResultElements()));
			}
			r = doSearchOriginal(query, results - (loop * 50), loop * 50 + start);
            //resultElementList.AddRange(r);
            resultElementList.addAll(PortUtil.toList(r.getResultElements()));
            if (r.getFirstResultPosition() == -1)
            {
            }
            else
            {
                totalResultsAvailable = r.getTotalResultsAvailable();
            }
			return new YahooJpWebSearchResult(query, totalResultsAvailable, resultElementList.size(), 1, 
					resultElementList.toArray(new YahooJpWebElement[0]));
		}

	//	#endregion


//		#region 雑用メソッド

        /**
         * XmlNodeListの初めのノードのテキストを取得する
         * @param nodeList XmlNodeList
         * <returns>XmlNodeListの初めのノードのInnerText
         *          XmlNodeListが空であれば空文字列を返す</returns>
         */
        private String getElementString(NodeList nodeList)
        {
            if (nodeList.getLength() == 0)
            {
                return null;
            }
            else
            {
            	return nodeList.item(0).getTextContent();
            }
        }


		/**
		 * 例外を投げる
		 * @param errorCode 
		 * @param errorMessage 
		 */
//		private void throwException(int errorCode, String errorMessage) throws YahooJpSearchException
//		{
//			switch (errorCode)
//			{
//				case HttpURLConnection.HTTP_BAD_REQUEST: // 400
//					throw new YahooJpSearchException(YahooJpSearchException.HttpCode.BadRequest, errorMessage);
//				case HttpURLConnection.HTTP_FORBIDDEN: // 403
//					throw new YahooJpSearchException(YahooJpSearchException.HttpCode.Forbidden, errorMessage);
//				case HttpURLConnection.HTTP_UNAVAILABLE: // 503
//					throw new YahooJpSearchException(YahooJpSearchException.HttpCode.ServiceUnavailable, errorMessage);
//				default:
//                    throw new YahooJpSearchException(YahooJpSearchException.HttpCode.Unknown, "YahooWebSearchで想定外のHTTPエラーが発生しました。（エラーコード: " + (int)errorCode + "）" + errorMessage);
//            }
//		}

        ///// <summary>
        ///// XmlNodeListの初めのノードのテキストを取得する
        ///// </summary>
        ///// <param name="nodeList">XmlNodeList</param>
        ///// <returns>XmlNodeListの初めのノードのInnerText
        /////          XmlNodeListが空であれば空文字列を返す</returns>
        //private String GetElementString(XmlNodeList nodeList)
        //{
        //    if (nodeList.Count == 0)
        //    {
        //        return null;
        //    }
        //    else
        //    {
        //        return nodeList[0].InnerText;
        //    }
        //}

		/**
		 * リクエストURLを作成する
		 * @param query 検索クエリー
		 * @param type 指定検索の種類　デフォルトはall
		 * @param results 検索取得数
		 * @param start 検索開始の先頭位置
		 * @param format 検索するファイルの種類　デフォルトはany
		 * @param adultOk アダルトコンテンツの検索結果を含めるかどうか　デフォルトはfalse
		 * @param similarOk 同じコンテンツを別の検索結果とするかどうか　デフォルトはfalse
		 * @param language languageで書かれた結果になります デフォルトは空文字列
		 * @param country ウェブサイトが位置する国の国コード　デフォルトは空文字列
		 * <param name="site">検索するドメイン（例えば www.yahoo.co.jp）を制限します。30ドメインまで指定することができます
		 * デフォルトはnull</param>
		 * @return URL
		 */
		private String makeRequestURL(String query, SearchType type, int results, int start, SearchFormat format, boolean adultOk,
			boolean similarOk, String language, String country, String[] site)
		{
			String strType = "";
			String strFormat = "";
			String strAdult = "";
			String strSimilar = "";
			String strLanguage = "";
			String strCountry = "";
			String strSite = "";

			switch (type)
			{
				case all:
					break;
				case any:
					strType = "&type=any";
					break;
				case phrase:
					strType = "&type=phrase";
					break;
			}

			switch (format)
			{
				case any:
					break;
				case html:
					strFormat = "&format=html";
					break;
				case msword:
					strFormat = "&format=msword";
					break;
				case pdf:
					strFormat = "&format=pdf";
					break;
				case ppt:
					strFormat = "&format=ppt";
					break;
				case rss:
					strFormat = "&format=rss";
					break;
				case txt:
					strFormat = "&format=txt";
					break;
				case xls:
					strFormat = "&format=xls";
					break;
			}

			if (adultOk == true)
			{
				strAdult = "&adulat_ok=1";
			}

			if (similarOk == true)
			{
				strSimilar = "&similar_ok=1";
			}

			if (PortUtil.isNullOrEmpty(language))
			{
				strLanguage = "&language=ja";
			}
			else
			{
				strLanguage = "&language=" + language;
			}

			if (PortUtil.isNullOrEmpty(country))
			{
			}
			else
			{
				strCountry = "&country=" + country;
			}

			if (site != null)
			{
				if (site.length > 30)
				{
					throw new IllegalArgumentException("siteに指定できるドメインは30個までです " + site);
				}
				else
				{
					for (String s: site)
					{
						strSite += "&site=" + s;
					}
				}
			}

			String codedQuery = query;
			try {
				codedQuery = URLEncoder.encode(query, "UTF-8");
			} catch (UnsupportedEncodingException e) {
				System.err.println(e.getMessage());
			}
			
			String uriString = "http://api.search.yahoo.co.jp/WebSearchService/V1/webSearch?"
			+ "appid=" + this.applicationID
			//+ "&query=" + System.Web.HttpUtility.UrlEncode(query, Encoding.UTF8)
			+ "&query=" + codedQuery
			+ strType
			+ "&results=" + results
			+ "&start=" + start
			+ strFormat
			+ strAdult
			+ strSimilar
			+ strLanguage
			+ strCountry
			+ strSite;


			//return uriString;

			String result = PortUtil.absoluteUri(uriString);
			//System.Diagnostics.Debug.WriteLine(result);

			return result;
		}




		// 引数で指定するための列挙型

		/**
		 * 指定検索の種類
		 */
		public enum SearchType
		{
			/**
			 * 全クエリー文字を含む検索結果を返します（デフォルト）。
			 */
			all,
			/**
			 * クエリー文字のうちいずれかを含む検索結果を返します。
			 */
			any,
			/**
			 * クエリー文字を文章として含む検索結果を返します。
			 */
			phrase
		}

		/**
		 * 検索するファイルの種類 デフォルトはany
		 */
		public enum SearchFormat
		{
			/**
			 * デフォルト
			 */
			any,
			/**
			 * htmlファイル
			 */
			html,
			/**
			 * Wordファイル
			 */
			msword,
			/**
			 * PDFファイル
			 */
			pdf,
			/**
			 * PowerPointファイル
			 */
			ppt,
			/**
			 * RSSファイル
			 */
			rss,
			/**
			 * テキストファイル
			 */
			txt,
			/**
			 * Excelファイル
			 */
			xls
		}




		// IWebSearch メンバ

        /**
         * ahoo!ウェブ検索を行う
         * @param query 検索クエリーです。このクエリーはYahoo!検索の全言語をサポートし、またメタキーワードも含みます。
         * @param resultNum 返却結果の数です。
         * @return YahooJapanWebSearchResult型の検索結果
         */
//		IWebSearchResult IWebSearch.doSearch(String query, int resultNum)
		//{
//			return this.doSearch(query, resultNum);
//		}




		// ISearch メンバ

        /**
         * ahoo!ウェブ検索を行う
         * @param query 検索クエリーです。このクエリーはYahoo!検索の全言語をサポートし、またメタキーワードも含みます。
         * @param resultNum 返却結果の数です。
         * @return YahooJapanWebSearchResult型の検索結果
         */
//		ISearchResult ISearch.doSearch(String query, int resultNum)
//		{
//			return this.DoSearch(query, resultNum);
//		}



	}

