package slothLib.NLP;

import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.ArrayList;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.LineNumberReader;

/// <summary>
/// ストップワードを除外するフィルタ
/// </summary>
public class StopWordFilter extends AbstractStringFilter
{

	// ストップワードを保持するMap
	private Map<String, Boolean> wordList;
	
	// 記号を保持するList
	private List<String> symbolList;
	
	

	/// <summary>
	/// コンストラクタ
	/// </summary>
	public StopWordFilter()
	{
		wordList = new HashMap<String, Boolean>();
		symbolList = new ArrayList<String>();
	}
	
		

	/// <summary>
	/// 語(word)は完全一致で除外、記号(symbol)は部分一致で除外する
	/// </summary>
	/// <param name="str">フィルタを適用する文字列</param>
	/// <returns>フィルタ適用後の文字列</returns>
	public String doFilter(String str)
	{
		// 小文字にする
		String lower= str.toLowerCase();
		
		// リストチェック
		if (wordList.containsKey(lower))
		{
			return null;
		}
		
		// 記号チェック
		for (String s: symbolList)
		{
			// ストップワードと判定されたらさようなら。
			if (lower.indexOf(s) >= 0)
			{
				return null;
			}
		}
		
		// 合格
		return str;
	}
	
	
	

	/// <summary>
	/// 語をワードリストに追加する
	/// </summary>
	/// <param name="words">追加する語</param>
	public void addToWordList(String... words)
	{
		for (String word: words)
		{
			if ((word.length() > 0) && (!wordList.containsKey(word)))
			{
				// ワードリストに追加（このfalseには意味は全くない。）
				wordList.put(word, false);
			}
		}
	}
	
	/// <summary>
	/// 語をワードリストから削除する
	/// </summary>
	/// <param name="words">削除する語</param>
	public void removeFromWordList(String... words)
	{
		for (String word: words)
		{
			if (wordList.containsKey(word))
			{
				// ワードリストから削除
				wordList.remove(word);
			}
		}
	}
	
	/// <summary>
	/// ワードリストをクリアする
	/// </summary>
	public void clearWordList()
	{
		wordList.clear();
	}
	
	/// <summary>
	/// 記号を記号リストに追加する
	/// </summary>
	/// <param name="words">追加する記号</param>
	public void addToSymbolList(String... words)
	{
		for (String word: words)
		{
			if ((word.length() > 0) && (!symbolList.contains(word)))
			{
				// シンボルリストに追加
				symbolList.add(word);
			}
		}
	}
	
	/// <summary>
	/// 記号を記号リストから削除する
	/// </summary>
	/// <param name="words">削除する記号</param>
	public void removeFromSymbolList(String... words)
	{
		for (String word: words)
		{
			if (symbolList.contains(word))
			{
				// ワードリストから削除
				symbolList.remove(word);
			}
		}
	}
	
	/// <summary>
	/// 記号リストをクリアする
	/// </summary>
	public void clearSymbolList()
	{
		symbolList.clear();
	}

	/// <summary>
	/// UTF-8で保存されたストップワードのリストファイルを読み込む。
	/// 完全に一致する場合にストップワードと判定する。
	/// 英語は小文字に直してからリストの語と比較する。
	/// </summary>
	/// <param name="filePaths">ファイルのパスが入った配列</param>
	/// <returns>全て問題なく読み込みできた場合はtrue、なんらかの問題があった場合はfalseを返す。例外を返すことはあまりない。</returns>
	public boolean loadWordList(String[] filePaths)
	{
		boolean result = true;
		//wordList.Clear();
		
		for (String path: filePaths)
		{
			File f = new File(path);
			if (f.exists())
			{
				try
				{
					LineNumberReader sr = new LineNumberReader(new FileReader(f));
					String str;
					while ((str = sr.readLine()) != null)
					{
						if ((str.length() > 0) && (!wordList.containsKey(str)))
						{
							// ワードリストに追加（このfalseには意味は全くない。）
							wordList.put(str, false);
						}
					}
					sr.close();
				}
				catch (IOException e)
				{
					result = false;
				}
			}
			else
			{
				result = false;
				//throw new FileNotFoundException("指定されたストップワードリストのファイルが存在しません。", path);
			}
		}
		
		return result;
	}
	
	/// <summary>
	/// UTF-8で保存されたストップワードのリストファイルを読み込む。
	/// 完全に一致する場合にストップワードと判定する。
	/// 英語は小文字に直してからリストの語と比較する。
	/// </summary>
	/// <param name="dirPath">ストップワードを羅列したテキストファイルが保存されたディレクトリのパス</param>
	public void loadWordList(String dirPath) throws IOException
	{
		File dir = new File(dirPath);
		if (!dir.exists())
		{
			throw new FileNotFoundException("指定されたストップワードリストのディレクトリが存在しません。");
		}
		else
		{			
			loadWordList(dir.list());
		}
	}
	

	/// <summary>
	/// 記号リストをロードする。
	/// 記号が含まれる語は、問答無用でストップワードと判定する。
	/// </summary>
	/// <param name="filePaths">ファイルのパスが入った配列</param>
	/// <returns>全て問題なく読み込みできた場合はtrue、なんらかの問題があった場合はfalseを返す。例外を返すことはあまりない。</returns>
	public boolean loadSymbolList(String[] filePaths)
	{
		boolean result = true;
		//symbolList.clear();
		
		for (String path: filePaths)
		{
			File f = new File(path);
			if (f.exists())
			{
				try
				{
					LineNumberReader sr = new LineNumberReader(new FileReader(f));					
					String str;
					while ((str = sr.readLine()) != null)
					{
						if ((str.length() > 0) && (!symbolList.contains(str)))
						{
							// シンボルリストに追加
							symbolList.add(str);
						}
					}
					sr.close();
				}
				catch (IOException e){
					result = false;
				}
			}
			else
			{
				result = false;
				//throw new FileNotFoundException("指定された記号リストのファイルが存在しません。", path);
			}
		}
		return result;
	}
	
	/// <summary>
	/// 記号リストをロードする。
	/// 記号リストは
	/// 記号が含まれる語は、問答無用でストップワードと判定する。
	/// </summary>
	/// <param name="dirPath">記号を羅列したテキストファイルが保存されたディレクトリのパス</param>
	public void loadSymbolList(String dirPath) throws IOException
	{
		File dir = new File(dirPath);
		if (!dir.exists())
		{
			throw new FileNotFoundException("指定された記号リストのディレクトリが存在しません。");
		}
		else
		{
			loadSymbolList(dir.list());
		}
	}

	
}
