package slothLib.NLP;

import java.util.List;
import java.util.ArrayList;

/// <summary>
/// フィルタのリスト
/// </summary>

// it is not possible to conform the required interfaces in java
public class MorphemeFilterList  // implements IMorphemeFilter, IMorphemeToStringFilter, IStringFilter  
{
	private List<IMorphemeFilter> morphemeFilterList;
	private IMorphemeToStringFilter morphemeToStringFilter;
	private List<IStringFilter> stringFilterList;
	
	/// <summary>
	/// コンストラクタ
	/// </summary>
	public MorphemeFilterList()
	{
		this.morphemeFilterList = new ArrayList<IMorphemeFilter>();
		this.morphemeToStringFilter = null;
		this.stringFilterList = new ArrayList<IStringFilter>();
	}
	/// <summary>
	/// コンストラクタ
	/// </summary>
	/// <param name="morphemeFilters">入力が形態素で出力も形態素のフィルタのリスト</param>
	public MorphemeFilterList(Iterable<IMorphemeFilter> morphemeFilters)
	{
		this.morphemeFilterList = createArrayList(morphemeFilters);
		this.morphemeToStringFilter = null;
		this.stringFilterList = new ArrayList<IStringFilter>();
	}
	/// <summary>
	/// コンストラクタ
	/// </summary>
	/// <param name="morphemeToStringFilter">入力が形態素で出力が文字列フィルタ</param>
	public MorphemeFilterList(IMorphemeToStringFilter morphemeToStringFilter)
	{
		this.morphemeFilterList = new ArrayList<IMorphemeFilter>();
		this.morphemeToStringFilter = morphemeToStringFilter;
		this.stringFilterList = new ArrayList<IStringFilter>();
	}
	/// <summary>
	/// コンストラクタ
	/// </summary>
	/// <param name="StringFilters">入力が文字列で出力も文字列のフィルタのリスト</param>
	/// 
	/// due to 'same erasure' added dummy parameter 
	public MorphemeFilterList(Iterable<IStringFilter> stringFilters, Object dummy)
	{
		this.morphemeFilterList = new ArrayList<IMorphemeFilter>();
		this.morphemeToStringFilter = null;
		this.stringFilterList = createArrayList(stringFilters);
	}
	/// <summary>
	/// コンストラクタ
	/// </summary>
	/// <param name="morphemeFilters">入力が形態素で出力も形態素のフィルタのリスト</param>
	/// <param name="morphemeToStringFilter">入力が形態素で出力が文字列フィルタ</param>
	/// <param name="StringFilters">入力が文字列で出力も文字列のフィルタのリスト</param>
	public MorphemeFilterList(Iterable<IMorphemeFilter> morphemeFilters,
			IMorphemeToStringFilter morphemeToStringFilter,
			Iterable<IStringFilter> stringFilters)
	{
		this.morphemeFilterList = createArrayList(morphemeFilters);
		this.morphemeToStringFilter = morphemeToStringFilter;
		this.stringFilterList = createArrayList(stringFilters);
		
	}
	
	
	/// <summary>
	/// MorphemeToStringFilterが登録されているかどうかを取得
	/// </summary>
	public boolean hasMorphemeToStringFilter()
	{
		if (this.morphemeToStringFilter == null)
		{
			return false;
		}
		return true;
	}


	/// <summary>
	/// フィルタを追加する
	/// </summary>
	/// <param name="morphemeFilter">入力が形態素で出力も形態素のフィルタ</param>
	public void add(IMorphemeFilter morphemeFilter)
	{
		this.morphemeFilterList.add(morphemeFilter);
	}
	/// <summary>
	/// フィルタを追加する
	/// </summary>
	/// <param name="morphemeToStringFilter">入力が形態素で出力が文字列フィルタ</param>
	public void add(IMorphemeToStringFilter morphemeToStringFilter)
	{
		this.morphemeToStringFilter = morphemeToStringFilter;
	}
	/// <summary>
	/// フィルタを追加する
	/// </summary>
	/// <param name="StringFilter">入力が文字列で出力も文字列のフィルタ</param>
	public void add(IStringFilter StringFilter)
	{
		this.stringFilterList.add(StringFilter);
	}


	/// <summary>
	/// 入力が形態素で出力も形態素のフィルタを実行する
	/// </summary>
	/// <param name="morpheme">フィルタを適用する形態素</param>
	/// <returns>フィルタ適用後の形態素</returns>
	public IMorpheme doMorphemeFilter(IMorpheme morpheme)
	{
		IMorpheme resultMorpheme = morpheme;
		for (IMorphemeFilter filter: this.morphemeFilterList)
		{
			resultMorpheme = filter.doFilter(resultMorpheme);
			if (resultMorpheme == null)
			{
				return null; //結果がnullになったらその場でnullを返す
			}
		}
		return resultMorpheme;
	}
	/// <summary>
	/// 入力が形態素で出力も形態素のフィルタを実行する
	/// </summary>
	/// <param name="morphemes">フィルタを適用する形態素のリスト</param>
	/// <returns>フィルタ適用後の形態素の配列</returns>
	public IMorpheme[] doMorphemeFilter(Iterable<IMorpheme> morphemes)
	{
		IMorpheme[] resultMorphemes = createArrayList(morphemes).toArray(new IMorpheme[0]);
		for (IMorphemeFilter filter: this.morphemeFilterList)
		{
			resultMorphemes = filter.doFilter(resultMorphemes);
		}
		return resultMorphemes;
	}
	
	
	/// <summary>
	/// 入力が文字列で出力も文字列のフィルタを実行する
	/// </summary>
	/// <param name="str">フィルタを適用する文字列</param>
	/// <returns>フィルタ適用後の文字列</returns>
	public String doStringFilter(String str)
	{
		String resultString = str;
		for (IStringFilter filter: this.stringFilterList)
		{
			resultString = filter.doFilter(resultString);
			if (resultString == null)
			{
				return null; //もし結果がnullになったらその場でnullを返す
			}
		}
		return resultString;
	}
	/// <summary>
	/// 入力が文字列で出力も文字列のフィルタを実行する
	/// </summary>
	/// <param name="Strings">フィルタを適用する文字列のリスト</param>
	/// <returns>フィルタ適用後の文字列の配列</returns>
	public String[] doStringFilter(Iterable<String> strings)
	{
		String[] resultStrings = createArrayList(strings).toArray(new String[0]);
		return doStringFilter(resultStrings);
	}
	
	public String[] doStringFilter(String [] resultStrings)
	{
		for (IStringFilter filter: this.stringFilterList)
		{
			resultStrings = filter.doFilter(resultStrings);
		}
		return resultStrings;
	}
	
	
	//全てのフィルターを通す
	/// <summary>
	/// 登録されている全てのフィルタを適用する
	/// </summary>
	/// <param name="morpheme">フィルタを適用する形態素</param>
	/// <returns>フィルタ適用後の文字列</returns>
	public String doMorphemeToStringFilter(IMorpheme morpheme)
	{
		if (this.morphemeToStringFilter == null)
		{
			throw new RuntimeException("IMorphemeToStringFilterがありません");
		}
		morpheme = doMorphemeFilter(morpheme);
		if (morpheme == null) return null;
		String str = this.morphemeToStringFilter.doFilter(morpheme);
		if (str == null) return null;
		return doStringFilter(str);
	}
	/// <summary>
	/// 登録されている全てのフィルタを適用する
	/// </summary>
	/// <param name="morphemes">フィルタを適用する形態素のリスト</param>
	/// <returns>フィルタ適用後の文字列の配列</returns>
	public String[] doMorphemeToStringFilter(Iterable<IMorpheme> morphemes)
	{
		if (this.morphemeToStringFilter == null)
		{
			throw new RuntimeException("IMorphemeToStringFilterがありません");
		}
		IMorpheme tmpMorphemes [] = doMorphemeFilter(morphemes);
		String[] strings = this.morphemeToStringFilter.doFilter(tmpMorphemes);
		return doStringFilter(strings);
	}
	

	IMorpheme iMorphemeFilter_doFilter(IMorpheme morpheme)
	{
		return doMorphemeFilter(morpheme);
	}
	
	IMorpheme[] IMorphemeFilter_doFilter(Iterable<IMorpheme> morphemes)
	{
		return doMorphemeFilter(morphemes);
	}
	
	
	
	public String iStringFilter_doFilter(String str)
	{
		return doStringFilter(str);
	}
	
	public String[] iStringFilter_doFilter(Iterable<String> strings)
	{
		return doStringFilter(strings);
	}
	
		
	// #region IMorphemeToStringFilter メンバ
	
	String iMorphemeToStringFilter_doFilter(IMorpheme morpheme)
	{
		return doMorphemeToStringFilter(morpheme);
	}
	
	String[] iMorphemeToStringFilter_doFilter(Iterable<IMorpheme> morphemes)
	{
		return doMorphemeToStringFilter(morphemes);
	}


	///////
	static <T> ArrayList<T> createArrayList(Iterable<T> src){
		ArrayList<T> tmp = new ArrayList<T>();
		for (T t: src)
			tmp.add(t);
		return tmp;
	}
	
}
