//using System;
//using System.Collections.Generic;
//using System.Text;
//using System.IO;
//using System.Diagnostics;
package slothLib.NLP;

import slothLib.SlothLibException;
import slothLib.portUtil.PortUtil;
import slothLib.portUtil.Redirector;

import java.io.*;
import java.util.*;

    /// <summary>
    /// POS TaggerであるSS Taggerを利用するクラス
    /// </summary>
	public class SSTaggerServer implements IPosTagger
	{

		static 	{slothLib.SlothLib.init();} // to load propertiy file
		private static final String ssTaggerPathPropKey = "slothLib.NLP.SSTagger.ssTaggerPath";
		private static final String SSTAGER_PATH_DEFAULT = 	"/usr/local/bin/tagger"; 

		/// <summary>
		/// SS Taggerのプロセス
		/// </summary>
		private static Process p = null;

		private static PrintWriter pw;
		private static LineNumberReader lnr;
		
		/// <summary>
		/// tagger.exe のパス
		/// </summary>
		private static String sstaggerPath;
		
		static {
			// initialize sstaggerPath with property
			sstaggerPath = System.getProperty(ssTaggerPathPropKey, SSTAGER_PATH_DEFAULT);
		}
		
		
		/// <summary>
		/// ロックのためだけのオブジェクト
		/// </summary>
		private static Object lockObject = "";


		/// <summary>
		/// コンストラクタ
		/// </summary>
        public SSTaggerServer() { }

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="sstaggerPath">tagger.exeのパス</param>
		public SSTaggerServer(String sstaggerPath)
		{
			setSSTaggerPath(sstaggerPath);
		}


		/// <summary>
		/// SS Taggerを実行する
		/// </summary>
		/// <param name="text">解析対象のテキスト</param>
		/// <returns>解析結果</returns>
		public static PosTaggerResult doSSTagger(String text) throws SlothLibException
		{

			// 結果格納用。
			List<PosTaggedWord> result = new ArrayList<PosTaggedWord>();

			synchronized (lockObject)
			{
				// プロセスが走っていなければ、走らせる。
				if (p == null)
				{
                    startProcess();
                }

				// 入力を行ごとに分ける。
				String[] splitText = text.split("\\n");


				// 結果を格納するため
				//StringBuffer sb = new StringBuffer();


				// 
				try {
					for (String input: splitText)
					{
						System.err.println(input);
						pw.println(input.trim());
						pw.flush();
						String line = lnr.readLine();
						
						String[] words = line.split(" ");
						
						for (String w: words)
						{
							int locSlash = w.lastIndexOf('/');
							if (locSlash >= 1 && locSlash < w.length() - 1)
							{
								String raw = w.substring(0, locSlash);
								String pos = w.substring(locSlash + 1);
								PosTaggedWord ptw = new PosTaggedWord(pos, raw);
								result.add(ptw);
							}
						}
					}
						//sb.appendLine(line);
				} catch (IOException e){
					throw new RuntimeException(e);
				}
			}
			// System.Diagnostics.Debug.WriteLine(sb.toString());

            return new PosTaggerResult(result);
        }

        //#region Start, Stop

        /// <summary>
        /// プロセスを起動する
        /// </summary>
        public static void startProcess() throws SlothLibException
        {
            if (PortUtil.isNullOrEmpty(sstaggerPath))
            {
                throw new SlothLibException("taggerのパスが設定されていません");
            }
            synchronized (lockObject)
            {
                // プロセスが走っていなければ、走らせる。
                if (p == null)
                {
                	try {
                		String absPath = (new File(sstaggerPath)).getAbsolutePath();
                		String installedDir = absPath.substring(0, absPath.lastIndexOf(File.separatorChar));
                		
                		ProcessBuilder pb = new ProcessBuilder(new String[]{sstaggerPath}); 
                		pb.directory(new File(installedDir));
                		p = pb.start();
                		lnr = new LineNumberReader(new InputStreamReader(p.getInputStream()));
                		pw = new PrintWriter(new OutputStreamWriter(p.getOutputStream()), true);
                		Redirector.redirect(p.getErrorStream(), null);
                	} catch (IOException e) {
                		p = null;
                		throw new SlothLibException(e);
                	}
                
                }
            }
        }


        /// <summary>
		/// プロセスを停止させる。
		/// </summary>
		public static void stopProcess()
		{
			synchronized (lockObject)
			{
				if (p != null)
				{
					p.destroy();
                    p = null;
				}
			}
        }

        //#endregion


        //#region プロパティ

        /// <summary>
		/// tagger.exeのパス
		/// </summary>
		public static String getSSTaggerPath()
		{
			return sstaggerPath;
		}

		public static void setSSTaggerPath(String value) 
		{
			if (!PortUtil.fileExists(value))
			{
				throw new RuntimeException("taggerを見つけることができませんでした。" + value);
			}
			synchronized (lockObject)
			{
				sstaggerPath = value;
			}
		}

		//#endregion


		/// <summary>
		/// インスタンス用のコマンド
		/// </summary>
		/// <param name="text">解析対象のテキスト</param>
		/// <returns>解析結果</returns>
		public PosTaggerResult doTagging(String text) throws SlothLibException
		{
			return doSSTagger(text);
		}


		//#region IPosTagger メンバ

		//IPosTaggerResult IPosTagger.doTagging(String text)
		//{
//			return DoSSTagger(text);
		//}

		//#endregion

	}

