//using System;
//using System.Collections.Generic;
//using System.Text;
//using System.IO;
//using System.Diagnostics;

package slothLib.NLP;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.List;

import slothLib.portUtil.*;
import slothLib.SlothLibException;

    /**
     * 形態素解析器MeCabをサーバとして使うクラス。解析する度にプロセスを起動しない
     */
    public class MeCabServer implements IMorphologicalAnalyzer
    {

        /**
         * mecabのプロセス
         */
        private static Process p = null;

        private static PrintWriter pw;
        private static LineNumberReader lnr; 
        
        /**
         * mecab.exe のパス
         */
        private static String meCabPath = null;

        /**
         * mecabrc のパス
         */
        private static String meCabRcPath = null;

        /**
         * 何回解析したら再起動するか
         */
        private static int restartCount = 1000;

        /**
         * 現在までに解析した回数
         */
        private static int doneCount = 0;


        private static Object lockObject = new Object();


        /**
         * コンストラクタ
         */
		public MeCabServer() throws SlothLibException
		{
			MeCabServer.setMeCabPath(MeCab._getMeCabPath());
			MeCabServer.setMeCabRcPath(MeCab._getMeCabRcPath());
		}

		
		//private void setMeCabPath(String path)
		//{
       //			if (!PortUtil.fileExists(path))
		//	{
//				throw new FileNotFoundException("mecab.exeを見つけることができませんでした。", path);
//			}
//			meCabPath = path;
//		}
//
//		private void setMeCabRcPath(String path)
//		{
//			if (!PortUtil.fileExists(path))
//			{
//				throw new FileNotFoundException("mecabrcを見つけることができませんでした。", path);
//			}
//			meCabRcPath = path;
//		}

        /**
         * コンストラクタ
         * @param meCabPath mecab.exeのパス
         * @param meCabRcPath mecabrcのパス
         */
        public MeCabServer(String meCabPath, String meCabRcPath)
        {
            MeCabServer.meCabPath = meCabPath;
            MeCabServer.meCabRcPath = meCabRcPath;
        }


        /**
         * MeCabを実行する
         * @param text 解析対象のテキスト
         * @return 解析結果
         */
        public static MeCabResult doMeCab(String text) throws SlothLibException
        {
            // 結果を格納するため
            StringBuffer sb = new StringBuffer();

            synchronized (lockObject)
            {
                // プロセスが走っていなければ、走らせる。
                if (p == null || PortUtil.processExited(p))
                {
					p = null;
                    startProcess();
                }

                // 入力を行ごとに分ける。
                String[] splitText = text.split("\\n");

                for (String input: splitText)
                {
					String inputTrim = input.trim();
					if (PortUtil.isNullOrEmpty(inputTrim))
					{
						continue;
					}
					//Debug.WriteLine("input:" + inputTrim);
					pw.println(inputTrim);
                    String line;
                    try {
                    	while (!(line = lnr.readLine()).equals("EOS") && line != null)
                    	{
                    		//Debug.WriteLine("line:" + line);
                    		sb.append(line + "\n");
                    	}
                    } catch (IOException e)
                    {
                    	throw new SlothLibException(e);
                    }
                    sb.append("EOS" + "\n");
                }
            }

            //テキストを解析した回数が設定回数を超えたら一度停止させる
            if (doneCount++ >= restartCount)
            {
                stopProcess();
            }

            return new MeCabResult(sb.toString());
        }


        // Start, Stop

        /**
         * プロセスを起動する
         */
        public static void startProcess() throws SlothLibException
        {
            synchronized (lockObject)
            {
                // プロセスが走っていなければ、走らせる。
                if (p == null)
                {
                    try
                    {
        				List<String> cmds = new ArrayList<String>(); 
        				cmds.add(getMeCabPath());
        				if (MeCabServer.getMeCabRcPath() != null){
        					cmds.add("-r");
        					cmds.add(MeCabServer.getMeCabRcPath());
        				}
        				ProcessBuilder pb = new ProcessBuilder(cmds); 
        				p = pb.start();
        				lnr = new LineNumberReader(new InputStreamReader(p.getInputStream(), MeCab.getMeCabEncoding()));
        				pw = new PrintWriter(new OutputStreamWriter(p.getOutputStream(), MeCab.getMeCabEncoding()), true);
                    }
                    catch (IOException e)
                    {
                        p = null;
                        throw new SlothLibException(e);
                    }
                }
            }
        }


        /**
         * プロセスを停止する
         */
        public static void stopProcess()
        {
            synchronized (lockObject)
            {
                if (p != null)
                {
                    p.destroy();
                    p = null;
                    doneCount = 0;
                }
            }
        }




        /**
         * インスタンスのメソッド
         * @param text 解析対象のテキスト
         * @return 解析結果
         */
        public MeCabResult doAnalyze(String text) throws SlothLibException
        {
            //staticのメソッドを呼ぶ
            return doMeCab(text);
        }


        // プロパティ系

        /**
         * mecab.exeのパス
         */
        public static String getMeCabPath()
        {
            return meCabPath; 
        }
        
        public static void setMeCabPath(String value) throws SlothLibException{
        	if (!PortUtil.fileExists(value))
        	{
        		throw new SlothLibException("mecab.exeを見つけることができませんでした。" + value);
        	}
        	synchronized (lockObject)
        	{
        		meCabPath = value;
        	}
        }

        /**
         * mecabrcのパス
         */
        public static String getMeCabRcPath()
        {
            return meCabRcPath;
        }
        public static void setMeCabRcPath(String value) throws SlothLibException
        {
        	if (!PortUtil.fileExists(value))
        	{
        		throw new SlothLibException("mecabrcを見つけることができませんでした。" + value);
        	}
        	synchronized (lockObject)
        	{
        		meCabRcPath = value;
        	}
        }

        /**
         * 何回解析したらMeCabServerを再起動するか
         */
        public static int getRestartCount()
        {
            return restartCount;
        }
        public static void setRestartCount(int value){ 
        	restartCount = value; 
        }

        ///// <summary>
        ///// mecab.exeのパスを設定する
        ///// </summary>
        ///// <param name="path">設定するmecab.exeのパス</param>
        //public static void SetMeCabPath(String path)
        //{
        //    if (!PortUtil.fileExists(path))
        //    {
        //        throw new FileNotFoundException("mecab.exeを見つけることができませんでした。", path);
        //    }
        //    meCabPath = path;
        //}

        ///// <summary>
        ///// mecabrcのパスを設定する
        ///// </summary>
        ///// <param name="path">設定するmecabrcのパス</param>
        //public static void SetMeCabRcPath(String path)
        //{
        //    if (!PortUtil.fileExists(path))
        //    {
        //        throw new FileNotFoundException("mecabrcを見つけることができませんでした。", path);
        //    }
        //    meCabRcPath = path;
        //}




        // IMorphologicalAnalyzer メンバ

//        IMorphologicalAnalyzerResult IMorphologicalAnalyzer.DoAnalyze(String text)
//        {
            //return this.DoAnalyze(text);
//        }



    }

