//using System;
//using System.Collections.Generic;
//using System.Text;
//using System.Text.RegularExpressions;

package slothLib.NLP;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;
import java.util.regex.Matcher;


	/**
	 * MeCabの結果クラス
	 * <remarks>
	 * 

	 * <newpara>[2007-04-22][ohshima]作成</newpara>
	 * <newpara>[2007-05-12][ohshima]修正</newpara>
	 * </remarks>
	 */
	public class MeCabResult implements  IMorphologicalAnalyzerResult
	{


		// private static フィールドの正規表現

		/**
		 * 茶筌の-fオプションの出力にマッチする正規表現
		 */
		private static Pattern regexResultLine = Pattern.compile("^(.*)\\t(.*,.*,.*,.*),(.*),(.*),(.*),(.*),(.*)$");
		/**
		 * 未知語の時になぜか項目が一つ欠落するのに対応した正規表現
		 */
		//private static Pattern regexResultLine2 = Pattern.compile("^(.*)\\t(.*,.*,.*,.*),(.*),(.*),(.*),(.*)$");
		private static Pattern regexResultLine2 = Pattern.compile("^(.*)\\t(.*,.*,.*,.*),(.*),(.*),(.*)$");



        private List<MeCabMorpheme> morphemeList;

		/**
		 * MeCabの生の結果
		 */
		private String rawResult;

        /**
         * EOSごとのMeCabの生の結果
         */
        private List<String> sequenceRawResult;

        //GetOriginalArray()等を実装するためのフィルタ
        private static RemainOriginalFilter remainOriginalFilter = new RemainOriginalFilter();
        private static RemainPosFilter remainPosFilter = new RemainPosFilter();
        private static RemainRawFilter remainRawFilter = new RemainRawFilter();


		// 表層形\t品詞,品詞細分類1,品詞細分類2,品詞細分類3,活用形,活用型,原形,読み,発音

        /**
         * コンストラクタ
         * @param rawResult MeCabが返す生の結果
         */
		public MeCabResult(String rawResult)
		{
            this.morphemeList = new ArrayList<MeCabMorpheme>();

			this.rawResult = rawResult;
            this.sequenceRawResult = new ArrayList<String>();

			StringBuffer sb = new StringBuffer();

			// それを一行ごとに切り分ける。
			String[] lineList = rawResult.split("\\n");

			// 一行ごとに見ていく。
			for (String line: lineList)
			{
				Matcher successMatch = null;
				String pronunciation = "*";

				Matcher match = regexResultLine.matcher(line);
				if (!match.matches() && line.equals("EOS"))
				{
					sequenceRawResult.add(sb.toString());
					sb = new StringBuffer();
				}

				Matcher match2 = null;
				if (match.matches())
				{
					pronunciation = match.group(7);
					successMatch = match;
				}
				else
				{
					match2 = regexResultLine2.matcher(line);
					if (match2.matches())
					{
						successMatch = match2;
					}
					else
					{
						// ここに処理がたどり着くなら何らかの対処をする必要があるかもしれない。
						continue;
					}
				}

				String raw = successMatch.group(1);
				String pos = successMatch.group(2);
				String conjugationPattern = successMatch.group(3);
				String conjugationForm = successMatch.group(4);
                String original = successMatch.group(5);
                String reading;
                if (!match.matches())
                {
                    original = raw; //originalが*の場合はrawを入れる
                    reading = raw;
                } 
                else
                {
                	reading  = successMatch.group(6);
                }
                sb.append(line);
				sb.append("\n");

				MeCabMorpheme morpheme = new MeCabMorpheme(raw, pos, conjugationPattern, conjugationForm, original, reading, pronunciation);
				this.morphemeList.add(morpheme);
			}
		}


		/**
		 * MeCabが出力した生の結果。
		 */
		public String getRawResult()
		{
            return this.rawResult; 
		}



		/**
		 * EOSごとの結果の数を返す
		 * @return EOSごとに分けた結果の数
		 */
		public int getSequenceCount()
		{
			return this.sequenceRawResult.size();
		}

		/**
		 * EOSごとに結果を分けて返す
         * @return EOSごとに分けられたChaSenResultの配列
		 */
		public MeCabResult[] getSequenceResults()
		{
			MeCabResult[] result = new MeCabResult[this.sequenceRawResult.size()];
			for (int i = 0; i < result.length; i++)
			{
				result[i] = new MeCabResult(this.sequenceRawResult.get(i));
			}
			return result;
		}


		/**
		 * インデクサ
		 * @param index 形態素の番号
		 * @return index番目の形態素
		 */
		public MeCabMorpheme get(int index)
		{
            return (MeCabMorpheme)this.morphemeList.get(index); 
		}
		public void set(int index, MeCabMorpheme value)
		{
			this.morphemeList.set(index, value);
		}
		
		
        /**
         * 格納する形態素の配列
         */
//        public MeCabMorpheme[] Morphemes
        //{
//            get
//            {
//                return this.morphemeList.ToArray();
//            }
//        }


        // IMorphologicalAnalyzerResult メンバ


        public IMorpheme[] getMorphemes()
        {
                return this.morphemeList.toArray(new IMorpheme[0]);
        }

        /**
         * 各形態素のRawを配列にして返す
         * @return Rawの配列
         */
        public String[] getRawArray()
        {
            return remainRawFilter.doFilter(this.getMorphemes());
        }

        /**
         * 各形態素のOriginalを配列にして返す
         * @return Originalの配列
         */
        public String[] getOriginalArray()
        {
            return remainOriginalFilter.doFilter(this.getMorphemes());
        }

        /**
         * 各形態素のPOSを配列にして返す
         * @return POSの配列
         */
        public String[] getPOSArray()
        {
            return remainPosFilter.doFilter(this.getMorphemes());
        }




    }

