//using System;
//using System.Collections.Generic;
//using System.Text;
//using System.IO;
//using Microsoft.Win32;

package slothLib.NLP;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.List;

import slothLib.SlothLibException;
import slothLib.portUtil.PortUtil;

	/**
	 * 形態素解析器MeCabを利用するクラス 
	 * <remarks>
	 * バージョンは0.95を想定しています。
	 * mecabrcとして、MeCabインストール時の標準のものを使うことが想定されています。
	 * <newpara>[2007-04-22][ohshima]作成</newpara>
	 * </remarks>
	 */
	public class MeCab implements IMorphologicalAnalyzer
	{
		static 	{slothLib.SlothLib.init();} // to load propertiy file
		
		
		private static final String mecabPathPropKey = "slothLib.NLP.MeCab.meCabPath"; 
		private static final String mecabRcPathPropKey = "slothLib.NLP.MeCab.meCabRcPath"; 
		private static final String mecabEncodingPropKey = "slothLib.NLP.MeCab.meCabEncoding"; 
		
		private static final String MECAB_PATH_DEFAULT = "/usr/local/bin/mecab";
		private static final String MECAB_RCPATH_DEFAULT = "/usr/local/etc/mecabrc";
		private static final String MECAB_ENCODING_DEFAULT = "EUC_JP";

		// private フィールド

		/**
		 * mecabのパス
		 */
		private String meCabPath;

		/**
		 * mecabrc のパス
		 */
		private String meCabRcPath;





		/**
		 * 乱数を生成する
		 */
		//private static Random random = new Random();


		// コンストラクタ

		/**
		 * コンストラクタ。MeCabのデフォルトの設定を利用する
		 */
		public MeCab() throws SlothLibException
		{
			// mecab と mecabrc のパスをレジストリの情報を基に設定する。
			this.setMeCabPath(_getMeCabPath());
			this.setMeCabRcPath(_getMeCabRcPath());
		}

        /**
         * コンストラクタ。mecabとmecabrcのパスを指定する
         * @param meCabPath mecabeのパス
         * @param meCabRcPath mecabrcのパス
         */
		public MeCab(String meCabPath, String meCabRcPath) throws SlothLibException
		{
			setMeCabPath(meCabPath);
			setMeCabRcPath(meCabRcPath);
		}




		// プロパティ系

        /**
         * mecabのパス
         */
		public String getMeCabPath()
		{
			return this.meCabPath;
		}

        /**
         * mecabrcのパス
         */
		public String getMeCabRcPath()
		{
            return this.meCabRcPath; 
		}

		private void setMeCabPath(String path) throws SlothLibException
		{
			if (!PortUtil.fileExists(path))
			{
				throw new SlothLibException("mecabを見つけることができませんでした。" + path);
			}
			this.meCabPath = path;
		}

		private void setMeCabRcPath(String path) throws SlothLibException
		{
			if (path == null || !PortUtil.fileExists(path))
			{
				throw new SlothLibException("mecabrcを見つけることができませんでした。"  + path);
			}
			this.meCabRcPath = path;
		}




		// 表層形\t品詞,品詞細分類1,品詞細分類2,品詞細分類3,活用形,活用型,原形,読み,発音


		// メインのメソッド DoAnalyze

        /**
         * 形態素解析を行う
         * @param text 解析対象のテキスト
         * @return 解析結果
         */
		public MeCabResult doAnalyze(String text) throws SlothLibException
		{

			try {
				if (text == null)
				{
					throw new IllegalArgumentException("text" +  "引数がnullです。");
				}
				
				// ファイルパス
				String tempFilePath;
				tempFilePath = File.createTempFile("SlothLib.NLP.MeCab.doAnalyze.", ".tmp").getPath();
				
				// コンテンツの書き出し
				
				// 入力を行ごとに分ける。
				String[] splitText = text.split("\\n");
				
				
				{
					PrintWriter sw = new PrintWriter(new OutputStreamWriter(new FileOutputStream(tempFilePath), getMeCabEncoding()));
					for (String input: splitText)
					{
						String inputTrim = input.trim();
						if (PortUtil.isNullOrEmpty(inputTrim))
						{
							continue;
						}
						//Debug.WriteLine("input:" + inputTrim);
						sw.println(inputTrim);
					}
					sw.flush();
					sw.close();
				}
				
				/*
			using (StreamWriter sw = new StreamWriter(tempFilePath, false, System.Text.Encoding.GetEncoding("shift_jis")))
			{
				sw.Write(text);
			}*/
				
			   // MeCabの生結果
				String parsedText = parseFile(tempFilePath);
				
				// 一時ファイルが不要になるはずなので、削除する。
				// 例外？そんなもん無視ですよ。
				PortUtil.fileDelete(tempFilePath);
				
				return new MeCabResult(parsedText);
			} catch (IOException e){
				throw new SlothLibException(e); 
			}
		}			
			/**
			 * テキストファイルを読み込んで形態素解析する。
		 * @param tempFilePath テキストファイルのパス。
		 * @return MeCabの解析結果
			 */
		private String parseFile(String tempFilePath) throws SlothLibException
		{
			try {
				// プロセス作成とかいろいろ
				List<String> cmds = new ArrayList<String>(); 
				cmds.add(this.getMeCabPath());
				if (this.getMeCabRcPath() != null){
					cmds.add("-r");
					cmds.add(this.getMeCabRcPath());
				}
				cmds.add(tempFilePath);
				ProcessBuilder pb = new ProcessBuilder(cmds); 
				Process p = pb.start();
				String output = PortUtil.readFully(p.getInputStream(), getMeCabEncoding());
				while (true) {
					try {
						p.waitFor();
						break;
					} catch (InterruptedException e){}
				}
				int ec = p.exitValue();
				if (ec != 0)				{
					throw new SlothLibException("mecabの実行時にエラーが起こりました。" + ec + "\n" + output);
				}
				return output;
			} catch (IOException e){
				throw new SlothLibException(e);
			}

		}



		// public static メソッド

        /**
         * レジストリからmecabのパスを取得する
         * @return mecabのパス
         */
		public static String _getMeCabPath()
		{
			return  System.getProperty(mecabPathPropKey, MECAB_PATH_DEFAULT);
		}
        /**
         * レジストリからmecabrcのパスを所得する
		 * properties からmecabrcのパスを所得する
         * @return mecabrcのパス
         */
		public static String _getMeCabRcPath()
		{
			return System.getProperty(mecabRcPathPropKey, MECAB_RCPATH_DEFAULT);
		}

		
		public static String getMeCabEncoding(){
			return System.getProperty(mecabEncodingPropKey, MECAB_ENCODING_DEFAULT);
		}
		
		
		



		// IMorphologicalAnalyzer メンバ

//		IMorphologicalAnalyzerResult IMorphologicalAnalyzer.DoAnalyze(String text)
//		{
//			return this.DoAnalyze(text);
//		}



	}
