//using System;
//using System.Collections.Generic;
//using System.Text;
//using System.IO;
//using System.Text.RegularExpressions;
//using Microsoft.Win32;
//using Microsoft.VisualBasic;

package slothLib.NLP;

import slothLib.SlothLibException;
import slothLib.portUtil.PortUtil;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.*;

/**
	 * 形態素解析器「茶筌」を利用するクラス。
	 * chasen.exeを実行する。
	 * <remarks>
	 * このクラスを利用するためには別途、茶筌のWindows版2.3.3または2.4.0をインストールする必要があります。
	 * http://sourceforge.jp/projects/chasen-legacy/
	 * 茶筌の利用にはchasen.exeとchasenrcの位置を指定する必要があります。
	 * 茶筌をインストールしたユーザでは、レジストリに情報が書き込まれているためそれらを推定することができます。
	 * このクラスを使ったソフトウェアがAdministrators権限を持ったユーザであれば、そのユーザで茶筌をインストールすることにより、上記推定が可能になります。
	 * しかし、Users権限しか持たない場合、情報の推定を行うことができませんし、インストールされるchasenrcのみでは辞書を発見することができません。
	 * 安全にこのクラスを利用するには、明示的にchasen.exeとchasenrcを指定し、chasenrcには辞書のフルパスを設定をするのが良いということです。
	 * また、茶筌では未知語に対して原形が出力されません。
	 * MeCabではこれらの問題はありません。
	 * [TODO]jオプションをコンストラクタで与えるなんて考えられません。その他、色々プロパティーでやりましょう。
	 * <newpara>[2007-04-21][ohshima]作成</newpara>
	 * </remarks>
 */
	public class ChaSen implements IMorphologicalAnalyzer
	{
		static 	{slothLib.SlothLib.init();} // to load propertiy file

		private static final String chasenPathPropKey = "slothLib.NLP.ChaSen.chaSenPath"; 
		private static final String chasenRcPathPropKey = "slothLib.NLP.ChaSen.chaSenRcPath"; 
		private static final String chasenEncodingPropKey = "slothLib.NLP.ChaSen.chaSenEncoding"; 
		
		private static final String CHASEN_PATH_DEFAULT = "/usr/local/bin/chasen";
		private static final String CHASEN_RCPATH_DEFAULT = "/usr/local/etc/chasenrc";
		private static final String CHASEN_ENCODING_DEFAULT = "EUC_JP";

		
		// private フィールド

		/**
		 * chasen.exe のパス
		 */
		private String chaSenPath;

		/**
		 * chasenrc のパス
		 */
		private String chaSenRcPath;

		/**
		 * jオプション用
		 */
		private String option = "";




		// private static フィールド

		/**
		 * 半角カナ判別用の正規表現
		 */
		private static Pattern regexHalfKana = Pattern.compile("^([^｡-ﾟ]*)([｡-ﾟ]*)(.*)$");
		/**
		 * 乱数を生成する
		 */
//		private static Random random = new Random();




		// コンストラクタ

		/**
		 * 茶筌のデフォルトの設定を利用する。
		 * 茶筌をインストールしたユーザ以外ではレジストリに情報が無いため利用できない。
		 */
		public ChaSen()
		throws SlothLibException
		{ 
			this(false);
		}

		/**
		 * コンストラクタ。茶筌のデフォルトの設定を利用する。
		 * 茶筌をインストールしたユーザ以外ではレジストリに情報が無いため利用できない。
		 * @param useOption_j 茶筌実行時に-jオプションを付ける場合はtrue。
		 */
		public ChaSen(boolean useOption_j)
		throws SlothLibException
		{
			// chasen.exe と chasenrc のパスをproperty情報を基に設定する。
			this.setChaSenPath(_getChaSenPath());
			this.setChaSenRcPath(_getChaSenRcPath());
			// -jオプションの設定
			if (useOption_j)
			{
				this.option = "-j ";
			}
		}

        /**
         * コンストラクタ。パスの情報を指定する
         * @param chaSenPath 
         * @param chaSenRcPath 
         */
		public ChaSen(String chaSenPath, String chaSenRcPath)
		throws SlothLibException
		{ 
			this(chaSenPath, chaSenRcPath, false);
		}

        /**
         * コンストラクタ。パス情報と-jオプションを指定する
         * @param chaSenPath 
         * @param chaSenRcPath 
         * @param useOption_j 
         */
		public ChaSen(String chaSenPath, String chaSenRcPath, boolean useOption_j) 
		throws SlothLibException
		{
			setChaSenPath(chaSenPath);
			setChaSenRcPath(chaSenRcPath);
			// -jオプションの設定
			if (useOption_j)
			{
				this.option = "-j ";
			}
		}





		// プロパティ系

        /**
         * chasen.exeのパス
         */
		public String getChaSenPath()
        {
            return this.chaSenPath; 
		}

        /**
         * chasenrcのパス
         */
		public String getChaSenRcPath()
		{
            return this.chaSenRcPath; 
		}

        /**
         * chasen.exeのパスを設定する
         * @param path 設定するchasen.exe
         */
		private void setChaSenPath(String path) throws SlothLibException
		{
			if (!PortUtil.fileExists(path))
			{
				throw new SlothLibException("chasen.exeを見つけることができませんでした。" + path);
			}
			this.chaSenPath = path;
		}

        /**
         * chasenrcのパスを設定する
         * @param path 設定するchasenrcのパス
         */
		private void setChaSenRcPath(String path) throws SlothLibException
		{
			if (!PortUtil.fileExists(path))
			{
				throw new SlothLibException("chasenrcを見つけることができませんでした。" + path);
			}
			this.chaSenRcPath = path;
		}




		// メインのメソッド DoAnalyze

        /**
         * ChaSenで形態素解析を行う
         * @param text 解析対象のテキスト
         * @return 解析結果
         */
		public ChaSenResult doAnalyze(String text) throws SlothLibException
		{

			try {
				if (text == null)
				{
					throw new IllegalArgumentException("text" + "引数がnullです。");
				}

				// ファイルパス
				String tempFilePath;
				tempFilePath = File.createTempFile("SlothLib.NLP.ChaSen.doAnalyze.", ".tmp").getPath();

				
				// 入力を行ごとに分ける。
				String[] splitText = text.split("\\n");
				
				
				{
					PrintWriter sw = new PrintWriter(new OutputStreamWriter(new FileOutputStream(tempFilePath), getChaSenEncoding()));
					for (String input: splitText)
					{
						String inputTrim = input.trim();
						if (PortUtil.isNullOrEmpty(inputTrim))
						{
							continue;
						}
						//Debug.WriteLine("input:" + inputTrim);
						sw.println(HankakuZenkaku.hankanaToZen(inputTrim));
						System.err.println(HankakuZenkaku.hankanaToZen(inputTrim));
					}
					sw.flush();
					sw.close();
				}
								
				// ChaSenの生結果
				String parsedText = parseFile(tempFilePath);
				
				
				// 一時ファイルが不要になるはずなので、削除する。
				// 例外？そんなもん無視ですよ。
				PortUtil.fileDelete(tempFilePath);
				return new ChaSenResult(parsedText);
			}
			catch (IOException e)
			{
				throw new SlothLibException(e);
			}
			

		}

		/**
		 * テキストファイルを読み込んで形態素解析する。
		 * @param tempFilePath テキストファイルのパス。
		 * @return ChaSenの解析結果
		 */
		private String parseFile(String tempFilePath) throws SlothLibException
		{
			// 標準入力に書き込んでいくよりも、chasenにファイルを食わせた方が高速。
			// 特にテキストが大きくなるに従って段違いになる。
			// プロセス作成とかいろいろ

			try {
				List<String> cmds = new ArrayList<String>(); 
				cmds.add(this.getChaSenPath());
//				cmds.add("-r");
//				cmds.add(this.getChaSenRcPath());
				if (!PortUtil.isNullOrEmpty(this.option))
					cmds.add(this.option);
				cmds.add(tempFilePath);
				
				ProcessBuilder pb = new ProcessBuilder(cmds); 
				Process p = pb.start();
				String output = PortUtil.readFully(p.getInputStream(), getChaSenEncoding());
				while (true) {
					try {
						p.waitFor();
						break;
					} catch (InterruptedException e){}
				}
				int ec = p.exitValue();			
				
				if (ec != 0)
				{
					throw new SlothLibException("chasen.exeの実行時にエラーが起こりました。");
				}
				return output;

			} catch (IOException e){
				throw new SlothLibException(e);
			}
		}




		// public static メソッド

        /**
         * レジストリからchasen.exeのパスを取得する
         * @return chasen.exeのパス
         */
		public static String _getChaSenPath() 
		{
			return System.getProperty(chasenPathPropKey, CHASEN_PATH_DEFAULT);
		}

        /**
         * レジストリからchasenrcのパスを取得する
         * @return chasenrcのパス
         */
		public static String _getChaSenRcPath()
		{
			return System.getProperty(chasenRcPathPropKey, CHASEN_RCPATH_DEFAULT);
		}

		public static String getChaSenEncoding(){
			return System.getProperty(chasenEncodingPropKey, CHASEN_ENCODING_DEFAULT);
		}





		// IMorphologicalAnalyzer メンバ

//		IMorphologicalAnalyzerResult IMorphologicalAnalyzer.doAnalyze(String text)
//		{
//			return this.doAnalyze(text);
//		}


	}
