//using System;
//using System.Collections.Generic;
//using System.Text;
//using System.IO;
package slothLib.io;

import java.util.Map;
import java.util.HashMap;

import slothLib.SlothLibException;
import slothLib.portUtil.*;

	/**
	 * IContentReaderを実装するクラスの共通部分のための抽象クラス
	 * [2006-11-12][ohshima] 作成
	 */
	public abstract class AbstractContentReader implements IContentReader
	{
		/**
		 * 対応する拡張子を格納しておくMap
		 */
		private Map<String, Boolean> readableExtensionMap;


		public AbstractContentReader(){
			this.readableExtensionMap = new HashMap<String, Boolean>();
		}
		
		/**
		 * コンストラクタ
		 * @param readableExtensions 
		 */
		public AbstractContentReader(String ... readableExtensions)
		{
			this();
			for (String ext: readableExtensions)
			{
				this.readableExtensionMap.put(ext.toLowerCase(), false);
			}
		}

		/**
		 * 引数で与えられた拡張子に対応しているかどうか
		 * @param extension ピリオド付きの拡張子
		 * @return 読み込み可能かどうか
		 */
		public boolean isReadable(String extension)
		{
			return this.readableExtensionMap.containsKey(extension.toLowerCase());
		}

		/**
		 * ファイルからテキストを抽出する。
		 * 対応しているかどうかはReadableを用いる。
		 * @param filePath ファイルのパス
		 * @return 抽出された内容
		 */
		public String read(String filePath) throws SlothLibException
		{
			
			if (!PortUtil.fileExists(filePath))
			{
				throw new SlothLibException("指定されたファイルが見つかりません。" + filePath);
			}

						// 拡張子を"."付きで小文字で取得
			String fileExt = filePath.substring(filePath.lastIndexOf(".")).toLowerCase();

			if (!isReadable(fileExt))
			{
				// 対応する拡張子でなければ例外を返す。
				throw new IllegalArgumentException("このクラスでは読み込みできないファイルです。");
			}

			// あとは個別のクラスの実装に任せる。
			return process(filePath);

		}

		/**
		 * ファイルからテキストを抽出する。
		 * ファイルの存在と、対応はすでにチェック済みである。
		 * サブクラスが実装する部分。
		 * @param filePath ファイルのパス
		 * @return 抽出された内容
		 */
		protected abstract String process(String filePath) throws SlothLibException;

	}

