﻿using System;
using System.Collections.Generic;
using System.Text;

namespace SlothLib.LinearAlgebra.FeatureVector
{
	/// <summary>
	/// 特徴ベクトルクラス
	/// T型のキーに対し数値を保持する。
	/// </summary>

	public class BasicVector<T> : AbstractVector<T>
	{
		private Dictionary<T,double> dic;

		#region コンストラクタ

		/// <summary>
		/// コンストラクタ
		/// </summary>
		public BasicVector()
		{
			dic = new Dictionary<T,double>();
		}

		/// <summary>
		/// コンストラクタ。vectorのコピーを行う
		/// </summary>
		public BasicVector(IVector<T> vector)
		{
			dic = new Dictionary<T, double>();

			foreach (T key in vector.Keys)
			{
				this[key] = vector[key];
			}
		}

		/// <summary>
		/// コンストラクタ。dictionaryを元にベクトルを作成する
		/// </summary>
		public BasicVector(Dictionary<T, double> dictionary)
		{
			dic = new Dictionary<T, double>();

			foreach (T key in dictionary.Keys)
			{
				this[key] = dictionary[key];
			}
		}

		#endregion

		/// <summary>
		/// ベクトルにアクセスする
		/// </summary>
		public override double this[T key] {
			get {
				if (dic.ContainsKey(key))
				{
					return dic[key];
				}
				else
				{
					return 0;
				}
			}
			set {
				dic[key] = value;

				if (value == 0)
				{
					dic.Remove(key);
				}
			}
		}

		/// <summary>
		/// ベクトルの複製を取得する
		/// </summary>
		public override object Clone()
		{
			return new BasicVector<T>(this);
		}

		/// <summary>
		/// ベクトルの次元（要素数）を取得する
		/// </summary>
		public int Dimension
		{
			get {
				return dic.Count;
			}
		}

		/// <summary>
		/// ベクトルをdouble[]型に変換する
		/// </summary>
		public override double[] GetPlainVector()
		{
			double[] r = new double[this.Dimension];
			int i = 0;

			foreach (T key in this.Keys)
			{
				r[i++] = this[key];
			}

			return r;
		}

		/// <summary>
		/// ベクトルがkeyの次元を持つかどうかを返す
		/// </summary>
		public override bool ContainsKey(T key)
		{
			return dic.ContainsKey(key);
		}


		/// <summary>
		/// ベクトルにある要素のうち、0以外の値をもつ要素の一覧
		/// </summary>
		public override Dictionary<T, double>.KeyCollection Keys
		{
			get { return dic.Keys; }
		}

        /// <summary>
        /// ベクトル中の値が 0 でない要素の数を取得する。
        /// </summary>
        public override int Count
        {
            get { return dic.Count; }
        }

		/// <summary>
		/// ベクトルにある要素のうち、0以外の値をもつ要素の一覧を取得する。
		/// </summary>
		public override T[] KeyList
		{
			get
			{
				T[] keyList = new T[this.dic.Count];
				this.dic.Keys.CopyTo(keyList, 0);
				return keyList;
			}
		}

		/// <summary>
		/// ソートされた要素のリスト
		/// </summary>
		/// <returns>ベクトルの重みが指定された順にソートされたKeyList</returns>
		public override T[] GetSortedKeyList()
		{
			T[] keyList = this.KeyList;
			Array.Sort(keyList, new VectorComparer<T>(this));
			return keyList;
		}
	}
}
