/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.sqlService.dbobject;

/**
 * ＳＱＬ指定形式の完全テーブル名の生成を管理します
 * 
 * @author sima
 * 
 */

public class SchemaTableName implements Comparable<SchemaTableName> {
	private String schemaName = "";

	private String tableName = "";

	private String aliasName = "";

	private boolean storesLowerCaseIdentifiers = false;

	private boolean storesUpperCaseIdentifiers = false;

	/**
	 * スキーマ名とテーブル名を空文字列として生成します
	 */
	public SchemaTableName() {
	}

	/**
	 * スキーマ名．テーブル名のＳＱＬ指定形式の文字列から、 ピリオドを区切り文字としてスキーマ名とテーブル名に分割して生成します。
	 * 
	 * @param completeTableName
	 *            ＳＱＬ形式の完全テーブル名
	 */
	public SchemaTableName(boolean storesLowerCaseIdentifiers, boolean storesUpperCaseIdentifiers,
	        String completeTableName) {
		this.storesLowerCaseIdentifiers = storesLowerCaseIdentifiers;
		this.storesUpperCaseIdentifiers = storesUpperCaseIdentifiers;

		if (completeTableName == null) {
			return;
		}

		int idx = completeTableName.indexOf('.');
		if (idx == -1) {
			setTableName(completeTableName);
			return;
		}

		setSchemaName(completeTableName.substring(0, idx));
		setTableName(completeTableName.substring(idx + 1));
	}

	/**
	 * 既にスキーマ名とテーブル名が分割してある場合の生成手段です
	 * 
	 * @param schemaName
	 *            スキーマ名
	 * @param tableName
	 *            テーブル名
	 */
	public SchemaTableName(boolean storesLowerCaseIdentifiers, boolean storesUpperCaseIdentifiers, String schemaName,
	        String tableName) {
		this.storesLowerCaseIdentifiers = storesLowerCaseIdentifiers;
		this.storesUpperCaseIdentifiers = storesUpperCaseIdentifiers;

		setSchemaName(schemaName);
		setTableName(tableName);
	}

	private void setSchemaName(String schemaName) {
		if (schemaName == null) {
			this.schemaName = "";
		} else {
			this.schemaName = convertStoresIdentifiers(schemaName);
		}
	}

	private void setTableName(String tableName) {
		if (tableName == null) {
			this.tableName = "";
		} else {
			this.tableName = convertStoresIdentifiers(tableName);
		}
	}

	private String convertStoresIdentifiers(String str) {
		if (storesLowerCaseIdentifiers) {
			return str.toLowerCase().trim();
		}
		if (storesUpperCaseIdentifiers) {
			return str.toUpperCase().trim();
		}
		return str;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}

		if (obj == null) {
			return false;
		}

		if (this.getClass() != obj.getClass()) {
			return false;
		}

		SchemaTableName other = (SchemaTableName) obj;

		// 何故か SQLiteでは、getTables と getColumns
		// の大文字小文字が異なる。たぶんgetTablesの実装が間違ってる。
		// しょうがないので、比較は大文字に統一して行う。保持するデータはドライバの値のままとする。
		// なんかヤバげな対応！
		return other.getSchemaName().toUpperCase().equals(this.schemaName.toUpperCase())
		       && other.getTableName().toUpperCase().equals(this.tableName.toUpperCase());
	}

	@Override
	public int hashCode() {
		return (this.schemaName + this.tableName).hashCode();
	}

	/**
	 * ＳＱＬ指定形式の完全テーブル名を、スキーマ名．テーブル名として返却します。 スキーマ名が空で生成された場合は、ピリオドは付加されません
	 * 
	 * @return ＳＱＬ形式の完全テーブル名
	 */
	public String getCompleteTableName() {
		if (schemaName.equals("")) {
			return tableName;
		}
		return schemaName + "." + tableName;

	}

	/**
	 * スキーマ名を返却します
	 * 
	 * @return スキーマ名
	 */
	public String getSchemaName() {
		return schemaName;
	}

	/**
	 * テーブル名を返却します
	 * 
	 * @return テーブル名
	 */
	public String getTableName() {
		return tableName;
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return getCompleteTableName();
	}

	public int compareTo(SchemaTableName o) {
		return getCompleteTableName().compareTo(o.getCompleteTableName());
	}

	public String getAliasName() {
		return aliasName;
	}

	public void setAliasName(String aliasName) {
		this.aliasName = aliasName;
	}
}
