/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.sqlService.dbobject;

import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import javax.swing.Icon;

import jdbcacsess2.sqlService.dbobject.DBObjectCatalogs.DBObjectCatalog;
import jdbcacsess2.sqlService.dbobject.DBObjectSchemas.DBObjectSchema;
import jdbcacsess2.sqlService.dbobject.icon.DbObjectIconImage;

/**
 * PROCEDUREの一覧
 * 
 * @author sima
 * 
 */
public class DBObjectProcedures extends DBObjects {

	private final ArrayList<DBObjectProcedure> procedures = new ArrayList<DBObjectProcedure>();

	/**
	 * コンストラクタ
	 * 
	 * @param catalog
	 * @param schema
	 * @param dmd
	 * @throws SQLException
	 */
	public DBObjectProcedures(DBObjectCatalog catalog, DBObjectSchema schema, DatabaseMetaData dmd) throws SQLException {

		String catalogName = null;
		if (catalog != null) {
			catalogName = catalog.catalogName;
		}

		String schemaName = null;
		if (schema != null) {
			schemaName = schema.schemaName;
		}

		ResultSet rs = dmd.getProcedures(catalogName, schemaName, "%");

		DBObjectProcedure prev = new DBObjectProcedure();
		while (rs.next()) {
			// 同一のプロシジャ名は捨てる（オーバロード分）
			DBObjectProcedure procedure = new DBObjectProcedure(rs);
			if (!prev.getKey().equals(procedure.getKey())) {
				procedures.add(procedure);
				prev = procedure;
			} else {
				prev.incrementDuplicateCnt();
			}
		}
		rs.close();
	}

	@Override
	public ArrayList<? extends DBObject> getDBObjects() {
		return procedures;
	}

	@Override
	public String toString() {
		return "PROCEDURE";
	}

	/**
	 * PROCEDUREのプロパティ
	 * 
	 * @author sima
	 * 
	 */
	public static class DBObjectProcedure extends DBObject {

		private String procedureCat = "";

		/**
		 * {@link DatabaseMetaData#getProcedures(String, String, String)} の第１要素
		 * 
		 * @return
		 */
		public String getProcedureCatalog() {
			return procedureCat;
		}

		private String procedureSchem = "";

		/**
		 * {@link DatabaseMetaData#getProcedures(String, String, String)} の第２要素
		 * 
		 * @return
		 */
		public String getProcedureSchema() {
			return procedureSchem;
		}

		private String procedureName = "";

		/**
		 * {@link DatabaseMetaData#getProcedures(String, String, String)} の第３要素
		 * 
		 * @return
		 */
		public String getProcedureName() {
			return procedureName;
		}

		private String remarks = "";

		/**
		 * {@link DatabaseMetaData#getProcedures(String, String, String)} の第７要素
		 * 
		 * @return
		 */
		public String getRemarks() {
			return remarks;
		}

		private short procedureType = 0;

		/**
		 * {@link DatabaseMetaData#getProcedures(String, String, String)} の第８要素
		 * 
		 * @return
		 */
		public short getProcedureType() {
			return procedureType;
		}

		public String getProcedureTypeName() {
			switch (procedureType) {
			case DatabaseMetaData.procedureResultUnknown:
				return "ResultUnknown";
			case DatabaseMetaData.procedureNoResult:
				return "NoResult";
			case DatabaseMetaData.procedureReturnsResult:
				return "ReturnsResult";
			}

			return "?";
		}

		private String specificName = "";

		/**
		 * {@link DatabaseMetaData#getProcedures(String, String, String)} の第９要素
		 * 
		 * @return
		 */
		public String getSpecificName() {
			return specificName;
		}

		private int duplicateCnt = 1;

		/**
		 * オーバロード関数の個数
		 * 
		 * @return
		 */
		public int getDuplicateCnt() {
			return duplicateCnt;
		}

		void incrementDuplicateCnt() {
			this.duplicateCnt++;
		}

		String getKey() {
			return procedureCat + procedureSchem + procedureName;
		}

		private final List<Property> properties = new ArrayList<Property>();

		@Override
		public List<Property> getProperties() {
			return properties;
		}
		/**
		 * コンストラクタ
		 * 
		 * @param rs
		 * @throws SQLException
		 */
		public DBObjectProcedure(ResultSet rs) throws SQLException {
			int colCnt = rs.getMetaData().getColumnCount();
			int i = 0;
			if (colCnt > i) {
				procedureCat = rs.getString(++i);
				properties.add(new Property("PROCEDURE_CAT", procedureCat));
			}
			if (colCnt > i) {
				procedureSchem = rs.getString(++i);
				properties.add(new Property("PROCEDURE_SCHEM", procedureSchem));
			}
			if (colCnt > i) {
				procedureName = rs.getString(++i);
				properties.add(new Property("PROCEDURE_NAME", procedureName));
			}
			if (colCnt > i) {
				/* NO4 RESERVE */
				++i;
			}
			if (colCnt > i) {
				/* NO5 RESERVE */;
				++i;
			}
			if (colCnt > i) {
				/* NO6 RESERVE */;
				++i;
			}
			if (colCnt > i) {
				remarks = rs.getString(++i);
				properties.add(new Property("REMARKS", remarks));
			}
			if (colCnt > i) {
				procedureType = rs.getShort(++i);
				properties.add(new Property("PROCEDURE_TYPE", getProcedureTypeName()));
			}
			if (colCnt > i) {
				specificName = rs.getString(++i);
				properties.add(new Property("SPECIFIC_NAME", specificName));
			}
		}

		public DBObjectProcedure() {
		}


		@Override
		public DBObjects getChildren(DatabaseMetaData dmd) throws SQLException {
			return new DBObjectProcedureColumns(this, dmd);
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#getName()
		 */
		@Override
		public String getName() {
			return procedureName;
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#getIconImage()
		 */
		@Override
		public Icon getIconImage() {
			return DbObjectIconImage.PROCEDURE.getValue();
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#isStrong()
		 */
		@Override
		public boolean isStrong() {
			return false;
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#isBottom()
		 */
		@Override
		public boolean isBottom() {
			return false;
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#getSummary()
		 */
		@Override
		public String getSummary() {
			return procedureName + (duplicateCnt > 1 ? "  [*" + duplicateCnt + "]" : "") + " " + specificName;
		}

		@Override
		public boolean isPropertyEnable() {
			return true;
		}

	}

}
