/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.sqlService.dbobject;

import java.sql.DatabaseMetaData;
import java.sql.SQLException;
import java.util.List;

import javax.swing.Icon;

import jdbcacsess2.sqlService.ConstSqlTypes;
import jdbcacsess2.sqlService.ConstSqlTypes.Categoly;
import jdbcacsess2.sqlService.DataBaseConnection;

abstract public class DBObject {

	enum DBobjectCacheState {
		NOCACHE, CACHING, CACHED
	};

	volatile private DBobjectCacheState state = DBobjectCacheState.NOCACHE;

	DBObjects list;

	public boolean isNoCache() {
		return (state == DBobjectCacheState.NOCACHE);
	}

	public boolean isCaching() {
		return (state == DBobjectCacheState.CACHING);
	}

	public void clearCache() {
		state = DBobjectCacheState.NOCACHE;
	}

	public void getAsyncChildren(final DataBaseConnection dataBaseConnection, final DBobjectChangeListener listener) {
		if (isBottom()) {
			throw new IllegalStateException("node is bottom(leaf)");
		}
		if (state == DBobjectCacheState.CACHED) {
			// 前回のキャッシュを通知する
			listener.valueChanged(list);
			return;
		}
		if (state == DBobjectCacheState.CACHING) {
			return;
		}

		state = DBobjectCacheState.CACHING;
		Runnable worker = new Runnable() {
			@Override
			public void run() {

				try {
					list = getChildren(dataBaseConnection.getMetaData());
				} catch (Throwable t) {
					listener.exception(t);
					return;
				}
				state = DBobjectCacheState.CACHED;
				listener.valueChanged(list);
			}
		};
		dataBaseConnection.getExecutorService().execute(worker);
	}

	static public class Property {
		private final String name;
		private final String value;

		public Property(String name, String value) {
			this.name = name;
			if (value == null) {
				this.value = "";
			} else {
				this.value = value;
			}
		}
		public String getName() {
			return name;
		}

		public String getValue() {
			return value;
		}
	}

	abstract public List<Property> getProperties();

	/**
	 * 
	 * @param dmd
	 * @return
	 * @throws SQLException
	 */
	abstract public DBObjects getChildren(DatabaseMetaData dmd) throws SQLException;

	/**
	 * TABLE系の場合のみスキーマ名とテーブル名の組み合わせをSchemaTableNameで返却する。
	 * TABLE系以外の場合、空のSchemaTableNameを返却する
	 * 
	 * @return
	 * @throws SQLException
	 */
	public SchemaTableName getSchemaTableName() throws SQLException {
		return new SchemaTableName();
	}

	/**
	 * データベースオブジェクトの名称を返却する。データベースオブジェクト以外の場合は、nullを返却する。
	 * 
	 * @return
	 */
	abstract public String getName();

	/**
	 * 
	 * @return
	 */
	abstract public Icon getIconImage();

	/**
	 * 強調表示する内容であるかを返却する。
	 * 
	 * @return
	 */
	abstract public boolean isStrong();

	/**
	 * 設計上で最終階層なのかを返却する。 JDBCの実装とは関係なく、下位階層を保持する可能性が有るかである
	 * 
	 * @return
	 */
	abstract public boolean isBottom();

	/**
	 * オブジェクトツリーに表示する文字列
	 * 
	 * @return
	 */
	abstract public String getSummary();

	public String getProperty1() {
		return "";
	}

	public String getProperty2() {
		return "";
	}

	public String getProperty3() {
		return "";
	}

	public String getProperty4() {
		return "";
	}

	/**
	 * プロパティ情報を画面表示する為に有用な情報を持ってるか。
	 * 
	 * @return true プロパティ情報が有効
	 */
	abstract public boolean isPropertyEnable();

	/**
	 * データ型の文字列表現を編集
	 * 
	 * @param dataType
	 * @param columnSize
	 * @param decimalDigits
	 * @return
	 */
	public StringBuilder editAttrString(int dataType, int columnSize, int decimalDigits) {
		StringBuilder attr = new StringBuilder();

		ConstSqlTypes constSqlTypes = ConstSqlTypes.searchTypes(dataType);
		attr.append(constSqlTypes.name()).append('(').append(columnSize);

		if (constSqlTypes.getCategoly() == Categoly.NUM) {
			if (decimalDigits != 0) {
				attr.append('.').append(decimalDigits);
			}
		}

		attr.append(')');

		return attr;

	}
}
