/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.sqlService;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;

import javax.swing.event.EventListenerList;

import jdbcacsess2.main.Jdbcacsess2;
import jdbcacsess2.main.OptionValues;
import jdbcacsess2.main.ShowDialog;
import jdbcacsess2.sqlService.SqlExecuteParmeter.Parameter;
import jdbcacsess2.sqlService.exception.DbConnectIllgalStateException;
import jdbcacsess2.sqlService.parse.SqlExecuteSentencies;
import jdbcacsess2.sqlService.parse.SqlExecuteSentencies.SqlExecuteSentence;
import jdbcacsess2.sqlService.parse.SqlInputParameter;

/**
 * SQL文を実行し、リスナーを通して処理結果を通知します。
 * 
 * @author sima
 * 
 */
public class SqlAsyncExecute {

	// private volatile int resultRowLimitCnt = 5000;
	/**
	 * SQL文の解析結果
	 */
	private final ArrayList<SqlExecuteSentence> sqlExecuteSentenceList;
	/**
	 * 処理結果通知リスナーのリスト
	 */
	private final EventListenerList sqlExecutedListeners;
	/**
	 * 実行中のスレッド
	 */
	private Thread thread;

	private final OptionValues optionValues;
	/**
	 * コンストラクタ
	 * 
	 * @param dataBaseConnection
	 * @param sqlSentence
	 *            実行したいSQL文
	 */
	public SqlAsyncExecute(String sqlSentence, String sentenceSeparator, OptionValues optionValues) {

		sqlExecutedListeners = new EventListenerList();

		sqlExecuteSentenceList = new SqlExecuteSentencies(sqlSentence, sentenceSeparator).getSqlExecuteSentenceList();

		this.optionValues = optionValues;
	}

	/**
	 * SELECT文が複数存在しているかを判定する
	 * 
	 * @return true 複数のSELECT文がある
	 */
	public boolean isDuplxSelect() {
		int cnt = 0;
		for (SqlExecuteSentence sqlExecuteSentence : sqlExecuteSentenceList) {
			if (sqlExecuteSentence.getSqlCommand().equals("SELECT")) {
				cnt++;
				if (cnt == 2) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * SQL文の解析結果をgetする。
	 * 
	 * @return SQL文の解析結果
	 */
	public ArrayList<SqlExecuteSentence> getSqlExecuteSentencies() {
		return sqlExecuteSentenceList;
	}

	/**
	 * 結果受け取りリスナーの登録
	 * 
	 * @param listener
	 */
	public void addSqlExcutedListener(SqlExecutedListener listener) {
		sqlExecutedListeners.add(SqlExecutedListener.class, listener);
	}

	/**
	 * 結果受け取りリスナーの削除
	 */
	public void removeSqlExcutedListener(SqlExecutedListener listener) {
		sqlExecutedListeners.remove(SqlExecutedListener.class, listener);
	}

	/**
	 * 非同期SQL実行。 実行中SQL文がある場合は、IllegalStateExceptionが発生します。
	 * 同期SQL実行したい場合は、返却されたSqlTskのtaskJoin()を呼び出して、待ち合わせをしてください。<br>
	 * 入力パラメータがある場合は事前に {@link #getSqlExecuteSentencies()}
	 * を使用して、それぞれの入力パラメータをセットしておいてください。
	 * 
	 * @param parameters
	 *            実行パラメータリスト
	 * @param sqlTypes
	 *            実行パラメータのインデックスをキーとしたSQL型マップ
	 * @return 実行SQLタスク
	 * @throws DbConnectIllgalStateException
	 */
	public SqlExecuteTask executeAsync(DataBaseConnection dataBaseConnection) throws DbConnectIllgalStateException {
		dataBaseConnection.lockConnection(sqlExecuteSentenceList.size() == 0 ? ""
				: sqlExecuteSentenceList.get(0).getSqlCommand());

		SqlTask sqlTask = new SqlTask(dataBaseConnection);
		thread = new Thread(sqlTask);
		thread.start();
		for (SqlExecutedListener l : sqlExecutedListeners.getListeners(SqlExecutedListener.class)) {
			l.taskAccept(sqlTask);
		}
		Jdbcacsess2.logger.fine("(execute) return.");
		return sqlTask;
	}

	/**
	 * SQL実行タスク
	 */
	class SqlTask implements Runnable, SqlExecuteTask {
		/**
		 * データベース接続
		 */
		private final DataBaseConnection dataBaseConnection;
		/**
		 * 実行中のPrepareStatement
		 */
		private PreparedStatement preparedStatement;
		/**
		 * 処理結果行数
		 */
		private int rowCnt;
		/**
		 * wait状態で想定外の再開が発生した場合に再度waitする為のwait状態フラグ
		 */
		private boolean available = false;
		/**
		 * 発生した例外
		 */
		private Throwable sqlTaskThrow;
		/**
		 * スレッド開始時刻
		 */
		volatile private long timeMillisThreadStart;
		/**
		 * スレッド終了時刻
		 */
		volatile private long timeMillisThreadEnd;

		private final DataBaseConnectionListener changeConnection = new DataBaseConnectionListener() {

			@Override
			public void dataBaseConnectionOpened(DataBaseConnection dataBaseConnection) {
			}

			@Override
			public void dataBaseConnectionClosing(DataBaseConnection dataBaseConnection) {
				if (preparedStatement != null) {
					try {
						preparedStatement.cancel();
					} catch (Exception e) {
						ShowDialog.errorMessage(e);
					}
				}
				taskCancel();
				taskJoin(dataBaseConnection.getTimeoutSeconds());
			}

			@Override
			public void dataBaseConnectionClosed(DataBaseConnection dataBaseConnection) {
			}

		};

		/**
		 * コンストラクタ。
		 */
		public SqlTask(DataBaseConnection dataBaseConnection) {
			this.dataBaseConnection = dataBaseConnection;
		}

		/**
		 * スレッド実行メソッド
		 */
		public void run() {
			Jdbcacsess2.logger.info("(task)beging.");

			// コネクションクローズ時に、スレッドの後始末を自動的に行う為、リスナー登録
			dataBaseConnection.addConnectionListener(changeConnection);

			for (SqlExecuteSentence sentence : sqlExecuteSentenceList) {
				if (Thread.interrupted()) {
					Jdbcacsess2.logger.info("  (task)canceled.");
					break;
				}
				sqlExec(sentence);
			}

			// 全てのSQLが終了した時にロックを開放する
			try {
				dataBaseConnection.unlockConnection();
			} catch (DbConnectIllgalStateException e) {
				// ありえない例外
				e.printStackTrace();
				throw new RuntimeException(e);
			}

			// 後始末リスナーの削除
			dataBaseConnection.removeConnectionlisteners(changeConnection);

			for (SqlExecutedListener l : sqlExecutedListeners.getListeners(SqlExecutedListener.class)) {
				l.executeAllEnd();
			}
		}

		private void sqlExec(SqlExecuteSentence sqlExecuteSentence) {
			sqlTaskThrow = null;
			timeMillisThreadStart = System.currentTimeMillis();
			timeMillisThreadEnd = 0;
			rowCnt = 0;
			try {
				Jdbcacsess2.logger.info("  (Statement)EXECSQL=[" + sqlExecuteSentence.getSqlSentence() + "]");
				for (SqlExecutedListener l : sqlExecutedListeners.getListeners(SqlExecutedListener.class)) {
					l.executBegin(sqlExecuteSentence);
				}

				if (sqlExecuteSentence.getSqlCommand().equals("CALL")) {
					preparedStatement = dataBaseConnection.prepareCall(sqlExecuteSentence.getSqlSentence());
				} else {
					preparedStatement = dataBaseConnection.prepareStatement(sqlExecuteSentence.getSqlSentence());
				}

				// パラメータ取り込み処理がfalseを返すまで繰り返す
				SqlInputParameter sqlInputParameter = sqlExecuteSentence.getSqlInputParameter();

				while (sqlInputParameter.getSqlExecuteParmeter().hasNext()) {
					if (Thread.interrupted()) {
						Jdbcacsess2.logger.info("  (task)canceled.");
						break;
					}

					// パラメータ設定を呼び出す
					Parameter parameter = sqlInputParameter.getSqlExecuteParmeter().getParameter();

					// preparedStatementのパラメータ設定
					setParameters(sqlInputParameter.getInputItemNames(), parameter);

					// SQL実行し、結果により受け取る方法を振り分ける
					if (preparedStatement.execute()) {
						readResultSet();
					} else {
						rowCnt += preparedStatement.getUpdateCount();
					}

				}

			} catch (Throwable t) {
				t.printStackTrace();
				sqlTaskThrow = t;
			} finally {
				try {
					if (preparedStatement != null) {
						preparedStatement.close();
						Jdbcacsess2.logger.info("  (task)Statement closed");
					}
				} catch (SQLException e) {
					Jdbcacsess2.logger.log(Level.WARNING, "WARNING", e);
				} finally {
					preparedStatement = null;

					timeMillisThreadEnd = System.currentTimeMillis();

					if (sqlTaskThrow == null) {
						for (SqlExecutedListener l : sqlExecutedListeners.getListeners(SqlExecutedListener.class)) {
							l.executNormalFinish(rowCnt);
						}
						Jdbcacsess2.logger.info("  (task)noramal end. cnt=[" + rowCnt + "] "
								+ (timeMillisThreadEnd - timeMillisThreadStart) + "ms");
					} else {
						for (SqlExecutedListener l : sqlExecutedListeners.getListeners(SqlExecutedListener.class)) {
							l.executeException(sqlTaskThrow);
						}
						Jdbcacsess2.logger.info("  (task)abnormal end. cnt=[" + rowCnt + "] "
								+ (timeMillisThreadEnd - timeMillisThreadStart) + "ms");
					}

				}
			}
		}

		/**
		 * JDBCprepareステートメントのパラメータを設定する。
		 * 
		 * @param sqlInputItems
		 *            パラメータ名称リスト
		 * @param p
		 *            KEY=パラメータリストのインデックス／VALUE=SQL型 のマップ。
		 */
		private void setParameters(ArrayList<String> sqlInputItems, Parameter p)
				throws SQLException {

			Jdbcacsess2.logger.fine("  (setParameters)VALUES=" + p.values + "SQLTYPES=" + p.sqlTypes + "");

			if (sqlInputItems.size() != p.values.size()) {
				throw new IllegalArgumentException("入力パラ数が分析結果と不一致:" + "Input=" + p.values.size() + " analized="
						+ sqlInputItems);
			}

			for (int i = 0; i < p.values.size(); i++) {
				Object o = p.values.get(i);
				ConstSqlTypes constSqlTypes = p.sqlTypes.get(i);

				if (constSqlTypes == null) {
					preparedStatement.setObject(i + 1, o);
				} else {
					preparedStatement.setObject(i + 1, o, constSqlTypes.getValue());
					Jdbcacsess2.logger.fine("  (setParameters)" + constSqlTypes);
				}
			}
		}

		/**
		 * 検索結果を取り出す
		 * 
		 * @throws Throwable
		 */
		private void readResultSet() throws Throwable {

			ResultSet resultSet = preparedStatement.getResultSet();
			// getMetaData()は、getResultSet()を実行してから行う。
			// sqliteは、"IllegalStateException: SQLite JDBC: inconsistent internal state"
			// 例外が発生する。
			List<ColumnAttributeResult> list = ColumnAttributeResult.convColumnAttributeResult(resultSet.getMetaData());

			Jdbcacsess2.logger.info("  (task)resultSet opened.");
			for (SqlExecutedListener l : sqlExecutedListeners.getListeners(SqlExecutedListener.class)) {
				l.resultHeader(list);
			}

			int columnCnt = list.size();

			try {
				while (resultSet.next()) {
					synchronized (this) {
						if (Thread.interrupted()) {
							Jdbcacsess2.logger.info("  (task)canceled.");
							break;
						}
						ArrayList<Result> results = new ArrayList<Result>(columnCnt);
						for (int i = 1; i <= columnCnt; i++) {
							results.add(new Result(list.get(i - 1), resultSet.getObject(i)));
						}

						rowCnt++;
						for (SqlExecutedListener l : sqlExecutedListeners.getListeners(SqlExecutedListener.class)) {
							l.resultDetail(rowCnt, results);
						}
						Jdbcacsess2.logger.finest("  (task)results notified.");

						if (rowCnt % optionValues.propResultrowsLimitCnt.getValue() == 0) {

							available = true;
							new Thread() {
								@Override
								public void run() {
									for (SqlExecutedListener l : sqlExecutedListeners.getListeners(SqlExecutedListener.class)) {
										l.statusContinue(rowCnt);
									}
								}
							}.start();
							Jdbcacsess2.logger.info("  (task)waiting.");

							while (available) {
								try {
									wait();
								} catch (InterruptedException e) {
									Jdbcacsess2.logger.fine("  (task)interrupted.");
									// wait() により割り込みフラグがクリアされたので、
									// 自身にもう一度割り込みをかける。ループ脱出をを検知させる為。
									Thread.currentThread().interrupt();
									break;
								}
							}
							Jdbcacsess2.logger.info("  (task)wake up.");
						}
					}
				}
			} catch (Throwable t) {
				throw t;
			} finally {
				resultSet.close();
				Jdbcacsess2.logger.info("  (task)resultSet closed.");
			}
		}

		/**
		 * 待機中のSQL実行スレッドを再開する
		 */
		@Override
		public void taskWakeUp() {
			synchronized (this) {
				available = false;
				notifyAll();
			}
		}

		/**
		 * 実行中のSQL実行スレッドを中止を要求する
		 */
		@Override
		public void taskCancel() {
			thread.interrupt();
		}

		/**
		 * SQL実行スレッドの終了を待ち合わせる
		 */
		@Override
		public void taskJoin(int timeoutSeconds) {
			Jdbcacsess2.logger.info("  (join)joining...");
			try {
				Executors.newSingleThreadExecutor().submit(new Callable<Object>() {
					@Override
					public Object call() throws Exception {
						thread.join();
						return null;
					}
				}).get(timeoutSeconds, TimeUnit.SECONDS);
			} catch (InterruptedException e) {
				Jdbcacsess2.logger.log(Level.WARNING, "WARNING", e);
			} catch (ExecutionException e) {
				e.printStackTrace();
			} catch (TimeoutException e) {
				ShowDialog.errorMessage(e);
			}
			Jdbcacsess2.logger.info("  (join)joined.");
		}

		/**
		 * 
		 * @return 処理時間　単位はミリ秒
		 */
		@Override
		public long getExecutionTime() {
			if (timeMillisThreadStart == 0) {
				return 0;
			}
			if (timeMillisThreadEnd == 0) {
				return 0;
			}
			return timeMillisThreadEnd - timeMillisThreadStart;
		}

		@Override
		public int getCnt() {
			return sqlExecuteSentenceList.size();

		}
	}// class SqlTask end


}
