/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.sqlService.parse;

import java.util.ArrayList;
import java.util.HashMap;

import jdbcacsess2.sqlService.ConstSqlTypes;
import jdbcacsess2.sqlService.SqlExecuteParmeter;
import jdbcacsess2.sqlService.SqlExecuteParmeter.Parameter;

/**
 * 
 * SQL文の入力パラメータに関する情報を、１つの実行SQL文分だけ集めたクラスです。
 * １SQL文には複数のパラメータが想定されるので、それぞれの内容はArrayListでリスト化されています。
 * <ul>
 * <li>SQL文中に記述された入力パラメータの名称：{@link SqlExecuteSentencies.SqlExecuteSentence}
 * が解析した結果を格納する事を想定しています。
 * <li>入力パラメータの値とそのSQLTYPE：３パターンの使い方が出来ます。
 * <ol>
 * <li>パラメータが無い
 * <li>入力パラメータを指定しSQLを１回だけ実行する時
 * <li>１つのSQLで入力パラメータを切り替えながら順次実行する
 * </ol>
 * ３番目の使用方法は、PrepareしたSQL文に対して、入力パラメータの設定とexecuteが、複数回実行されるイメージです。
 * </ul>
 * 
 * @author sima
 * 
 */
public class SqlInputParameter {

	private final Parameter parameter;

	public SqlInputParameter() {
		parameter = new SqlExecuteParmeter.Parameter();
		parameter.values = new ArrayList<Object>();
		parameter.sqlTypes = new HashMap<Integer, ConstSqlTypes>();
	}

	/**
	 * 入力パラメータ名のSQL文字句要素
	 */
	final private ArrayList<String> inputItemNames = new ArrayList<String>();

	/**
	 * 入力パラメータ名のSQL文字句要素を返却します。このリストの順番に
	 * {@link SqlInputParameter#addParameter(Object, ConstSqlTypes)}
	 * でパラメータを設定する必要があります。
	 * 
	 * @return
	 */
	public ArrayList<String> getInputItemNames() {
		return inputItemNames;
	}

	/**
	 * 入力パラメータ名のSQL文字句要素を追加します。これは、{@link SqlExecuteSentencies}が結果格納する時に使用されます。
	 * 
	 * @param inputItemName
	 */
	void addInputItemName(String inputItemName) {
		inputItemNames.add(inputItemName);
	}

	/**
	 * パラメータの値とタイプをパラメータリストの最後に追加します。
	 * 
	 * @param value
	 *            値
	 * @param constSqlType
	 *            値のSQLTYPE
	 */
	public void addParameter(Object value, ConstSqlTypes constSqlType) {
		addParameter(parameter.values.size() - 1, value, constSqlType);
	}

	/**
	 * パラメータの値とタイプをパラメータリストの順番を指定して追加します。
	 * 
	 * @param index
	 *            {@link SqlInputParameter#inputItemNames}のリストに存在するindex
	 * @param value
	 *            値
	 * @param constSqlType
	 *            値のSQLTYPE
	 */
	public void addParameter(Integer index, Object value, ConstSqlTypes constSqlType) {
		inputItemNames.get(index); // check

		parameter.values.add(value);
		parameter.sqlTypes.put(index, constSqlType);
	}

	public SqlExecuteParmeter getSqlExecuteParmeter() {
		return sqlExecuteParmeter;
	}

	/**
	 * デフォルトではパラメータ値は通常１SQLで１組となっています。<br>
	 * 同一SQLで複数組を処理したい場合、値とSQLTYPEを順次外部から与える方法を指定出来ます。
	 * 
	 * @param sqlExecuteParmeter
	 *            {@link SqlExecuteParmeter}を実装した値とSQLTYPEの生成器。
	 */
	public void setSqlExecuteParmeter(SqlExecuteParmeter sqlExecuteParmeter) {
		this.sqlExecuteParmeter = sqlExecuteParmeter;
	}

	private SqlExecuteParmeter sqlExecuteParmeter = new SqlExecuteParmeter() {

		boolean flg = true;

		@Override
		public boolean hasNext() {
			// １回目の呼出はtrue、２回目以降はfalse
			if (flg) {
				flg = false;
				return true;
			}
			return false;
		}

		@Override
		public Parameter getParameter() {
			return parameter;
		}
	};

	/**
	 * 入力パラメータ名とパラメータ値を全て初期化します。
	 */
	public void clear() {
		inputItemNames.clear();
		parameter.values.clear();
		parameter.sqlTypes.clear();
	}

}
