/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jdbcacsess2.sqlService.history;

import java.sql.SQLException;
import java.util.Date;
import java.util.List;

import javax.swing.SwingUtilities;

import jdbcacsess2.main.Config;
import jdbcacsess2.main.ShowDialog;
import jdbcacsess2.sqlService.ColumnAttributeResult;
import jdbcacsess2.sqlService.Result;
import jdbcacsess2.sqlService.SqlExecuteTask;
import jdbcacsess2.sqlService.SqlExecutedListener;
import jdbcacsess2.sqlService.history.ExecHistory.ResultStatus;
import jdbcacsess2.sqlService.parse.SqlExecuteSentencies.SqlExecuteSentence;
import net.java.ao.DBParam;
import net.java.ao.EntityManager;

/**
 * @author sima
 *
 */
public class Record implements SqlExecutedListener {
	private final EntityManager entityManager;

	private ExecHistory execHistory;
	private int execHistoryIndex = -1;
	private final String connectName;
	private final String url;
	private final String user;
	private final HistryTableModel histryTableModel;

	public Record(Config config, String connectName, String url, String user, HistryTableModel histryTableModel) {
		entityManager = config.getEntityManager();

		this.connectName = connectName;
		this.url = url;
		this.user = user;

		this.histryTableModel = histryTableModel;
	}

	enum FireType {
		INSERT, UPDATE, DELETE
	}

	private class Fire implements Runnable {
		private final int index;
		private final FireType type;

		public Fire(int index, FireType type) {
			this.index = index;
			this.type = type;
		}

		@Override
		public void run() {
			switch (type) {
			case INSERT:
				histryTableModel.fireTableRowsInserted(index, index);
				break;
			case UPDATE:
				histryTableModel.fireTableRowsUpdated(index, index);
				break;

			default:
				break;
			}
		}

	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see
	 * jdbcacsess2.sqlService.SqlExecutedListener#taskAccept(jdbcacsess2.sqlService
	 * .SqlExecuteTask)
	 */
	@Override
	public void taskAccept(SqlExecuteTask sqlExecuteTask) {
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see
	 * jdbcacsess2.sqlService.SqlExecutedListener#executBegin(jdbcacsess2.sqlService
	 * .SqlExecuteSentencies.SqlExecuteSentence)
	 */
	@Override
	public void executBegin(final SqlExecuteSentence sqlExecuteSentence) {
		try {
			execHistory = entityManager.create(ExecHistory.class,
			                                   new DBParam(ExecHistory.CONNECTNAME, connectName),
			                                   new DBParam(ExecHistory.STARTDATE, new Date()),
			                                   new DBParam(ExecHistory.URL, url),
			                                   new DBParam(ExecHistory.CONNECTUSER, user));
			execHistory.setSqlCommand(sqlExecuteSentence.getSqlCommand());
			execHistory.setSqlText(sqlExecuteSentence.getSqlSentence());
			execHistory.setResultStatus(ResultStatus.ACCEPT);
			execHistoryIndex = histryTableModel.add(execHistory);
			saveLater(execHistory);

			SwingUtilities.invokeLater(new Fire(execHistoryIndex, FireType.INSERT));
		} catch (SQLException e) {
			ShowDialog.errorMessage(e);
		}
	}

	/* (非 Javadoc)
	 * @see jdbcacsess2.sqlService.SqlExecutedListener#executNormalFinish(int)
	 */
	@Override
	public void executNormalFinish(final int rowCnt) {
		if (execHistory == null) {
			return;
		}

		execHistory.setResultCnt(rowCnt);
		if (execHistory.getEndDate() == null) {
			execHistory.setEndDate(new Date());
		}
		execHistory.setResultStatus(ResultStatus.NORMALEND);
		saveLater(execHistory);
		SwingUtilities.invokeLater(new Fire(execHistoryIndex, FireType.UPDATE));
	}

	/* (非 Javadoc)
	 * @see jdbcacsess2.sqlService.SqlExecutedListener#executeAllEnd()
	 */
	@Override
	public void executeAllEnd() {
	}

	/* (非 Javadoc)
	 * @see jdbcacsess2.sqlService.SqlExecutedListener#executeException(java.lang.Throwable)
	 */
	@Override
	public void executeException(Throwable t) {
		if (execHistory == null) {
			return;
		}

		if (execHistory.getEndDate() == null) {
			execHistory.setEndDate(new Date());
		}
		execHistory.setResultStatus(ResultStatus.ABNORMALEND);
		saveLater(execHistory);
		SwingUtilities.invokeLater(new Fire(execHistoryIndex, FireType.UPDATE));
	}

	/* (非 Javadoc)
	 * @see jdbcacsess2.sqlService.SqlExecutedListener#resultDetail(int, java.util.List)
	 */
	@Override
	public void resultDetail(int seq, List<Result> detail) {
	}

	/* (非 Javadoc)
	 * @see jdbcacsess2.sqlService.SqlExecutedListener#resultHeader(java.util.List)
	 */
	@Override
	public void resultHeader(List<ColumnAttributeResult> header) {
		if (execHistory == null) {
			return;
		}

		execHistory.setEndDate(new Date());
		saveLater(execHistory);
		SwingUtilities.invokeLater(new Fire(execHistoryIndex, FireType.UPDATE));
	}

	/* (非 Javadoc)
	 * @see jdbcacsess2.sqlService.SqlExecutedListener#statusContinue(int)
	 */
	@Override
	public void statusContinue(final int seqNo) {
		if (execHistory == null) {
			return;
		}

		execHistory.setResultCnt(seqNo);
		if (execHistory.getEndDate() == null) {
			execHistory.setEndDate(new Date());
		}
		execHistory.setResultStatus(ResultStatus.EXECUTING);
		saveLater(execHistory);
		SwingUtilities.invokeLater(new Fire(execHistoryIndex, FireType.UPDATE));
	}

	private void saveLater(final ExecHistory execHistory) {
		Thread thread = new Thread() {
			@Override
			public void run() {
				execHistory.save();
			}
		};
		thread.start();
	}
}
