/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.sqlService.dbobject;

import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import javax.swing.Icon;

import jdbcacsess2.main.Jdbcacsess2;
import jdbcacsess2.sqlService.ReferenceColumnResult;
import jdbcacsess2.sqlService.dbobject.DBObjectCatalogs.DBObjectCatalog;
import jdbcacsess2.sqlService.dbobject.DBObjectSchemas.DBObjectSchema;
import jdbcacsess2.sqlService.dbobject.DBObjectTableTypes.DBObjectTableType;
import jdbcacsess2.sqlService.dbobject.icon.DbObjectIconImage;

/**
 * TABLEの一覧
 * 
 * @author sima
 * 
 */
public class DBObjectTables extends DBObjects {

	private final ArrayList<DBObjectTable> tables = new ArrayList<DBObjectTable>();

	private final boolean storesLowerCaseIdentifiers;
	private final boolean storesUpperCaseQuotedIdentifiers;

	private final DBObjectTableType dbObTableType;

	/**
	 * コンストラクタ
	 * 
	 * @param schema
	 * @param tableType
	 * @param dmd
	 * @throws SQLException
	 */
	public DBObjectTables(DBObjectCatalog catalog, DBObjectSchema schema, DBObjectTableType tableType,
	                      DatabaseMetaData dmd) throws SQLException {

		storesLowerCaseIdentifiers = dmd.storesLowerCaseIdentifiers();
		storesUpperCaseQuotedIdentifiers = dmd.storesUpperCaseQuotedIdentifiers();

		dbObTableType = tableType;

		String[] type = new String[1];
		type[0] = tableType.tableTypeName;
		ResultSet rs;

		String catalogName = null;
		if (catalog != null) {
			catalogName = catalog.catalogName;
		}

		String schemaName = null;
		if (schema != null) {
			schemaName = schema.schemaName;
		}

		rs = dmd.getTables(catalogName, schemaName, null, type);
		while (rs.next()) {
			DBObjectTable e = new DBObjectTable(rs, dmd);
			tables.add(e);

		}
		rs.close();
	}

	@Override
	public ArrayList<DBObjectTable> getDBObjects() {
		return tables;
	}

	@Override
	public String toString() {
		return "TABLE";
	}

	/**
	 * TABLEのプロパティ
	 * 
	 * @author sima
	 * 
	 */
	public class DBObjectTable extends DBObject {

		private final List<ReferenceColumnResult> exportedKey;

		public List<ReferenceColumnResult> getExportedKey() {
			return exportedKey;
		}


		/**
		 * {@link DatabaseMetaData#getTables(String, String, String, String[])}
		 * の第１要素
		 */
		public String tableCatalog;

		/**
		 * {@link DatabaseMetaData#getTables(String, String, String, String[])}
		 * の第２要素
		 */
		public String tableSchema;

		private String tableName;

		/**
		 * {@link DatabaseMetaData#getTables(String, String, String, String[])}
		 * の第３要素
		 * 
		 * @return
		 */
		public String getTableName() {
			return tableName;
		}

		private String tableType;

		/**
		 * {@link DatabaseMetaData#getTables(String, String, String, String[])}
		 * の第４要素
		 * 
		 * @return
		 */
		public String getTableType() {
			return tableType;
		}

		private String remarks;

		/**
		 * {@link DatabaseMetaData#getTables(String, String, String, String[])}
		 * の第５要素
		 * 
		 * @return
		 */
		public String getRemarks() {
			if (remarks == null) {
				return "";
			}
			return remarks;
		}

		private String typeCatalog;

		/**
		 * {@link DatabaseMetaData#getTables(String, String, String, String[])}
		 * の第６要素
		 * 
		 * @return
		 */
		public String getTypeCatalog() {
			return typeCatalog;
		}

		private String typeSchema;

		/**
		 * {@link DatabaseMetaData#getTables(String, String, String, String[])}
		 * の第７要素
		 * 
		 * @return
		 */
		public String getTypeSchema() {
			return typeSchema;
		}

		private String typeName;

		/**
		 * {@link DatabaseMetaData#getTables(String, String, String, String[])}
		 * の第８要素
		 * 
		 * @return
		 */
		public String getTypeName() {
			return typeName;
		}

		private String selfReferencingColName;

		/**
		 * {@link DatabaseMetaData#getTables(String, String, String, String[])}
		 * の第９要素
		 * 
		 * @return
		 */
		public String getSelfReferencingColName() {
			return selfReferencingColName;
		}

		private String refGeneration;

		/**
		 * {@link DatabaseMetaData#getTables(String, String, String, String[])}
		 * の第１０要素
		 * 
		 * @return
		 */
		public String getRefGeneration() {
			return refGeneration;
		}

		private final List<Property> properties = new ArrayList<Property>();

		@Override
		public List<Property> getProperties() {
			return properties;
		}
		/**
		 * コンストラクタ
		 * 
		 * @param rs
		 * @throws SQLException
		 */
		public DBObjectTable(ResultSet rs, DatabaseMetaData dmd) throws SQLException {

			int colCnt = rs.getMetaData().getColumnCount();
			int i = 0;
			if (colCnt > i) {
				tableCatalog = rs.getString(++i);
				properties.add(new Property("TABLE_CAT", tableCatalog));
			}
			if (colCnt > i) {
				tableSchema = rs.getString(++i);
				properties.add(new Property("TABLE_SCHEM", tableSchema));
			}
			if (colCnt > i) {
				tableName = rs.getString(++i);
				properties.add(new Property("TABLE_NAME", tableName));
			}
			if (colCnt > i) {
				tableType = rs.getString(++i);
				properties.add(new Property("TABLE_TYPE", tableType));
			}
			if (colCnt > i) {
				remarks = rs.getString(++i);
				properties.add(new Property("REMARKS", remarks));
			}
			if (colCnt > i) {
				typeCatalog = rs.getString(++i);
				properties.add(new Property("TYPE_CAT", typeCatalog));
			}
			if (colCnt > i) {
				typeSchema = rs.getString(++i);
				properties.add(new Property("TYPE_SCHEM", typeSchema));
			}
			if (colCnt > i) {
				typeName = rs.getString(++i);
				properties.add(new Property("TYPE_NAME", typeName));
			}
			if (colCnt > i) {
				selfReferencingColName = rs.getString(++i);
				properties.add(new Property("SELF_REFERENCING_COL_NAME", selfReferencingColName));
			}
			if (colCnt > i) {
				refGeneration = rs.getString(++i);
				properties.add(new Property("REF_GENERATION", refGeneration));
			}

			ResultSet exportedKeys = dmd.getExportedKeys(tableCatalog, tableSchema, tableName);
			List<ReferenceColumnResult> rcr = ReferenceColumnResult.convReferenceColumnResult(exportedKeys);
			if (rcr.size() != 0) {
				Jdbcacsess2.logger.fine(rcr.toString());
			}
			exportedKeys.close();
			exportedKey = rcr;
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#toString()
		 */
		@Override
		public String toString() {
			return tableName;
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @seejdbcacsess2.sqlService.dbobject.DBObject#getChildren(java.sql.
		 * DatabaseMetaData)
		 */
		@Override
		public DBObjects getChildren(DatabaseMetaData dmd) throws SQLException {
			return new DBObjectTableColumns(this, dmd);
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#getSchemaTableName()
		 */
		@Override
		public SchemaTableName getSchemaTableName() throws SQLException {
			return new SchemaTableName(storesLowerCaseIdentifiers,
			                           storesUpperCaseQuotedIdentifiers,
			                           tableSchema,
			                           tableName);
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#getName()
		 */
		@Override
		public String getName() {
			return tableName;
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#getIconImage()
		 */
		@Override
		public Icon getIconImage() {
			if (dbObTableType.tableTypeName.indexOf("TABLE") != -1
					|| dbObTableType.tableTypeName.indexOf("table") != -1) {
				return DbObjectIconImage.TABLE.getValue();
			}
			if (dbObTableType.tableTypeName.equals("VIEW")) {
				return DbObjectIconImage.VIEW.getValue();
			}
			return DbObjectIconImage.TABLETYPE.getValue();
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#isStrong()
		 */
		@Override
		public boolean isStrong() {
			return false;
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#isBottom()
		 */
		@Override
		public boolean isBottom() {
			return false;
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#getSummary()
		 */
		@Override
		public String getSummary() {
			return getName();
		}

		@Override
		public String getProperty1() {
			return getRemarks();
		}

		@Override
		public String getProperty2() {
			return getSelfReferencingColName();
		}

		@Override
		public String getProperty3() {
			return getRefGeneration();
		}

		@Override
		public boolean isPropertyEnable() {
			return true;
		}
	}

}
