/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.sqlService.dbobject;

import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import javax.swing.Icon;

import jdbcacsess2.sqlService.dbobject.DBObjectCatalogs.DBObjectCatalog;
import jdbcacsess2.sqlService.dbobject.icon.DbObjectIconImage;

/**
 * SCHEMAの一覧
 * 
 * @author sima
 * 
 */
public class DBObjectSchemas extends DBObjects {


	private final ArrayList<DBObjectSchema> schemas = new ArrayList<DBObjectSchema>();

	DBObjectCatalog dbCatalog;

	/**
	 * コンストラクタ
	 * 
	 * @param dmd
	 * @throws SQLException
	 */
	public DBObjectSchemas(DBObjectCatalog catalog, DatabaseMetaData dmd) throws SQLException {
		dbCatalog = catalog;

		ResultSet rs = dmd.getSchemas();
		while (rs.next()) {
			schemas.add(new DBObjectSchema(rs));
		}
		rs.close();
	}

	@Override
	public ArrayList<? extends DBObject> getDBObjects() {
		return schemas;
	}

	@Override
	public String toString() {
		return "SCHEMA";
	}

	/**
	 * SCHEMAのプロパティ
	 * 
	 * @author sima
	 * 
	 */
	public class DBObjectSchema extends DBObject {

		/**
		 * スキーマ名 {@link DatabaseMetaData#getSchemas()}の第１要素
		 */
		public String schemaName;

		/**
		 * カタログ名 {@link DatabaseMetaData#getSchemas()}の第２要素
		 */
		public String catalogName;

		private final List<Property> properties = new ArrayList<Property>();

		@Override
		public List<Property> getProperties() {
			return properties;
		}

		public DBObjectSchema(ResultSet rs) throws SQLException {
			int colCnt = rs.getMetaData().getColumnCount();
			int i = 0;
			if (colCnt > i) {
				schemaName = rs.getString(++i);
				properties.add(new Property("TABLE_SCHEM", schemaName));
			}
			if (colCnt > i) {
				catalogName = rs.getString(++i);
				properties.add(new Property("TABLE_CATALOG", catalogName));
			}
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @seejdbcacsess2.sqlService.dbobject.DBObject#getChildren(java.sql.
		 * DatabaseMetaData)
		 */
		@Override
		public DBObjects getChildren(DatabaseMetaData dmd) throws SQLException {
			return new DBObjectTableTypes(dbCatalog, this, dmd);
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#getName()
		 */
		@Override
		public String getName() {
			return schemaName;
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#getIconImage()
		 */
		@Override
		public Icon getIconImage() {
			return DbObjectIconImage.SCHEMA.getValue();
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#isStrong()
		 */
		@Override
		public boolean isStrong() {
			return false;
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#isBottom()
		 */
		@Override
		public boolean isBottom() {
			return false;
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#getSummary()
		 */
		@Override
		public String getSummary() {
			if (schemaName == null) {
				return "(無し)";
			}
			return schemaName;
		}

		@Override
		public boolean isPropertyEnable() {
			return false;
		}
	}
}
