/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.sqlService.dbobject;

import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import javax.swing.Icon;

import jdbcacsess2.sqlService.dbobject.DBObjectProcedures.DBObjectProcedure;
import jdbcacsess2.sqlService.dbobject.icon.DbObjectIconImage;

/**
 * TABLECOLUMNの一覧
 * 
 * @author sima
 * 
 */
public class DBObjectProcedureColumns extends DBObjects {


	private final ArrayList<DBObjectProcedureColumn> procedureColumns = new ArrayList<DBObjectProcedureColumn>();

	/**
	 * コンストラクタ
	 * 
	 * @param table
	 * @param dmd
	 * @throws SQLException
	 */
	public DBObjectProcedureColumns(DBObjectProcedure procedure, DatabaseMetaData dmd) throws SQLException {

		ResultSet rs = dmd.getProcedureColumns(procedure.getProcedureCatalog(),
		                                       procedure.getProcedureSchema(),
		                                       procedure.getProcedureName(),
		                                       null);
		while (rs.next()) {
			procedureColumns.add(new DBObjectProcedureColumn(rs));
		}
		rs.close();

	}

	@Override
	public ArrayList<? extends DBObject> getDBObjects() {
		return procedureColumns;
	}

	@Override
	public String toString() {
		return "PROCEDURE_COLUMNS";
	}

	/**
	 * TABLECOLUMNのプロパティ
	 * 
	 * @author sima
	 * 
	 */
	public static class DBObjectProcedureColumn extends DBObject {
		/**
		 * {@link DatabaseMetaData#getProcedureColumns(String, String, String, String)}
		 * の第１要素
		 */
		public String procedureCatalog;

		/**
		 * {@link DatabaseMetaData#getProcedureColumns(String, String, String, String)}
		 * の第２要素
		 */
		public String procedureSchema;

		/**
		 * {@link DatabaseMetaData#getProcedureColumns(String, String, String, String)}
		 * の第３要素
		 */
		public String procedureName;

		/**
		 * {@link DatabaseMetaData#getProcedureColumns(String, String, String, String)}
		 * の第４要素
		 */
		public String columnName;

		/**
		 * {@link DatabaseMetaData#getProcedureColumns(String, String, String, String)}
		 * の第５要素
		 */
		public Short columnType;

		private String getColumnTypeName() {
			switch (columnType) {
			case DatabaseMetaData.procedureColumnUnknown:
				return "ColumnUnknown";
			case DatabaseMetaData.procedureColumnIn:
				return "ColumnIn";
			case DatabaseMetaData.procedureColumnInOut:
				return "ColumnInOut";
			case DatabaseMetaData.procedureColumnOut:
				return "ColumnOut";
			case DatabaseMetaData.procedureColumnReturn:
				return "ColumnReturn";
			case DatabaseMetaData.procedureColumnResult:
				return "ColumnResult";
			}
			return "?";
		}


		/**
		 * {@link DatabaseMetaData#getProcedureColumns(String, String, String, String)}
		 * の第６要素
		 */
		public int dataType;

		/**
		 * {@link DatabaseMetaData#getProcedureColumns(String, String, String, String)}
		 * の第７要素
		 */
		public String typeName;

		/**
		 * {@link DatabaseMetaData#getProcedureColumns(String, String, String, String)}
		 * の第８要素
		 */
		public int precision;

		/**
		 * {@link DatabaseMetaData#getProcedureColumns(String, String, String, String)}
		 * の第９要素
		 */
		public int length;

		/**
		 * {@link DatabaseMetaData#getProcedureColumns(String, String, String, String)}
		 * の第１０要素
		 */
		public short scale;

		/**
		 * {@link DatabaseMetaData#getProcedureColumns(String, String, String, String)}
		 * の第１１要素
		 */
		public short radix;

		/**
		 * {@link DatabaseMetaData#getProcedureColumns(String, String, String, String)}
		 * の第１２要素
		 */
		public short nullable;

		private String getNullableName() {
			switch (nullable) {
			case DatabaseMetaData.procedureNoNulls:
				return "NoNulls";
			case DatabaseMetaData.procedureNullable:
				return "Nullable";
			case DatabaseMetaData.procedureNullableUnknown:
				return "NullableUnknown";
			}
			return "?";
		}

		/**
		 * {@link DatabaseMetaData#getProcedureColumns(String, String, String, String)}
		 * の第１３要素
		 */
		public String remarks;

		/**
		 * {@link DatabaseMetaData#getProcedureColumns(String, String, String, String)}
		 * の第１４要素
		 */
		public String columnDef;

		/**
		 * {@link DatabaseMetaData#getProcedureColumns(String, String, String, String)}
		 * の第１７要素
		 */
		public int charOctetLength;

		/**
		 * {@link DatabaseMetaData#getProcedureColumns(String, String, String, String)}
		 * の第１８要素
		 */
		public int ordinalPosition;

		/**
		 * {@link DatabaseMetaData#getProcedureColumns(String, String, String, String)}
		 * の第１９要素
		 */
		public String isNullable;

		/**
		 * {@link DatabaseMetaData#getProcedureColumns(String, String, String, String)}
		 * の第２０要素
		 */
		public String specificName;

		private final StringBuilder attr;

		private final List<Property> properties = new ArrayList<Property>();

		@Override
		public List<Property> getProperties() {
			return properties;
		}

		/**
		 * コンストラクタ
		 * 
		 * @param rs
		 * @param pkmap
		 * @throws SQLException
		 */
		public DBObjectProcedureColumn(ResultSet rs) throws SQLException {

			int colCnt = rs.getMetaData().getColumnCount();
			int i=0;
			if (colCnt > i){
				procedureCatalog = rs.getString(++i);
				properties.add(new Property("PROCEDURE_CAT", procedureCatalog));
			}
			if (colCnt > i){
				procedureSchema = rs.getString(++i);
				properties.add(new Property("PROCEDURE_SCHEM", procedureSchema));
			}
			if (colCnt > i){
				procedureName = rs.getString(++i);
				properties.add(new Property("PROCEDURE_NAME", procedureName));
			}

			if (colCnt > i){
				columnName = rs.getString(++i);
				properties.add(new Property("COLUMN_NAME",columnName));
			}
			if (colCnt > i){
				columnType = rs.getShort(++i);
				properties.add(new Property("COLUMN_TYPE", getColumnTypeName()));
			}
			if (colCnt > i){
				dataType = rs.getInt(++i);
				properties.add(new Property("DATA_TYPE",Integer.toString(dataType)));
			}
			if (colCnt > i){
				typeName = rs.getString(++i);
				properties.add(new Property("TYPE_NAME",typeName));
			}

			if (colCnt > i){
				precision = rs.getInt(++i);
				properties.add(new Property("PRECISION",Integer.toString(precision)));
			}
			if (colCnt > i){
				length = rs.getInt(++i);
				properties.add(new Property("LENGTH", Integer.toString(length)));
			}
			if (colCnt > i){
				scale = rs.getShort(++i);
				properties.add(new Property("SCALE", Short.toString(scale)));
			}
			if (colCnt > i) {
				radix = rs.getShort(++i);
				properties.add(new Property("RADIX",Short.toString(radix)));
			}

			if (colCnt > i){
				nullable = rs.getShort(++i);
				properties.add(new Property("NULLABLE", getNullableName()));
			}

			if (colCnt > i){
				remarks = rs.getString(++i);
				properties.add(new Property("REMARKS", remarks));
			}
			if (colCnt > i){
				columnDef = rs.getString(++i);
				properties.add(new Property("COLUMN_DEF", columnDef));
			}

			if (colCnt > i){
				++i;
				/* "SQL_DATA_TYPE" */
			}
			if (colCnt > i){
				++i;
				/* "SQL_DATETIME_SUB" */
			}

			if (colCnt > i){
				charOctetLength = rs.getInt(++i);
				properties.add(new Property("CHAR_OCTET_LENGTH", Integer.toString(charOctetLength)));
			}
			if (colCnt > i){
				ordinalPosition = rs.getInt(++i);
				properties.add(new Property("ORDINAL_POSITION", Integer.toString(ordinalPosition)));
			}

			if (colCnt > i){
				isNullable = rs.getString(++i);
				properties.add(new Property("IS_NULLABLE", isNullable));
			}

			if (colCnt > i){
				specificName = rs.getString(++i);
				properties.add(new Property("SPECIFIC_NAME", specificName));
			}


			attr = editAttrString(dataType, precision, scale);

		}

		/*
		 * (非 Javadoc)
		 * 
		 * @seejdbcacsess2.sqlService.dbobject.DBObject#getChildren(java.sql.
		 * DatabaseMetaData)
		 */
		@Override
		public DBObjects getChildren(DatabaseMetaData dmd) throws SQLException {
			return null;
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#getName()
		 */
		@Override
		public String getName() {
			return columnName;
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#getIconImage()
		 */
		@Override
		public Icon getIconImage() {
			return DbObjectIconImage.TABLECOLUMN.getValue();
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#isStrong()
		 */
		@Override
		public boolean isStrong() {
			return false;
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#isBottom()
		 */
		@Override
		public boolean isBottom() {
			return true;
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#getSummary()
		 */
		@Override
		public String getSummary() {
			return getName();
		}

		@Override
		public String getProperty1() {
			return attr.toString();
		}

		@Override
		public String getProperty2() {
			return remarks;
		}

		@Override
		public boolean isPropertyEnable() {
			return true;
		}
	}

}
