/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.sqlService;

import java.io.File;
import java.io.FileNotFoundException;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.Driver;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;

import javax.swing.event.EventListenerList;

import jdbcacsess2.main.Jdbcacsess2;
import jdbcacsess2.main.OptionValues;
import jdbcacsess2.sqlService.exception.DbConnectAlreadyException;
import jdbcacsess2.sqlService.exception.DbConnectDriverLoadException;
import jdbcacsess2.sqlService.exception.DbConnectIllgalStateException;
import jdbcacsess2.sqlService.exception.DbConnectNotConectException;

/**
 * java.sql.Connectionの一部を委譲しており、データベースへの接続を行います。<br>
 * 接続情報でインスタンス生成後、createConnectionを実行することで接続が行われます。<br>
 * リスナーを登録すれば接続開始・接続終了のイベントを受け取れます。<br>
 * 接続中状態の時に接続要求すると既接続例外が発生します。 jarは必ず指定する必要があります。
 * 
 * @author sima
 * 
 */
public class DataBaseConnection {

	private Connection connection;

	private final String url;
	private final String user;
	private final String password;
	private final String driverClassName;
	private final URL[] driverUrlPaths;
	private Driver driver;
	private String connectName;
	private final OptionValues optionValues;

	/**
	 * Connection の open/close等のコネクション状態変更またはトランザクションの開始、AutoCommitの状態変更
	 * を通知するリスナーリスト
	 */
	private final EventListenerList listeners = new EventListenerList();

	/**
	 * connectionの使用中であるか
	 */
	private volatile boolean use = false;

	/**
	 * connectionの使用中のSQL文
	 */
	private String sql;

	/**
	 * SQLドライバーをキャシュする
	 */
	static final Map<String, Driver> driverCaches = new HashMap<String, Driver>();

	/**
	 * シングルスレッド実行のExecutorService
	 */
	private ExecutorService executorService;

	/**
	 * １つのconnectionで複数のSQL文を実行しないように、SQL実行開始時にロックを取得する。
	 * 
	 * @param sql
	 *            実行するSQL文。例外発生時にメッセージとして使われます。
	 * @throws DbConnectIllgalStateException
	 *             SQL実行中の場合
	 */
	public synchronized void lockConnection(String sql) throws DbConnectIllgalStateException {
		if (use) {
			throw new DbConnectIllgalStateException("database connection in use.[" + this.toString() + "]"
					+ this.sql);
		}
		use = true;
		this.sql = sql;
	}

	/**
	 * #lockConnectionで取得したロックを開放する。
	 * 
	 * @throws DbConnectIllgalStateException
	 *             SQL未実行の場合
	 */
	public synchronized void unlockConnection() throws DbConnectIllgalStateException {
		if (!use) {
			throw new DbConnectIllgalStateException("database connection not in use.[" + this.toString() + "]"
					+ this.sql);
		}
		use = false;
	}

	/**
	 * 接続開始・終了リスナーの登録
	 * 
	 * @param listener
	 */
	public void addConnectionListener(DataBaseConnectionListener listener) {
		listeners.add(DataBaseConnectionListener.class, listener);
	}

	/**
	 * 接続開始・終了リスナーの解除
	 */
	public void removeConnectionlisteners(DataBaseConnectionListener listener) {
		listeners.remove(DataBaseConnectionListener.class, listener);
	}

	/**
	 * トランザクション終了リスナーの登録
	 * 
	 * @param listener
	 */
	public void addTransactionListener(DataBaseTransactionListener listener) {
		listeners.add(DataBaseTransactionListener.class, listener);
	}

	/**
	 * トランザクション終了リスナーの解除
	 */
	public void removeTransactionlisteners(DataBaseTransactionListener listener) {
		listeners.remove(DataBaseTransactionListener.class, listener);
	}

	/**
	 * コンストラクタ。
	 * 
	 * @param url
	 * @param user
	 * @param password
	 * @param driverClassName
	 * @param driverUrlPaths
	 * @param connectName
	 *            この接続につける名前(コメント)。nullを指定した場合、空文字列として処理されます。
	 * @throws DbConnectDriverLoadException
	 * @throws FileNotFoundException
	 */
	public DataBaseConnection(String url, String user, String password, String driverClassName, URL[] driverUrlPaths,
	                          String connectName, OptionValues optionValues) throws DbConnectDriverLoadException, FileNotFoundException {
		this.optionValues = optionValues;
		this.url = url;
		this.user = user;
		this.password = password;
		this.driverClassName = driverClassName;
		this.driverUrlPaths = driverUrlPaths;
		if (connectName == null) {
			this.connectName = "";
		} else {
			this.connectName = connectName;
		}
		loadDriver();
	}

	/**
	 * コンストラクタ。既存インスタンスと同じ接続情報 接続先情報が同じで異なるDBコネクションを、作成する場合に使用する。
	 * 但し、登録済みのリスナーはコピーされるないので、必要に応じて再登録すること。
	 * 
	 * @param dataBaseConnection
	 * @throws DbConnectDriverLoadException
	 * @throws FileNotFoundException
	 */
	public DataBaseConnection(DataBaseConnection dataBaseConnection) throws DbConnectDriverLoadException,
	FileNotFoundException {
		this.url = dataBaseConnection.getUrl();
		this.user = dataBaseConnection.getUser();
		this.password = dataBaseConnection.password;
		this.driverClassName = dataBaseConnection.getDriverClassName();
		this.driverUrlPaths = dataBaseConnection.driverUrlPaths;
		this.connectName = dataBaseConnection.connectName;
		this.optionValues = dataBaseConnection.optionValues;
		loadDriver();
	}

	/**
	 * DriverManagerを使用せずに、直接JDBCドライバをロードする。
	 * 
	 * @throws DbConnectDriverLoadException
	 *             チェック例外だけでなく非チェック例外もラップされる。
	 * @throws FileNotFoundException
	 */
	private void loadDriver() throws DbConnectDriverLoadException, FileNotFoundException {

		try {
			String key;
			if (driverUrlPaths.length == 0) {
				key = "";
			} else {

				List<URL> list = Arrays.asList(driverUrlPaths);
				for (URL url : list) {
					File f = new File(url.toURI());
					if (!f.canRead()) {
						throw new FileNotFoundException("not found or can't read:" + f.toString());
					}
				}
				key = list.toString() + driverClassName;
			}
			synchronized (driverCaches) {
				if (driverCaches.containsKey(key)) {
					driver = driverCaches.get(key);
				} else {
					if (driverUrlPaths.length == 0) {
						driver = (Driver) Class.forName(driverClassName).newInstance();
					} else {
						driver = (Driver) Class.forName(driverClassName, true, new URLClassLoader(driverUrlPaths)).newInstance();
					}
					// ドライバファイルパスをキーにして、ドライバインスタンスをキャッシュする
					driverCaches.put(key, driver);
					Jdbcacsess2.logger.info(driverCaches + " size=" + driverCaches.size());
				}
			}
		} catch (FileNotFoundException e) {
			throw e;
		} catch (Throwable e) {
			// Class.forname
			// で、存在しないクラスが指定された場合は、NoClassDefFoundError(ExceptionでなくError)が発生する。
			// ExceptionとErrorを全部一つにまとめて、自前で作成したEXceptionに載せ替える。
			throw new DbConnectDriverLoadException(e);
		}

	}

	/**
	 * コンストラクタで指定された設定情報にもとづいて connection を作成し、接続開始リスナーにイベント通知する。
	 * 
	 * @throws SQLException
	 * @throws DataBaseConnectionAlreadyConectException
	 */
	public void open() throws SQLException, DbConnectAlreadyException {
		if (connection != null) {
			throw new DbConnectAlreadyException("Already opend.[" + url + "]");
		}
		Properties info = new java.util.Properties();
		info.put("user", user);
		info.put("password", password);
		connection = driver.connect(url, info);

		executorService = Executors.newSingleThreadExecutor();

		for (DataBaseConnectionListener listener : listeners.getListeners(DataBaseConnectionListener.class)) {
			Jdbcacsess2.logger.fine("***opened***" + " " + listener.toString());
			listener.dataBaseConnectionOpened(this);
		}
	}

	/**
	 * JDBC接続URL文字列を返却。
	 * 
	 * @return JDBC接続URL
	 */
	public String getUrl() {
		return url;
	}

	/**
	 * ユーザを返却
	 * 
	 * @return ユーザ
	 */
	public String getUser() {
		return user;
	}

	/**
	 * ドライバクラス名文字列を返却
	 * 
	 * @return ドライバクラス名
	 */
	public String getDriverClassName() {
		return driverClassName;
	}

	/**
	 * 名前を返却する。
	 * 
	 * @return この接続につけた名前。
	 */
	public String getConnectName() {
		return connectName;
	}

	/**
	 * クローズし、接続終了リスナーにイベント通知
	 * 
	 * @throws DbConnectIllgalStateException
	 *             SQL実行中の場合
	 */
	public void close() throws DbConnectIllgalStateException {
		for (DataBaseConnectionListener listener : listeners.getListeners(DataBaseConnectionListener.class)) {
			Jdbcacsess2.logger.fine("***closing***" + " " + listener.toString());
			listener.dataBaseConnectionClosing(this);
		}

		if (use) {
			throw new DbConnectIllgalStateException("database connection in use.[close]");
		}

		if (connection != null) {
			List<Runnable> task = executorService.shutdownNow();
			Jdbcacsess2.logger.log(Level.INFO, "shutdownNow rtn:" + task.toString() + " isTerminated:"
					+ executorService.isTerminated());

			try {
				Executors.newSingleThreadExecutor().submit(new Callable<Object>() {
					@Override
					public Object call() throws Exception {
						connection.close();
						return null;
					}
				}).get(getTimeoutSeconds(), TimeUnit.SECONDS);

				// CLOSEが成功したときだけ通知する。
				for (DataBaseConnectionListener listener : listeners.getListeners(DataBaseConnectionListener.class)) {
					Jdbcacsess2.logger.fine("***closed***" + " " + listener.toString());
					listener.dataBaseConnectionClosed(this);
				}
			} catch (InterruptedException e) {
				Jdbcacsess2.logger.log(Level.WARNING, "connection close exception " + getConnectName(), e);
			} catch (ExecutionException e) {
				Jdbcacsess2.logger.log(Level.WARNING, "connection close exception " + getConnectName(), e);
			} catch (TimeoutException e) {
				Jdbcacsess2.logger.log(Level.WARNING, "connection close exception " + getConnectName(), e);
			}

			// CLOSEの成功失敗にかかわらず、コネクションはクリアする
			connection = null;
		}
	}

	/**
	 * セパレータで区切られたファイルパス文字列を、独自クラスローダを作成する際のURL[]に変換する
	 * 
	 * @param urls
	 * @return コンストラクタで指定するURL配列
	 * @throws MalformedURLException
	 */
	static public URL[] splitPath(String urls) throws MalformedURLException {
		StringTokenizer st = new StringTokenizer(urls, File.pathSeparator);

		ArrayList<URL> jarUrlList = new ArrayList<URL>();
		while (st.hasMoreTokens()) {
			String s = st.nextToken();
			if (s.equals("")) {
				continue;
			}
			File file = new File(s);
			jarUrlList.add(file.toURI().toURL());

		}
		return jarUrlList.toArray(new URL[jarUrlList.size()]);
	}

	/**
	 * @throws Exception
	 * @see java.sql.Connection#commit()
	 */
	public void commit() throws Exception {
		if (isClosed()) {
			throw new DbConnectNotConectException();
		}

		try {
			getExecutorService().submit(new Callable<Object>() {
				@Override
				public Object call() throws SQLException {
					connection.commit();
					return null;
				}
			}).get(getTimeoutSeconds(), TimeUnit.SECONDS);

			// 成功したときだけ通知する。
			for (DataBaseTransactionListener listener : listeners.getListeners(DataBaseTransactionListener.class)) {
				listener.commitEnd(this);
			}

		} catch (Exception e1) {
			throw e1;
		}
	}

	/**
	 * @throws Exception
	 * @see java.sql.Connection#rollback()
	 */
	public void rollback() throws Exception {
		if (isClosed()) {
			throw new DbConnectNotConectException();
		}
		try {
			getExecutorService().submit(new Callable<Object>() {
				@Override
				public Object call() throws SQLException {
					connection.rollback();
					return null;
				}
			}).get(getTimeoutSeconds(), TimeUnit.SECONDS);

			// 成功したときだけ通知する。
			for (DataBaseTransactionListener listener : listeners.getListeners(DataBaseTransactionListener.class)) {
				listener.rollbackEnd(this);
			}

		} catch (Exception e1) {
			throw e1;
		}
	}

	/**
	 * @return autocommitの状態
	 * @throws SQLException
	 * @throws DbConnectNotConectException
	 * @see java.sql.Connection#getAutoCommit()
	 */
	public boolean getAutoCommit() throws SQLException, DbConnectNotConectException {
		if (isClosed()) {
			throw new DbConnectNotConectException();
		}
		return connection.getAutoCommit();
	}

	/**
	 * @param autoCommit
	 * @throws Exception
	 * @see java.sql.Connection#setAutoCommit(boolean)
	 */
	public void setAutoCommit(final boolean autoCommit) throws Exception {
		if (isClosed()) {
			throw new DbConnectNotConectException();
		}
		try {

			getExecutorService().submit(new Callable<Object>() {
				@Override
				public Object call() throws SQLException {
					connection.setAutoCommit(autoCommit);
					return null;
				}
			}).get(getTimeoutSeconds(), TimeUnit.SECONDS);

			// 成功したときだけ通知する。
			for (DataBaseTransactionListener listener : listeners.getListeners(DataBaseTransactionListener.class)) {
				listener.autoCommitChange(this, autoCommit);
			}

		} catch (Exception e1) {
			throw e1;
		}
	}

	/**
	 * @return　DatabaseMetaData
	 * @throws SQLException
	 * @throws DbConnectNotConectException
	 * @see java.sql.Connection#getMetaData()
	 */
	public DatabaseMetaData getMetaData() throws SQLException, DbConnectNotConectException {
		if (isClosed()) {
			throw new DbConnectNotConectException();
		}
		return connection.getMetaData();
	}

	/**
	 * connectionがNULL（未接続）の時にtrueとなるように対応。
	 * 
	 * @return　接続状態
	 * @throws SQLException
	 * @throws DbConnectIllgalStateException
	 * @see java.sql.Connection#isClosed()
	 */
	public boolean isClosed() throws SQLException {
		if (connection == null) {
			return true;
		}
		return connection.isClosed();
	}

	/**
	 * @param sql
	 * @return CallableStatement
	 * @throws SQLException
	 * @throws DbConnectNotConectException
	 * @throws DbConnectIllgalStateException
	 * @see java.sql.Connection#prepareCall(java.lang.String)
	 */
	public CallableStatement prepareCall(String sql) throws SQLException, DbConnectNotConectException,
	DbConnectIllgalStateException {
		if (connection == null) {
			throw new DbConnectNotConectException();
		}
		CallableStatement statement = connection.prepareCall(sql);
		return (CallableStatement) newProxyStatementInstance(CallableStatement.class, statement);
	}

	/**
	 * @param sql
	 * @return PreparedStatement
	 * @throws SQLException
	 * @throws DbConnectNotConectException
	 * @throws DbConnectIllgalStateException
	 * @see java.sql.Connection#prepareStatement(java.lang.String)
	 */
	public PreparedStatement prepareStatement(String sql) throws SQLException, DbConnectNotConectException,
	DbConnectIllgalStateException {
		if (connection == null) {
			throw new DbConnectNotConectException();
		}
		PreparedStatement statement = connection.prepareStatement(sql);
		return (PreparedStatement) newProxyStatementInstance(PreparedStatement.class, statement);
	}

	/**
	 * Statement の close 呼び出しタイミングを補足するために、プロキシ経由のインスタンスに置き換える
	 * 
	 * @param c
	 *            proxyさせたいstatementオブジェクトのクラス
	 * @param statement
	 *            proxyさせたいstatementオブジェクト
	 * @return プロキシが差し込まれたstatementオブジェクト
	 */
	private Object newProxyStatementInstance(Class<?> c, Statement statement) {
		return Proxy.newProxyInstance(statement.getClass().getClassLoader(),
		                              new Class[] { c },
		                              new StatementeHandler(statement));

	}

	private static class StatementeHandler implements InvocationHandler {
		Object target;

		StatementeHandler(Object target) {
			this.target = target;
		}

		@Override
		public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
			Jdbcacsess2.logger.fine("---" + method.getName() + " start");
			Object o = send(target, method, args);
			Jdbcacsess2.logger.fine("---" + method.getName() + " end");
			return o;
		}

		private Object send(Object target, Method m, Object[] args) throws Throwable {
			try {
				return m.invoke(target, args);
			} catch (InvocationTargetException e) {
				if (e.getCause() != null) {
					throw e.getCause();
				}
				throw e;
			}
		}

	}

	/**
	 * １コネクションでは１スレッドを保証する為、シングルスレッド実行のExecutorServiceを用意する。
	 * 
	 * @return executorService
	 */
	public ExecutorService getExecutorService() {
		return executorService;
	}

	/**
	 * @return tIMEOUT_SECONDS
	 */
	public int getTimeoutSeconds() {
		return optionValues.propTimeOutSeconds.getValue();
	}

}
