﻿/*
 * マルチプラットフォーム描画エンジン「Sherry」
 * Copyright(C) 2010-2011 SherryProject. all rights reserved.
 *
 * The MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/*
 *	shTexture.c
 *		Copyright (C) 2010-2011 Cap5ule. all rights reserved.
 *
 *	Date		|Version	|Author		|Summary
 *	2010/11/16	|v0.01		|Cap5ule	|初回版
 *	2010/11/27	|v0.02		|onakah		|PNGデータ構造体作成、PNGファイルの読み込み、
 *				|						|破棄の関数を作成。
 *	2010/12/23	|v0.03		|onakah		|埋め込みテクスチャリソースの読み込みを追加
 */

/*---- インクルードファイル ----*/

#include "sherry.h"			//!<共通ヘッダー
#include "shApplication.h"	//!<必要なSHヘッダー
#include "shTexture.h"		//!<ヘッダー

#include <png.h>			//!<PNGライブラリ

/*---- マクロ定義 ----*/


/*---- スタティックプロトタイプ宣言 ----*/


/*---- グローバル変数 ----*/


/*---- スタティック関数 ----*/


/*---- 関数 ----*/

/*
 *	int shCreateTextureFromPNG(shTexture* stex, const char* filepath)
 *
 *	＜引数＞
 *　　shTexture* stex	: テクスチャ構造体ポインタ
 *　　char* filepath	: ファイルパス
 *	＜戻り値＞
 *　　int				: エラーコードが返ります
 *	＜説明＞
 *　　テクスチャデータを作成する。
 *　　PNGフォーマットからテクスチャデータを読み込む
 */
int shCreateTextureFromPNG(shTexture* stex, const char* filepath)
{
	// PNGフォーマット構造体
	struct {
		char*			data;			// 生データを保存する
		unsigned int	width;			// 横幅
		unsigned int	height;			// 縦幅
		int				depth;
		int				colortype;
		int				interlacetype;
	} pngFormat;

	int				rb;			// メモリカウント用
	unsigned int	i;			// カウンタ
	png_bytepp		image;		// PNGイメージデータ
	FILE*			fp;
	png_structp		sp;			// PNG情報
	png_infop		ip;

	// ファイルを開く
	fp = fopen(filepath, "rb");
	if (!fp)
		return 1;

	sp = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
	ip = png_create_info_struct(sp);
	png_init_io(sp, fp);
	png_read_info(sp, ip);
	png_get_IHDR(sp, ip, (png_uint_32*)&pngFormat.width, (png_uint_32*)&pngFormat.height,
		&pngFormat.depth, &pngFormat.colortype, &pngFormat.interlacetype, NULL, NULL);
	// メモリ領域確保
	rb = png_get_rowbytes(sp, ip);

	pngFormat.data = (char*)malloc(sizeof(unsigned char) * (pngFormat.height) * rb);
	image = (png_bytepp)malloc(sizeof(unsigned char*) * (pngFormat.height));

	for (i = 0; i < pngFormat.height; i++)
		image[i] = (png_byte*)&pngFormat.data[i * rb];

	// png読み込み
	png_read_image(sp, image);
	png_read_end(sp, ip);
	png_destroy_read_struct(&sp, &ip, NULL);

	fclose(fp);
	free(image);

	// テクスチャデータ作成
	if(shCreateTexture(stex,pngFormat.width,pngFormat.height,pngFormat.colortype,pngFormat.data))
		return 1;	/*作成に失敗*/

	// 使用後は解放
	if(pngFormat.data != NULL)
		free(pngFormat.data);
	pngFormat.width = 0;
	pngFormat.height = 0;
	
	return 0;
}

/*
 *	int shCreateTextureFromData(shTexture* stex, int w, int h,
 *								 unsigned char* data)
 *
 *	＜引数＞
 *		shTexture* stex	: テクスチャ構造体ポインタ
 *		int w			: テクスチャサイズ横幅
 *		int h			: テクスチャサイズ縦幅
 *		unsigned char* data		: テクスチャデータ本体
 *	＜戻り値＞
 *		int				: エラーコードが返ります
 *	＜説明＞
 *		テクスチャデータを作成する。
 *		埋め込みデータからテクスチャデータを生成する。
 */
int shCreateTextureFromData(shTexture* stex, int w, int h, unsigned char* data)
{
	if(g_sh_DrawMode == OpenGL) {
		
		// OpenGLテクスチャを生成
		glGenTextures(1, &stex->gltex);		//テクスチャバッファ作成
		if(stex->gltex == 0)
			return 1;	/*生成エラー*/

		glBindTexture(GL_TEXTURE_2D, stex->gltex);

		// テクスチャ属性の設定（デフォルト）
		glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

		// テクスチャデータを入れる
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, w, h, 0, GL_RGBA, GL_UNSIGNED_BYTE, data);

		glBindTexture(GL_TEXTURE_2D, 0);

	} else {
		/*使用しない*/
	}

	return 0;
}

/*
 *	int shCreateTexture(shTexture* stex, int w, int h,
 *					int colortype, char* texbuf)
 *
 *	＜引数＞
 *　　shTexture* stex	: テクスチャ構造体ポインタ
 *　　int w			: テクスチャサイズ横幅
 *　　int h			: テクスチャサイズ縦幅
 *　　int colortype	: テクスチャデータ色配置
 *　　int texbuf		: テクスチャバッファポインタ
 *	＜戻り値＞
 *　　int				: エラーコードが返ります
 *	＜説明＞
 *　　テクスチャデータを作成する。データを描画に使えるようにする。
 */
int shCreateTexture(shTexture* stex, int w, int h,
					int colortype, char* texbuf)
{
	unsigned int type = 0;

	if(g_sh_DrawMode == OpenGL) {
		
		//色配置のタイプを指定
		switch(colortype) {
			//case PNG_COLOR_TYPE_GRAY: type = GL_RGB;
			//	break;
			case PNG_COLOR_TYPE_RGB: type = GL_RGB;
				break;
			case PNG_COLOR_TYPE_RGB_ALPHA: type = GL_RGBA;
				break;
		}
		// OpenGLテクスチャを生成
		glGenTextures(1, &stex->gltex);		//テクスチャバッファ作成
		if(stex->gltex == 0)
			return 1;	/*生成エラー*/

		glBindTexture(GL_TEXTURE_2D, stex->gltex);
		glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

		// テクスチャ属性の設定（デフォルト）
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

		// テクスチャデータを入れる
		glTexImage2D(GL_TEXTURE_2D, 0, type, w, h, 0, type, GL_UNSIGNED_BYTE, texbuf);

		glBindTexture(GL_TEXTURE_2D, 0);

	} else {
		/*使用しない*/
	}

	return 0;
}

/*
 *	int shDeleteTexture(shTexture* stex)
 *
 *	＜引数＞
 *　　shTexture* stex	: テクスチャ構造体ポインタ
 *	＜戻り値＞
 *　　int				: エラーコードが返ります
 *	＜説明＞
 *　　テクスチャデータを削除する。メモリを開放します。
 */
int shDeleteTexture(shTexture* stex)
{
	if(g_sh_DrawMode == OpenGL) {
		//テクスチャ削除
		glDeleteTextures(1, &stex->gltex);
	} else {
		/*使用しない*/
	}
	return 0;
}

/*
 *	void shSetTextureWarp(texparam u, texparam v)
 *
 *	＜引数＞
 *　　texparam u	: U座標テクスチャワープパラメータ
 *　　texparam v	: V座標テクスチャワープパラメータ
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　テクスチャのワープの設定をします。
 *　　[SH_TP_REPEAT][SH_TP_CLAMP]
 */
void shSetTextureWarp(texparam u, texparam v)
{
	if(g_sh_DrawMode == OpenGL) {
		//テクスチャ設定
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, u);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, v);
	} else {
		/*使用しない*/
	}
}

/*
 *	void shSetTextureFilter(texparam min, texparam mag)
 *
 *	＜引数＞
 *　　texparam min	: MINテクスチャフィルタパラメータ
 *　　texparam mag	: MAGテクスチャフィルタパラメータ
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　テクスチャのフィルタの設定をします。
 *　　[SH_TP_NEAREST][SH_TP_LINEAR]
 */
void shSetTextureFilter(texparam min, texparam mag)
{
	if(g_sh_DrawMode == OpenGL) {
		//テクスチャ削除
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, min);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, mag);
	} else {
		/*使用しない*/
	}
}

