﻿/*
 * マルチプラットフォーム描画エンジン「Sherry」
 * Copyright(C) 2010-2011 SherryProject. all rights reserved.
 *
 * The MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/*
 *	shSprite.c
 *		Copyright (C) 2010-2011 Cap5ule. all rights reserved.
 *
 *	Date		|Version	|Author		|Summary
 *	2010/11/16	|v0.01		|Cap5ule	|初回版
 */


/*---- インクルードファイル ----*/

#include "sherry.h"			//!<共通ヘッダー
#include "shApplication.h"	//!<必要なSHヘッダー
#include "shTexture.h"
#include "shMath.h"
#include "shShader.h"
#include "shSprite.h"		//!<ヘッダー

/*---- マクロ定義 ----*/

#define SPRITE_VERTEXNUM	4		//!<スプライト用頂点要素数

/*---- スタティックプロトタイプ宣言 ----*/


/*---- グローバル変数 ----*/

/*
 *	シェーダプログラムソース
 */
// バーテックスシェーダ
// 頂点情報とテクスチャＵＶ座標をもらう
static const char* g_sh_spsource_vs = {
"attribute vec2 vPos;\n"
"attribute vec2 vTex;\n"
"uniform mat4 u_worldViewProjMatrix;\n"
"varying vec2 texCoord;\n"
"void main(void)\n"
"{\n"
"	texCoord = vTex;\n"
"	gl_Position = u_worldViewProjMatrix * vec4(vPos,0,1);\n"
"}\n"
};

// フラグメントシェーダ
static const char* g_sh_spsource_fs = {
"#ifdef GL_ES\n"
"precision mediump float;\n"
"#endif\n"
"uniform sampler2D sampler2d;\n"
"uniform vec4 u_color;\n"
"varying vec2 texCoord;\n"
"void main (void)\n"
"{\n"
"gl_FragColor = texture2D(sampler2d, texCoord) * u_color;\n"
"}\n"
};

// スプライト用頂点要素配列
shVertexElements g_sh_sprite_ve[] = {
	{0,SHVETYPE_FLOAT2,SHVEUSAGE_POSITION,"vPos"},
	{0,SHVETYPE_FLOAT2,SHVEUSAGE_NORMAL,"vTex"},
	SHVE_TMT()	//ターミネータ
};

/*
 *	スプライト頂点情報構造体
 */
static struct {
	union {
		float v[SPRITE_VERTEXNUM * 2];		//!<2 = X,Yの２つ
		shVector2 vec[SPRITE_VERTEXNUM];
	} vartex;			//!<頂点の情報
	union {
		float v[SPRITE_VERTEXNUM * 2];		//!<2 = U,Vの２つ
		shVector2 vec[SPRITE_VERTEXNUM];
	} texcoord;			//!<頂点のＵＶ座標情報
	shVector3 position;	//!<頂点の座標情報
	float angle;		//!<回転アングル（RADIAN）
} g_sh_spriteVertex;		//!<スプライト頂点情報

static shShaderProg	g_sh_spriteShader;		//!<スプライト表示用シェーダ
static shVector2	g_sh_spriteScSize;		//!<スプライト表示用画面サイズ
static shVector4	g_sh_spriteColor;		//!<スプライトのカラー値

/*---- スタティック関数 ----*/



/*---- 関数 ----*/

/*
 *	void shInitSprite()
 *
 *	＜引数＞
 *　　なし
 *	＜戻り値＞
 *　　int			: エラーコードが返ります
 *	＜説明＞
 *　　スプライト機能の初期化
 *　　スプライト機能の初期化を行い使用可能にします。
 *　　スプライトを使う前呼び出す必要があります。
 */
int shInitSprite()
{
	// 初期化
	memset(&g_sh_spriteVertex,0,sizeof(g_sh_spriteVertex));
	g_sh_spriteShader.shaderProg_gl = 0;

	// シェーダ作成
	if(shCreateShader(&g_sh_spriteShader,
		g_sh_spsource_vs,g_sh_spsource_fs,g_sh_sprite_ve))
		return 1;

	// デフォルト設定
	g_sh_spriteScSize.x = 640;	// 画面サイズ
	g_sh_spriteScSize.y = 480;
	shSetSpritePosition(0,0,-0.5f);
	shSetSpriteSize(100,100);
	shSetSpriteTexCoord(0.0f,0.0f,1.0f,1.0f);
	g_sh_spriteColor.v[0] = g_sh_spriteColor.v[1] = g_sh_spriteColor.v[2] = g_sh_spriteColor.v[3] = 1.0f;

	return 0;
}

/*
 *	void shSetSpritePosition(float x, float y, float z)
 *
 *	＜引数＞
 *　　float x		: Ｘ位置
 *　　float y		: Ｙ位置
 *　　float z		: Ｚ位置（Ｚソート）
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　スプライトのポジション（位置）とサイズを設定する。
 *　　無駄な計算を省き高速化するため、ポジションとサイズを同じにしている。
 */
void shSetSpritePosition(float x, float y, float z)
{
	g_sh_spriteVertex.position.x = x;
	g_sh_spriteVertex.position.y = y;
	g_sh_spriteVertex.position.z = z;
}


/*
 *	void shSetSpriteSize(float w, float h)
 *
 *	＜引数＞
 *		float w		: サイズ横幅
 *		float h		: サイズ縦幅
 *	＜戻り値＞
 *		なし
 *	＜説明＞
 *		スプライトのポジション（位置）とサイズを設定する。
 *		無駄な計算を省き高速化するため、ポジションとサイズを同じにしている。
 */
void shSetSpriteSize(float w, float h)
{
	w *= 0.5f;	//半分にする
	h *= 0.5f;

	// 2Dプレート（TRIANGLE STRIP型 配置）
	g_sh_spriteVertex.vartex.vec[0].x = -w;
	g_sh_spriteVertex.vartex.vec[0].y = -h;
	g_sh_spriteVertex.vartex.vec[1].x = w;
	g_sh_spriteVertex.vartex.vec[1].y = -h;
	g_sh_spriteVertex.vartex.vec[2].x = -w;
	g_sh_spriteVertex.vartex.vec[2].y = h;
	g_sh_spriteVertex.vartex.vec[3].x = w;
	g_sh_spriteVertex.vartex.vec[3].y = h;
}

/*
 *	void shSetSpriteTexCoord(float u, float v, float uw, float vh)
 *
 *	＜引数＞
 *		float u		: Ｕ位置
 *		float v		: Ｖ位置
 *		float uw	: Ｕサイズ横幅
 *		float vh	: Ｖサイズ縦幅
 *	＜戻り値＞
 *		なし
 *	＜説明＞
 *		スプライトのテクスチャ座標を設定する。
 */
void shSetSpriteTexCoord(float u, float v, float uw, float vh)
{
	// 2Dプレート（TRIANGLE STRIP型 配置）
	g_sh_spriteVertex.texcoord.vec[0].x = u;
	g_sh_spriteVertex.texcoord.vec[0].y = v;
	g_sh_spriteVertex.texcoord.vec[1].x = u + uw;
	g_sh_spriteVertex.texcoord.vec[1].y = v;
	g_sh_spriteVertex.texcoord.vec[2].x = u;
	g_sh_spriteVertex.texcoord.vec[2].y = v + vh;
	g_sh_spriteVertex.texcoord.vec[3].x = u + uw;
	g_sh_spriteVertex.texcoord.vec[3].y = v + vh;
}

/*
 *	void shSetSpriteRotate(float angdeg)
 *
 *	＜引数＞
 *		float angdeg	: 回転角（DEGREE）
 *	＜戻り値＞
 *		なし
 *	＜説明＞
 *		スプライトのテクスチャ座標を設定する。
 */
void shSetSpriteRotate(float angdeg)
{
	g_sh_spriteVertex.angle = angdeg;
}


/*
 *	void shSetSpriteScSize(float w, float h)
 *
 *	＜引数＞
 *		float w		: サイズ横幅
 *		float h		: サイズ縦幅
 *	＜戻り値＞
 *		なし
 *	＜説明＞
 *		スプライトのスクリーンサイズを設定する。
 *		個別に設定しておくことで、実画面サイズに依存せず表示可能になります。
 */
void shSetSpriteScSize(float w, float h)
{
	g_sh_spriteScSize.x = w;
	g_sh_spriteScSize.y = h;
}

/*
 *	void shSetSpriteColor(float r, float g, float b, float a)
 *
 *	＜引数＞
 *		float r		: 赤（0.0～1.0）
 *		float g		: 緑（0.0～1.0）
 *		float b		: 青（0.0～1.0）
 *		float a		: アルファ値（0.0～1.0）
 *	＜戻り値＞
 *		なし
 *	＜説明＞
 *		スプライトの色を設定する。
 */
void shSetSpriteColor(float r, float g, float b, float a)
{
	g_sh_spriteColor.v[0] = r;
	g_sh_spriteColor.v[1] = g;
	g_sh_spriteColor.v[2] = b;
	g_sh_spriteColor.v[3] = a;
}

/*
 *	void shSetSpriteTexture(shTexture* texture)
 *
 *	＜引数＞
 *　　shTexture* texture	: テクスチャ構造体
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　スプライトのテクスチャを設定する。
 */
void shSetSpriteTexture(shTexture* texture)
{
	if(g_sh_DrawMode == OpenGL) {
		//テクスチャをバインドする
		glBindTexture(GL_TEXTURE_2D, texture->gltex);
		glActiveTexture( GL_TEXTURE0 ); 
	} else {
		/*使用しない*/
	}
}

/*
 *	void shSpriteDraw()
 *
 *	＜引数＞
 *　　なし
 *	＜戻り値＞
 *　　なし
 *	＜説明＞
 *　　設定に基づいてスプライトを描画します。
 */
void shSpriteDraw()
{
	// ローカル変数
	shMatrix drawans, drawcalc;	// 変換用の行列

	glDisable(GL_DEPTH_TEST);

	// 行列変換
	shMatrixIdentity(&drawans);	// 単位行列化
	shMatrixOrthoRH(&drawans,0.0f,-g_sh_spriteScSize.x,0.0f,-g_sh_spriteScSize.y,0.0f,1.0f);

	//計算
	shMatrixTranslation(&drawcalc,g_sh_spriteVertex.position.x,g_sh_spriteVertex.position.y,g_sh_spriteVertex.position.z);
	shMatrixMultiply(&drawans,&drawcalc,&drawans);
	shMatrixRotationZ(&drawcalc,g_sh_spriteVertex.angle);
	shMatrixMultiply(&drawans,&drawcalc,&drawans);
	
	if(g_sh_DrawMode == OpenGL) {
		// シェーダプログラムの適用
		glUseProgram(g_sh_spriteShader.shaderProg_gl);

		// 行列を入れる
		glUniformMatrix4fv(glGetUniformLocation(g_sh_spriteShader.shaderProg_gl,
			"u_worldViewProjMatrix"), 1, GL_FALSE, (float*)drawans.m);
		glUniform4fv(glGetUniformLocation(g_sh_spriteShader.shaderProg_gl,
			"u_color"), 1, (float*)g_sh_spriteColor.v);

		// 頂点属性に情報を入れる
		glVertexAttribPointer(0, 2, GL_FLOAT, GL_FALSE, 0, (float*)g_sh_spriteVertex.vartex.v);
		glEnableVertexAttribArray(1);
		glVertexAttribPointer(1, 2, GL_FLOAT, GL_FALSE, 0, (float*)g_sh_spriteVertex.texcoord.v);
		glEnableVertexAttribArray(0);
		
		// 描画コマンド発行
		glDrawArrays(GL_TRIANGLE_STRIP, 0, SPRITE_VERTEXNUM);

		glDisableVertexAttribArray(0);
		glDisableVertexAttribArray(1);

	} else {
		/*使用しない*/
	}

	glEnable(GL_DEPTH_TEST);
}

