﻿///////////////////////////////////////////////////////////////////////////////
// This file is part of WebBrowserController.
///////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2009-2011 bis5 <bis5@users.sourceforge.jp>
// All rights reserved.
///////////////////////////////////////////////////////////////////////////////
// WebBrowserController is one of modules for SharpWebBrowser.
///////////////////////////////////////////////////////////////////////////////
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the Microsoft Public Licene (Ms-PL) as published by 
// Microsoft Corp.
//
// You should have received a copy of the Microsoft Public License along 
// with this program. 
// If not, see <http://www.microsoft.com/opensource/licenses.mspx>
//
///////////////////////////////////////////////////////////////////////////////
//
// File information
// Name: WebBrowserController.cs
// Author: bis5
// Module: WebBrowserController
//
///////////////////////////////////////////////////////////////////////////////

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace Bis5Products.SharpWebBrowser {
    public class WebBrowserController {

        public WebBrowserController() { throw new ArgumentNullException(); }

        public WebBrowserController(ILangMgr language) { lang = language; }

        private Dictionary<int, WebBrowser_NewWindowHacked> wblist = new Dictionary<Int32, WebBrowser_NewWindowHacked>();
        private int wbcount = 0;

        private ILangMgr lang;
        private string strNoTitlePage;

        private Form mf = null;
        private const string strName = "SharpWebBrowser";
        private ToolStripSpringTextBox tbURL = null;
        private TabControl tabControl1 = null;
        private ToolStripStatusLabel statusLabel = null;
        private ToolStripProgressBar toolStripProgressBar1 = null;

        public WebBrowser_NewWindowHacked CreateControl() {
            var wb = new WebBrowser_NewWindowHacked();
            wb.control_id = wbcount;
            wblist.Add(wbcount, wb);
            wbcount++;
            wb.Dock = DockStyle.Fill;
            wb.DocumentTitleChanged += new EventHandler(webBrowser_DocumentTitleChanged);

            wb.StatusTextChanged += new EventHandler(webBrowser_StatusTextChanged);
            wb.ProgressChanged += new WebBrowserProgressChangedEventHandler(webBrowser_ProgressChanged);
            wb.Navigating += new WebBrowserNavigatingEventHandler(webBrowser_Navigating);

            wb.EncryptionLevelChanged += new EventHandler(wb_EncryptionLevelChanged);
            return wb;
        }

        private void wb_EncryptionLevelChanged(object sender, EventArgs e) {
            if (((Control)sender).Parent == tabControl1.SelectedTab) {
                if (((WebBrowser)sender).EncryptionLevel == WebBrowserEncryptionLevel.Insecure ||
                    ((WebBrowser)sender).EncryptionLevel == WebBrowserEncryptionLevel.Unknown)
                    tbURL.BackColor = System.Drawing.Color.FromKnownColor(System.Drawing.KnownColor.Window);
                else
                    tbURL.BackColor = System.Drawing.Color.Khaki;
            }
        }

        public void RemoveControl(WebBrowser_NewWindowHacked wb) {
            wblist.Remove(wb.control_id);

            wb.Navigated -= new WebBrowserNavigatedEventHandler(webBrowser_Navigated);
            wb.Navigated += new WebBrowserNavigatedEventHandler(wb_disposing);
            wb.Navigate("about:blank");
        }

        private void wb_disposing(object sender, WebBrowserNavigatedEventArgs e) {
            ((WebBrowser)sender).Dispose();
        }

        #region WebBrowser_NewWindowHacked Events

        public void RegisterMainForm(Form main, ToolStripSpringTextBox tb, TabControl tc, ToolStripStatusLabel st,
                                        ToolStripProgressBar pg) {
            mf = main;
            tbURL = tb;
            tabControl1 = tc;
            statusLabel = st;
            toolStripProgressBar1 = pg;

            strNoTitlePage = lang.Get("strNoTitlePage");
        }

        /// <summary>
        /// SharpWebBrowser終了時のスクリプトエラー発生を抑制する
        /// </summary>
        public void Dispose() {
            for (int i = wbcount - 1; i >= 0; i--) {
                try {
                    var wb = wblist[i];
                    wb.Navigated -= new WebBrowserNavigatedEventHandler(webBrowser_Navigated);
                    wb.Navigated += new WebBrowserNavigatedEventHandler(wb_disposing);
                    wb.Navigate("about:blank");
                } catch (KeyNotFoundException) {
                    Console.WriteLine("KeyNotFoundException at WebBrowserController.Dispose()");
                    continue;
                } catch (System.Runtime.InteropServices.COMException) {
                    Console.WriteLine("COMException at WebBrowserController.Dispose()");
                    continue;
                }
            }
            //ナビゲート完了を待機するために、スレッドを5000ミリ秒待機させます
            System.Threading.Thread.SpinWait(5000);
        }

        private void webBrowser_DocumentTitleChanged(object sender, System.EventArgs e) {
            var wb = (WebBrowser)sender;
            TabPage tp = (TabPage)wb.Parent;

            if (String.IsNullOrEmpty(wb.DocumentTitle)) {
                tp.Text = strNoTitlePage;
            } else {
                if (wb.DocumentTitle.Length < 30)
                    tp.Text = wb.DocumentTitle;
                else {
                    var text = wb.DocumentTitle;
                    tp.Text = text.Remove(30) + "...";
                }
            }

            if (tabControl1.SelectedTab.Controls.Contains(wb)) {
                if (String.IsNullOrEmpty(wb.DocumentTitle)) {
                    mf.Text = strName;
                } else {
                    mf.Text = wb.DocumentTitle + " - " + strName;
                }
                ChangeUrlTextBox(sender, e);
            }
        }
        private void webBrowser_Navigated(object sender, WebBrowserNavigatedEventArgs e) {
            throw new NotImplementedException();
        }
        void webBrowser_StatusTextChanged(object sender, EventArgs e) {
            if (tabControl1.SelectedTab.Controls.Contains((Control)sender)) {
                foreach (WebBrowser_NewWindowHacked c in tabControl1.SelectedTab.Controls) {
                    statusLabel.Text = c.StatusText;
                }
            }
        }
        private void webBrowser_ProgressChanged(object sender, WebBrowserProgressChangedEventArgs e) {
            WebBrowser_NewWindowHacked tempWB = (WebBrowser_NewWindowHacked)sender;
            if (tabControl1.SelectedTab == tempWB.Parent) {
                toolStripProgressBar1.Maximum = (int)e.MaximumProgress;

                if (e.CurrentProgress == -1 | e.CurrentProgress == 0 | e.CurrentProgress == e.MaximumProgress) {
                    toolStripProgressBar1.Value = toolStripProgressBar1.Maximum;
                    toolStripProgressBar1.Visible = false;
                } else {
                    try { toolStripProgressBar1.Value = (int)e.CurrentProgress; } catch { toolStripProgressBar1.Value = (int)e.MaximumProgress; }
                    toolStripProgressBar1.Visible = true;
                }
            }
            ChangeUrlTextBox(sender, e);
        }

        private void webBrowser_Navigating(object sender, WebBrowserNavigatingEventArgs e) {
            ChangeUrlTextBox(sender, e);
        }

        private void ChangeUrlTextBox(object sender, EventArgs e, bool force = false) {
            if (tabControl1.SelectedTab.Controls.Contains((Control)sender)) {
                if (!tbURL.Focused || force) {
                    string url;
                    if (e is WebBrowserNavigatingEventArgs) {
                        url = ((WebBrowserNavigatingEventArgs)e).Url.ToString();
                    } else {
                        url = ((WebBrowser)sender).Url.ToString();
                    }
                    tbURL.Text = url;
                }
            }
        }

        /* REMIND:webBrowser_NewWindow2だけはMainFormで処理すること */

        #endregion

        public string[] GetVersion() {
            var asm = System.Reflection.Assembly.GetExecutingAssembly();
            return new string[]{
                asm.ManifestModule.Name.Replace(".dll","\0"),
                asm.GetName().Version.ToString(),
                "bis5",
                asm.ManifestModule.Name
            };
        }
    }
}