﻿///////////////////////////////////////////////////////////////////////////////
// This file is part of ConfigurationManager.
///////////////////////////////////////////////////////////////////////////////
// Copyright (C) 2009-2011 bis5 <bis5@users.sourceforge.jp>
// All rights reserved.
///////////////////////////////////////////////////////////////////////////////
// ConfigurationManager is one of modules for SharpWebBrowser.
///////////////////////////////////////////////////////////////////////////////
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the Microsoft Public Licence (Ms-PL) as published by 
// Microsoft Corp.
//
// You should have received a copy of the Microsoft Public License along 
// with this program. 
// If not, see <http://www.microsoft.com/opensource/licenses.mspx>
//
///////////////////////////////////////////////////////////////////////////////
//
// File information
// Name: ConfigurationManager.cs
// Author: bis5
// Module: ConfigurationManager
//
///////////////////////////////////////////////////////////////////////////////

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Xml;
using System.Xml.Serialization;
using System.Runtime.InteropServices;

namespace Bis5Products.SharpWebBrowser {

    public class ConfigurationManager : ConfigurationManagerBase {

        private bool _readonly;

        public ConfigurationManager() { }

        public ConfigurationManager(string path) {
            base.apppath = path;
            base.Load();
        }

        public void SetLangMgr(ILangMgr lang) {
            base.frmConfig.SetLanguageManager(lang);
        }

        /// <param name="path">実行パス</param>
        /// <param name="filename">設定ファイル名</param>
        /// <param name="element">Element名</param>
        /// <param name="isdefault">デフォルトであることを検出するのに用いる設定項目名と設定値</param>
        public ConfigurationManager(string path, string filename, string element, string[] isdefault) {
            base.apppath = path;
            base.fname = filename;
            base.elename = element;
            base.defaultset = isdefault;
            base.Load();
        }

        public ConfigurationManager(string path, bool isReadOnly) {
            this._readonly = isReadOnly;
            base.apppath = path;
            base.Load();
        }

        public ConfigurationManager(string path, string filename, string element, string[] isdefault, bool isReadOnly) {
            _readonly = isReadOnly;
            base.apppath = path;
            base.fname = filename;
            base.elename = element;
            base.defaultset = isdefault;
            base.Load();
        }

        public bool ReadOnly {
            get { return _readonly; }
        }

        public bool Add(string key, object value) {
            if (ReadOnly) {
                throw new System.Data.ReadOnlyException();
            }
            return base.Add(key, value);
        }

        public bool Set(string key, object value) {
            if (ReadOnly) {
                throw new System.Data.ReadOnlyException();
            }
            return base.Set(key, value);
        }

        public bool Remove(string key) {
            if (ReadOnly) {
                throw new System.Data.ReadOnlyException();
            }
            return base.Remove(key);
        }

        public ConfigurationManager Current2ReadOnly() {
            var cm = new ConfigurationManager(base.apppath, base.fname, base.elename, base.defaultset, false);
            cm.Clear();
            foreach (string key in base.conf.Keys) {
                cm.Add(key, base.conf[key]);
            }
            return cm;
        }

        public void Clear() {
            base.conf.Clear();
        }

       private const uint BCM_SETSHIELD= 0x160C;
       [DllImport("user32.dll")]
       internal static extern int SendMessage(IntPtr hWnd, uint msg, int wParam, bool lParam);

        public void SetButtonUac() {
            ((System.Windows.Forms.Button)frmConfig.AcceptButton).FlatStyle = System.Windows.Forms.FlatStyle.System;
            SendMessage(((System.Windows.Forms.Button)frmConfig.AcceptButton).Handle, BCM_SETSHIELD, 0, true);
        }

        public string[] GetVersion() {
            var asm = System.Reflection.Assembly.GetExecutingAssembly();
            return new string[]{
                asm.ManifestModule.Name.Replace(".dll","\0"),
                asm.GetName().Version.ToString(),
                "bis5",
                asm.ManifestModule.Name
            };
        }

    }

    public abstract class ConfigurationManagerBase {

        protected string apppath;
        protected string fname = "\\SWBCore2.config";
        protected string elename = "SWBCoreConfiguration";
        protected string[] defaultset = { "IsFirstRun", "True" };
        private int _count = 0;
        internal SerializableDictionary<string, object> conf = new SerializableDictionary<string, object>();

        internal static List<ISWBPropertySheet> propsheet = new List<ISWBPropertySheet>();
        protected Configure frmConfig = new Configure();

        public ConfigurationManagerBase() { }

        public ConfigurationManagerBase(string path) {
            apppath = path;
            Load();
        }

        /// <param name="path">実行パス</param>
        /// <param name="filename">設定ファイル名</param>
        /// <param name="element">Element名</param>
        /// <param name="isdefault">デフォルトであることを検出するのに用いる設定項目名と設定値</param>
        public ConfigurationManagerBase(string path, string filename, string element, string[] isdefault) {
            apppath = path;
            fname = filename;
            elename = element;
            defaultset = isdefault;
            Load();
        }

        internal void WriteConsoleError(Exception ex) {
            Console.WriteLine(ex.Message + "\r\n" + ex.StackTrace);
        }

        //-Configuration Controler-//
        public bool Load() {
            FileStream fs = null;
            if (File.Exists(apppath + fname)) {
                try {
                    fs = new FileStream(apppath + fname, FileMode.Open);
                } catch (IOException ex) {
                    WriteConsoleError(ex);
                    return false;
                }
                var reader = XmlReader.Create(fs);
                for (int i = 0; i < 5; i++) {
                    reader.Read();//先頭から最初の設定項目まで読み進める
                }
                conf.Clear();
                while (reader.NodeType == XmlNodeType.Element) {
                    conf.Add(reader.Name, reader.ReadElementString());
                    if (conf.Keys.Last<string>() == "WindowState") {
                        switch (conf.Values.Last<object>().ToString()) {
                            case "Normal":
                                this.Set("WindowState", System.Windows.Forms.FormWindowState.Normal);
                                break;
                            case "Maximized":
                                this.Set("WindowState", System.Windows.Forms.FormWindowState.Maximized);
                                break;
                            default:
                                this.Set("WindowState", System.Windows.Forms.FormWindowState.Minimized); break;
                        }
                    } else if (conf.Keys.Last<string>() == "NewTabOpenMode") {
                        switch (conf.Values.Last<object>().ToString()) {
                            case "hp":
                                this.Set("NewTabOpenMode", TabOpenMode.hp);
                                break;
                            case "blank":
                                this.Set("NewTabOpenMode", TabOpenMode.blank);
                                break;
                            default:
                                this.Set("NewTabOpenMode", TabOpenMode.any); break;
                        }
                    }
                    _count++;
                    reader.Read();
                }
                reader.Close();
                fs.Close();
            } else {
                if (defaultset != null) {
                    conf.Add(defaultset[0], defaultset[1]);
                }
                Save();
            }
            return true;
        }

        public bool Save() {
            FileStream fs = null;
            try {
                if (File.Exists(apppath + fname)) {
                    fs = new FileStream(apppath + fname, FileMode.Truncate);
                } else {
                    fs = new FileStream(apppath + fname, FileMode.Create);
                }
            } catch (IOException ex) {
                WriteConsoleError(ex);
                return false;
            }
            var setting = new XmlWriterSettings();
            setting.Indent = true;
            try {
                var writer = XmlWriter.Create(fs, setting);

                writer.WriteStartDocument();
                writer.WriteStartElement(elename);
                foreach (var name in conf.Keys) {
                    writer.WriteElementString(name, conf[name].ToString());
                }
                writer.WriteEndElement();
                writer.Flush();
                writer.Close();
                fs.Close();
            } catch (Exception ex) { WriteConsoleError(ex); return false; }
            return true;
        }

        public bool IsContains(string key) {
            return conf.ContainsKey(key);
        }

        public object Get(string key) {
            try { return conf[key]; } catch (ArgumentOutOfRangeException ex) { WriteConsoleError(ex); return null; }
        }
        public object Get(Int32 id) {
            try { return conf.Values.ToList<object>()[id]; } catch (ArgumentOutOfRangeException ex) { WriteConsoleError(ex); return null; }
        }

        public object GetObj(string key, Type type) {
            if (key == "WindowState") {
                switch (conf[key].ToString()) {
                    case "Normal":
                        return System.Windows.Forms.FormWindowState.Normal;
                    case "Maximized":
                        return System.Windows.Forms.FormWindowState.Maximized;
                    default:
                        return System.Windows.Forms.FormWindowState.Minimized;
                }
            } else if (key == "NewTabOpenMode") {
                switch (conf[key].ToString()) {
                    case "hp":
                        return TabOpenMode.hp;
                    case "blank":
                        return TabOpenMode.blank;
                    default:
                        return TabOpenMode.any;
                }
            }
            try { return Convert.ChangeType(conf[key], type); } catch (Exception ex) { WriteConsoleError(ex); return null; }
        }
        public object GetObj(int id, Type type) {
            try { return Convert.ChangeType(conf.Values.ToList<object>()[id], type); } catch (Exception ex) { WriteConsoleError(ex); return null; }
        }

        public string GetStr(string key) {
            object ret = null;
            try { ret = conf[key]; } catch (ArgumentOutOfRangeException ex) { WriteConsoleError(ex); return String.Empty; }

            try { return (string)ret; } catch (InvalidCastException ex) { WriteConsoleError(ex); return String.Empty; }
        }
        public string GetStr(Int32 id) {
            object ret = null;
            try { ret = conf.Values.ToList<object>()[id]; } catch (ArgumentOutOfRangeException ex) { WriteConsoleError(ex); return String.Empty; }

            try { return (string)ret; } catch (InvalidCastException ex) { WriteConsoleError(ex); return String.Empty; }
        }

        public Int32 GetInt32(string key) {
            object ret = null;
            try { ret = conf[key]; } catch (ArgumentOutOfRangeException ex) { WriteConsoleError(ex); return -1; }

            try { return Convert.ToInt32(ret); } catch (InvalidCastException ex) { WriteConsoleError(ex); return -1; }
        }
        public Int32 GetInt32(Int32 id) {
            object ret = null;
            try { ret = conf.Values.ToList<object>()[id]; } catch (ArgumentOutOfRangeException ex) { WriteConsoleError(ex); return -1; }

            try { return Convert.ToInt32(ret); } catch (InvalidCastException ex) { WriteConsoleError(ex); return -1; }
        }

        public bool GetBool(string key) {
            object ret = null;
            try { ret = conf[key]; } catch (ArgumentOutOfRangeException ex) { WriteConsoleError(ex); return false; }
            switch (ret.ToString()) {
                case "1":
                case "True":
                case "true":
                    conf[key] = true;
                    return true;
                    break;
                case "0":
                case "false":
                case "False":
                    conf[key] = false;
                    return false;
                    break;
                default:
                    throw new InvalidCastException();
            }
            WriteConsoleError(new ArgumentException());
            return false;
        }
        public bool GetBool(int id) {
            object ret = null;
            try { ret = conf.Values.ToList<object>()[id]; } catch (ArgumentOutOfRangeException ex) { WriteConsoleError(ex); return false; }
            if (ret.ToString() == "1" || ret.ToString() == "0" || ret.ToString() == "True" || ret.ToString() == "False" || ret.ToString() == "true" || ret.ToString() == "false") {
                return (bool)ret;
            }
            WriteConsoleError(new ArgumentException());
            return false;
        }

        public bool Add(string key, object value) {
            try { conf.Add(key, value); _count++; } catch (ArgumentException ex) { WriteConsoleError(ex); return false; }
            return true;
        }

        public bool Set(string key, object value) {
            try {
                if (conf.ContainsKey(key)) {
                    conf[key] = value;
                } else {
                    Add(key, value);
                }
            } catch (Exception ex) { WriteConsoleError(ex); return false; }
            return true;
        }

        public bool Remove(string key) {
            try {
                conf.Remove(key);
                return true;
            } catch (NullReferenceException ex) { WriteConsoleError(ex); return false; }
        }


        public int Count {
            get {
                return _count;
            }
        }
        //-PropertySheet-//
        public System.Windows.Forms.DialogResult ShowConfigDialog() {
            if (frmConfig.listBox1.Items.Count == 0) {
                for (int i = 0; i < propsheet.Count; i++) {
                    frmConfig.listBox1.Items.Add(propsheet[i].GetPropertySheetName());
                }
            }
            return frmConfig.ShowDialog();
        }

        public void AddPropertySheet(ISWBPropertySheet prop) {
            propsheet.Add(prop);
            if (frmConfig.listBox1.Items.Count != 0) {//never
                frmConfig.listBox1.Items.Add(prop.GetPropertySheetName());
            }
        }

        public void ResetConfigDialog() {
            frmConfig.Dispose();
            frmConfig = new Configure();
            this.ShowConfigDialog();
        }

        public void ResetPropertyList() {
            propsheet = new List<ISWBPropertySheet>();
        }

        public string SolveIndex(object value) {
            for (int i = 0; i < conf.Count; i++) {
                if (conf.Values.ToList<object>()[i].Equals(value)) {
                    return conf.Keys.ToList<string>()[i];
                }
            }
            return null;
        }

        public bool ChangeIndex(string oldindex, string newindex) {
            object obj = null;
            try { obj = conf[oldindex]; } catch (ArgumentOutOfRangeException ex) { WriteConsoleError(ex); return false; }
            conf.Add(newindex, obj);
            conf.Remove(oldindex);
            return true;
        }
    }

    public class ConfigurationManagerWithXmlSerializer : ConfigurationManager {
        private string apppath;
        private string fname;
        private string elename;
        private XmlSerializer serializer;
        private SerializableDictionary<string, object> conf = new SerializableDictionary<string, object>();

        public ConfigurationManagerWithXmlSerializer(string path, string filename, string element) {
            apppath = path;
            fname = filename;
            elename = element;
        }

        public void Load() {
            FileStream fs = null;
            if (File.Exists(apppath + fname)) {
                fs = new FileStream(apppath + fname, FileMode.Open);
            } else {
                fs = new FileStream(apppath + fname, FileMode.Create);
            }
            var reader = XmlReader.Create(fs);

            reader.ReadStartElement(elename);
            conf.ReadXml(reader);
            reader.Close();
            fs.Close();
        }
        public void Save() {
            FileStream fs = null;
            if (File.Exists(apppath + fname)) {
                fs = new FileStream(apppath + fname, FileMode.Truncate);
            } else {
                fs = new FileStream(apppath + fname, FileMode.Create);
            }
            var writer = XmlWriter.Create(fs);

            writer.WriteStartDocument();
            writer.WriteStartElement(elename);
            conf.WriteXml(writer);
            writer.Close();
            fs.Close();
        }
    }

}