require 'sinatra/base'
require 'haml'
require 'grit'
require 'tmpdir'
require 'fileutils'
require 'pathname'
require 'logger'

class Grit::Tree
  def project?
    blobs.map(&:name).include? 'MainClass.txt'
  end
end

class Grit::Blob
  def project?
    false
  end
end

class Logger
  attr_reader :logdev
end

$stdout.reopen open('stdout.log', 'a')
$stdout.sync = true
$stderr.reopen open('stderr.log', 'a')
$stderr.sync = true

class App < Sinatra::Base
  if development?
    require 'sinatra/reloader'
    register Sinatra::Reloader
  end

  enable :logging

  def initialize
    super
    @repo = Grit::Repo.new(ENV['GIT_DIR'] || File.expand_path('../../..', __FILE__))
    @logger = Logger.new "autobuild.log"
    @logdev = @logger.logdev.dev
  end

  def fetch
    @repo.remote_fetch 'origin'
  end

  get '/' do
    fetch
    @remotes = @repo.remotes
    haml :index
  end

  get %r!^/remotes/(.+)! do
    remote_name = params[:captures].first
    @remote = @repo.remotes.find { |r| r.name == remote_name }
    return 404 if @remote.nil?
    @objects = @remote.commit.tree.contents
    haml :remote
  end

  get '/build/:sha/:project' do
    @commit = @repo.commit params[:sha]
    return 404 if @commit.nil?
    @project = params[:project]
    cache_dir = cache_dir @commit, @project
    name_path = cache_dir.join 'MainClass.txt'

    if not name_path.exist?
      Dir.mktmpdir do |tmpdir|
        Dir.chdir tmpdir do
          fname = "archive.tar"
          @repo.archive_to_file @commit, nil, fname, nil, 'cat'
          system 'tar', 'xf', fname
          Dir.chdir @project do
            build_and_copy @commit, @project, cache_dir, name_path
          end
        end
      end
    end
    @files = Dir.glob("#{cache_dir.to_s}/*.apk").map { |file| File.basename file }.sort
    haml :build
  end

  get '/apk/:sha/:project/:file' do
    commit = @repo.commit params[:sha]
    return 404 if commit.nil?
    project = params[:project]
    cache_dir = cache_dir commit, project
    name_path = cache_dir.join 'MainClass.txt'
    file_path = cache_dir.join params[:file]
    return 404 if not file_path.exist?

    fname = name_path.read.chomp + ".apk"
    send_file file_path.to_s, :filename => fname
  end

  def build_and_copy(commit, project, cache_dir, name_path)
    @logger.info "Building #{commit} #{project}"
    system 'rake clean', [:out, :err] => @logdev
    system 'rake', [:out, :err] => @logdev
    FileUtils.cp 'MainClass.txt', name_path.to_s
    Dir.glob('bin/*-debug.apk') do |file|
      FileUtils.cp file, cache_dir.to_s
    end
  end

  def cache_dir(commit, project)
    dir = Pathname.new 'cache'
    if not dir.exist?
      dir.mkpath
    end
    dir = dir.join "#{commit}-#{project}"
    if not dir.exist?
      dir.mkpath
    end
    dir.realpath
  end
end
