require 'test_helper'

class LecturesControllerTest < ActionController::TestCase
  include Devise::TestHelpers

  setup do
    @apk = FactoryGirl.create :apk
    @lecture = @apk.task.lecture
    @teacher = @lecture.teacher
  end

  test "should get index with JSON" do
    get :index, :format => :json
    assert_response :success
    assert_not_nil assigns(:lectures)
  end

  test "should have icon url in JSON" do
    get :index, :format => :json
    assert_response :success
    j = JSON.parse @response.body
    assert j.is_a?(Array)
    assert j.all? { |lec| lec.is_a?(Hash) && lec.has_key?('teacher') }
    teachers = j.map { |lec| lec['teacher'] }
    assert teachers.all? { |t| t.is_a?(Hash) && t.has_key?('icon_path') && !t['icon_path'].nil? }
  end

  test "non-teacher should not get new" do
    user = FactoryGirl.create :user
    sign_in user

    get :new
    assert_redirected_to root_path
  end

  test "teacher should get new" do
    sign_in @teacher

    get :new
    assert_response :success
    assert_not_nil assigns(:lecture)
  end

  test "others should not create lecture" do
    post :create, :lecture => { :name => @lecture.name }
    assert_redirected_to new_teacher_session_path
  end

  test "non-teacher should not create lecture" do
    user = FactoryGirl.create :user
    sign_in user

    post :create, :lecture => { :name => @lecture.name }
    assert_redirected_to root_path
  end

  test "teacher should create lecture" do
    sign_in @teacher

    assert_difference('Lecture.count') do
      post :create, :lecture => { :name => @lecture.name }
    end
    lecture = assigns :lecture
    assert_not_nil lecture
    assert_equal @teacher.id, lecture.teacher.id
  end

  test "teacher should not create lecture with empty lecture name" do
    sign_in @teacher

    assert_difference('Lecture.count', 0) do
      post :create, :lecture => { :name => '' }
    end
  end

  test "should show lecture" do
    get :show, id: @lecture
    assert_response :success
  end

  test "should NOT update lecture" do
    assert_no_routes do
      put :update, id: @lecture, lecture: { name: @lecture.name }
    end
  end

  test "should NOT destroy lecture" do
    assert_no_routes do
      delete :destroy, id: @lecture
    end
  end

  test "should NOT return tasks without ids" do
    get :index, :format => :json
    assert_response :success
    lecs = JSON.parse @response.body
    lecs.each do |lec|
      assert !lec.has_key?('tasks')
    end
  end

  test "should return tasks with ids" do
    get :index, :ids => [@lecture], :format => :json
    assert_response :success
    lecs = JSON.parse @response.body
    assert_equal lecs.size, 1
    assert lecs[0].has_key?('tasks')

    tasks = lecs[0]['tasks']
    assert_equal 1, tasks.size
    assert tasks[0].has_key?('apk')

    apk = tasks[0]['apk']
    assert apk.has_key?('id')
    assert_equal @apk.id, apk['id']
  end

  test "icon_path should not be nil if teacher has icon" do
    get :index, :ids => [@lecture.id], :format => :json
    assert_response :success
    lecs = JSON.parse @response.body
    t = lecs[0]['teacher']
    assert t.has_key?('icon_path')
    assert_not_nil t['icon_path']
  end

  test "icon_path should be nil if teacher has no icon" do
    no_icon = FactoryGirl.create :teacher, :icon => nil
    lecture = FactoryGirl.create :lecture, :teacher => no_icon
    get :index, :ids => [lecture.id], :format => :json
    assert_response :success
    lecs = JSON.parse @response.body
    t = lecs[0]['teacher']
    assert t.has_key?('icon_path')
    assert_nil t['icon_path']
  end
end
