package jp.ac.titech.sharp4k.cuten;

import java.util.ArrayList;
import java.util.List;

import jp.ac.titech.sharp4k.cuten.util.XmlUtility;

import org.apache.http.StatusLine;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import android.app.AlertDialog;
import android.app.ProgressDialog;
import android.content.DialogInterface;
import android.content.Intent;
import android.database.sqlite.SQLiteDatabase;
import android.os.Bundle;
import android.text.InputType;
import android.util.Log;
import android.view.View;
import android.view.View.OnClickListener;
import android.widget.AdapterView;
import android.widget.AdapterView.OnItemLongClickListener;
import android.widget.EditText;
import android.widget.ExpandableListView;
import android.widget.ExpandableListView.OnChildClickListener;
import android.widget.ExpandableListView.OnGroupClickListener;
import android.widget.ImageButton;
import android.widget.TextView;
import android.widget.Toast;

import com.google.inject.Inject;

public class LectureFolderActivity extends BaseMenuActivity {
	private static final String TAG = "LectureFolder";

	// 画面遷移の際に引き渡すインテントに付加するキー。
	public static final String LECTURE_KEY = "LECTURE";

	// 新規作成ボタン
	ImageButton createBtn, clearFlagBtn, updateBtn, moveBtn;
	// 講義リスト(のびーる)
	ExpandableListView folderList;

	// 削除することのできないデフォルトのフォルダーのインスタンス。デフォルトであることの確認や、ダウンロードの際の宛先として使える？
	private LectureFolder defaultFolder;

	// プログレスダイアログ
	ProgressDialog pDialog;

	private static enum Mode {
		NORMAL, MOVE_LECTURE, SWAP_FOLDER, MERGE_FOLDER
	}

	private Mode currentMode;

	// ExpandableListViewの内部を構成するアダプター。
	PandaAdapter<Lecture, LectureFolder> listAdapter;
	// 画面上部につけたテキストビュー。現在は主にデバッグよう。
	TextView t;
	// OnGroupClickListenerの拡張クラスのインスタンス。スワップができるように拡張されている。
	GroupClick gc;

	SQLiteDatabase database;

	@Inject
	private HttpAPIClient apiClient;

	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.main);

		if (!Token.isExist(this)) {
			startTokenAuthenticationActivity();
		}
		listAdapter = new PandaAdapter<Lecture, LectureFolder>(this,
				new ArrayList<LectureFolder>());
		SQLHelper sqlHelper = new SQLHelper(this);
		database = sqlHelper.getReadableDatabase();
		setLayout();
		changeCurrentMode(Mode.NORMAL);

		setActionListener();
	}

	/**
	 * 各ビューにリスナーをセットするメソッド
	 */
	private void setActionListener() {
		folderList.setOnGroupClickListener(gc);

		folderList.setOnChildClickListener(new OnChildClickListener() {
			@Override
			public boolean onChildClick(ExpandableListView parent, View v,
					int groupPosition, int childPosition, long id) {
				if (currentMode != Mode.NORMAL) {
					gc.clearModeFlag();
				} else {
					Lecture lecture = listAdapter.getChild(groupPosition,
							childPosition);
					startTaskActivity(lecture);
					return true;
				}
				return false;
			}
		});

		folderList.setOnItemLongClickListener(new LongClick());

		createBtn.setOnClickListener(new OnClickListener() {
			@Override
			public void onClick(View v) {
				gc.clearModeFlag();
				showCreateDialog();
			}
		});

		moveBtn.setOnClickListener(new OnClickListener() {
			@Override
			public void onClick(View v) {
				startDownloadActivity();
			}
		});

		clearFlagBtn.setOnClickListener(new OnClickListener() {
			@Override
			public void onClick(View v) {
				gc.clearModeFlag();
			}

		});

		updateBtn.setOnClickListener(new OnClickListener() {
			@Override
			public void onClick(View v) {
				if (noLectures()) {
					showAlertDialog("先に講義を追加してね☆");
				} else {
					showProgressDialog("サーバにアクセスしています・・・");
					updateTaskInfo();
				}
			}
		});
	}

	@Override
	public void onStart() {
		super.onStart();
		getArrayAdapter();
		writeListToXml();
	}

	@Override
	public void onDestroy() {
		super.onDestroy();
		if (database != null) {
			database.close();
		}
	}

	/**
	 * 画面レイアウトから、インスタンスを確保するメソッド
	 */
	private void setLayout() {
		createBtn = (ImageButton) findViewById(R.id.createFolderBtn);
		moveBtn = (ImageButton) findViewById(R.id.moveBtn);
		clearFlagBtn = (ImageButton) findViewById(R.id.clearFlagBtn);
		updateBtn = (ImageButton) findViewById(R.id.updateBtn);
		folderList = (ExpandableListView) findViewById(R.id.folderList);
		folderList.setAdapter(listAdapter);
		t = (TextView) findViewById(R.id.MessageTextView);
		gc = new GroupClick();
		getArrayAdapter();
	}

	private void changeCurrentMode(Mode mode) {
		switch (mode) {
		case NORMAL:
			createBtn.setVisibility(View.VISIBLE);
			moveBtn.setVisibility(View.VISIBLE);
			updateBtn.setVisibility(View.VISIBLE);
			clearFlagBtn.setVisibility(View.GONE);
			t.setVisibility(View.GONE);
			break;
		case MOVE_LECTURE:
		case SWAP_FOLDER:
		case MERGE_FOLDER:
			createBtn.setVisibility(View.GONE);
			moveBtn.setVisibility(View.GONE);
			updateBtn.setVisibility(View.GONE);
			clearFlagBtn.setVisibility(View.VISIBLE);
			t.setVisibility(View.VISIBLE);
			break;
		}

		switch (mode) {
		case NORMAL:
			break;
		case MOVE_LECTURE:
			t.setText("移す先のフォルダを選ぶどん!");
			break;
		case SWAP_FOLDER:
			t.setText("スワップする");
			break;
		case MERGE_FOLDER:
			t.setText("マージ先のフォルダを選択");
			break;
		}

		currentMode = mode;
	}

	/**
	 * 初回起動時、フォルダー構成のxmlが存在しないため、仮のリストを構成するメソッド こちらは、フォルダーを作成する
	 */
	private void makeDefaultList() {
		defaultFolder = new LectureFolder("default", new ArrayList<Lecture>());
		defaultFolder.setDefault();
		listAdapter.addGroup(defaultFolder);
		makeDefaultChildList();
	}

	/**
	 * 初回起動時、フォルダー構成のxmlが存在しないため、仮のリストを構成するメソッド. こちらは、講義のリストを作成する。
	 */
	private void makeDefaultChildList() {
		defaultFolder.addAll(Lecture.findSelectedAll(database));
	}

	/**
	 * expandableListViewにアダプターをセットするメソッド xmlから構成データを受け取り、新しいアダプターを構成する。
	 * いちいち読み込むのは、セーブ状態と現在の内容を一致させたいため。
	 */
	private void getArrayAdapter() {

		loadXml();
		if (listAdapter.isEmpty()) {
			makeDefaultList();
		}

		List<Lecture> selectedList = Lecture.findSelectedAll(database);

		// あるフォルダの
		for (int i = 0; i < listAdapter.getGroupCount(); i++) {
			List<Lecture> lecList = listAdapter.getGroup(i).getLectures();
			// リストの要素が
			for (int j = 0; j < lecList.size(); j++) {
				Lecture lec = lecList.get(j);
				boolean isExist = false;
				// 選択された講義に入っているか
				for (int k = 0; k < selectedList.size(); k++) {
					if (lec.getId() == selectedList.get(k).getId()) {
						selectedList.remove(k);
						isExist = true;
						break;
					}
				}
				// 選択されたリストにない
				if (!isExist) {
					lecList.remove(j);
					j--;
				}
			}
		}

		defaultFolder.addAll(selectedList);
		listAdapter.notifyDataSetChanged();
		Log.d("Mes", "Adapter Update");
	}

	/**
	 * 新規作成するフォルダーの名前を入力するダイアログを表示する。 現在は、画面上の「新規作成」ボタンを押下することで表示される。
	 */
	private void showCreateDialog() {
		final EditText editText = new EditText(this);
		editText.setInputType(InputType.TYPE_CLASS_TEXT);
		String defaultNewFolderName = "新しいフォルダ";
		String newFolderName = defaultNewFolderName;
		if (isExist(newFolderName)) {
			for (int i = 2; i <= listAdapter.getGroupCount(); i++) {
				newFolderName = defaultNewFolderName + i;
				if (!isExist(newFolderName))
					break;
			}
		}
		editText.setText(newFolderName);

		AlertDialog alertDialog = new AlertDialog.Builder(this)
				.setTitle("新規作成").setView(editText)
				.setPositiveButton("作成", new DialogInterface.OnClickListener() {
					@Override
					public void onClick(DialogInterface dialog, int which) {
						String s = editText.getText().toString();
						if (isExist(s)) {
							Toast.makeText(LectureFolderActivity.this,
									"名前がかぶります。適当な名前を入力してください。",
									Toast.LENGTH_SHORT).show();
						} else {
							LectureFolder newFolder = new LectureFolder(s);
							listAdapter.addGroup(
									listAdapter.getGroupCount() - 1, newFolder);
							syncGroupExpansion();
							writeListToXml();
						}
					}
				}).setCancelable(true).create();
		alertDialog.setCanceledOnTouchOutside(true);
		alertDialog.show();
	}

	/**
	 * スワップを選択したフォルダと、指定先のフォルダを入れ替えるメソッド
	 * 
	 * @param position1
	 *            　「スワップする」を選択したフォルダのリスト番号
	 * @param position2
	 *            　スワップ先に指定したフォルダのリスト番号
	 */
	public void swapGroup(int position1, int position2) {
		if (!(listAdapter.getGroup(position1).isDefault() || listAdapter
				.getGroup(position2).isDefault())) {
			listAdapter.swapGroup(position1, position2);
			syncGroupExpansion();
			writeListToXml();
		} else {
			Toast.makeText(LectureFolderActivity.this,
					"It is default! can not swap!", Toast.LENGTH_SHORT).show();
		}
	}

	public void mergeGroup(int position1, int position2) {
		LectureFolder mergeTo = listAdapter.mergeGroup(position1, position2);
		mergeTo.sortLectureList();
		listAdapter.notifyDataSetChanged();
		syncGroupExpansion();
		writeListToXml();
	}

	/**
	 * 講義のアイテムを、別のフォルダーに移動するメソッド
	 * 
	 * @param toGroup
	 *            　移動先のフォルダーのリスト番号
	 * @param fromGroup
	 *            　移動元のフォルダーのリスト番号
	 * @param fromChild
	 *            　移動する講義アイテムのリスト番号
	 */
	public void moveChild(int toGroup, int fromGroup, int fromChild) {
		listAdapter.moveChild(toGroup, fromGroup, fromChild);
		LectureFolder toFolder = (LectureFolder) listAdapter.getGroup(toGroup);
		toFolder.sortLectureList();
		listAdapter.notifyDataSetChanged();
		writeListToXml();
	}

	/**
	 * OnGroupClickListenerを拡張し、フォルダーのスワップ、講義アイテムの移動を行えるようにした内部クラス
	 * 
	 * @author yuta-sh4k
	 * 
	 */
	class GroupClick implements OnGroupClickListener {
		private int group, child;

		@Override
		public boolean onGroupClick(ExpandableListView parent, View v,
				int groupPosition, long id) {
			switch (currentMode) {
			case NORMAL:
				if (parent.expandGroup(groupPosition)) {
					for (int i = 0; i < listAdapter.getGroupCount(); i++) {
						if (i != groupPosition)
							parent.collapseGroup(i);
					}
				} else {
					parent.collapseGroup(groupPosition);
				}
				break;

			case MOVE_LECTURE:
				moveChild(groupPosition, group, child);
				changeCurrentMode(Mode.NORMAL);
				break;

			case SWAP_FOLDER:
				swapGroup(group, groupPosition);
				changeCurrentMode(Mode.NORMAL);
				break;

			case MERGE_FOLDER:
				mergeGroup(group, groupPosition);
				changeCurrentMode(Mode.NORMAL);
				break;
			}
			return true;
		}

		public void setMoveMode(int group, int child) {
			changeCurrentMode(Mode.MOVE_LECTURE);
			this.group = group;
			this.child = child;
		}

		public void setSwapMode(int group) {
			changeCurrentMode(Mode.SWAP_FOLDER);
			this.group = group;
		}

		public void setMergeMode(int group) {
			changeCurrentMode(Mode.MERGE_FOLDER);
			this.group = group;
		}

		public void clearModeFlag() {
			changeCurrentMode(Mode.NORMAL);
		}
	}

	/**
	 * OnItemLongClickListenerを拡張し、講義アイテムの移動、フォルダーのスワップ、リネーム、削除を行えるようにした内部クラス
	 * 
	 * @author yuta-sh4k
	 * 
	 */
	class LongClick implements OnItemLongClickListener {
		@Override
		public boolean onItemLongClick(AdapterView<?> parent, View v,
				int position, long id) {
			ExpandableListView listView = (ExpandableListView) parent;
			long packed = listView.getExpandableListPosition(position);
			int group = ExpandableListView.getPackedPositionGroup(packed);
			int child = ExpandableListView.getPackedPositionChild(packed);

			if (ExpandableListView.getPackedPositionType(packed) == ExpandableListView.PACKED_POSITION_TYPE_CHILD) {
				gc.setMoveMode(group, child);
			} else if (ExpandableListView.getPackedPositionType(packed) == ExpandableListView.PACKED_POSITION_TYPE_GROUP) {
				showSelectDialog(group);

			}
			return true;
		}

	}

	/**
	 * アイテムを長押しした際に呼び出されるダイアログ。リネーム、削除、スワップを選択できる。
	 * 
	 * @param group
	 *            　選択されたフォルダーのリスト番号
	 */
	private void showSelectDialog(final int group) {
		String[] SelectList = { "リネーム", "削除", "スワップ", "フォルダをマージ" };

		AlertDialog.Builder builder = new AlertDialog.Builder(
				LectureFolderActivity.this);
		builder.setItems(SelectList, new DialogInterface.OnClickListener() {

			@Override
			public void onClick(DialogInterface dialog, int which) {
				switch (which) {
				case 0: {
					gc.clearModeFlag();
					showRenameDialog(group);
					break;
				}
				case 1: {
					gc.clearModeFlag();
					deleteFolderElement(group);

					break;
				}
				case 2: {
					gc.setSwapMode(group);
					break;
				}
				case 3: {
					if (listAdapter.getGroup(group).isDefault()) {
						Toast.makeText(LectureFolderActivity.this,
								"It is default! can not mergeFrom",
								Toast.LENGTH_SHORT).show();
					} else {
						gc.setMergeMode(group);
					}
				}
				}
			}
		});

		builder.setCancelable(true);
		builder.create().show();
	}

	/**
	 * 指定した番号のフォルダーを削除し、講義アイテムをデフォルトフォルダに移動するメソッド。
	 * ただし、デフォルトに設定されたフォルダにこのメソッドが実行された場合、削除命令は取り消される。
	 * 
	 * @param numberOfSelectedListItem
	 *            　削除するフォルダのリスト番号
	 */
	private void deleteFolderElement(int numberOfSelectedListItem) {
		if (listAdapter.getGroup(numberOfSelectedListItem).isDefault()) {
			Toast.makeText(this, "It is default folder! can't delete.",
					Toast.LENGTH_LONG).show();
		} else {
			LectureFolder deletedFolderElement = listAdapter
					.removeGroup(numberOfSelectedListItem);

			defaultFolder.addAll(deletedFolderElement.getLectures());
			listAdapter.notifyDataSetChanged();
			syncGroupExpansion();
			writeListToXml();
		}
	}

	// フォルダの構造が変化したときに、フォルダの開閉状況を Adapter と同期する
	private void syncGroupExpansion() {
		for (int i = 0; i < listAdapter.getGroupCount(); i++) {
			if (listAdapter.getGroupType(i) == PandaAdapter.GROUP_TYPE_EXPANDED) {
				folderList.expandGroup(i);
			} else {
				folderList.collapseGroup(i);
			}
		}
	}

	/**
	 * 　フォルダーのリネームを行うダイアログを表示するめそっど 　基本的な構成はshowCreateDialogと同じ
	 * 
	 * @param group
	 */
	private void showRenameDialog(final int group) {
		final EditText editText = new EditText(this);
		editText.setInputType(InputType.TYPE_CLASS_TEXT);
		editText.setText(listAdapter.getGroup(group).getTitle());

		AlertDialog alertDialog = new AlertDialog.Builder(this)
				.setTitle("リネーム")
				.setView(editText)
				.setPositiveButton("リネーム",
						new DialogInterface.OnClickListener() {
							@Override
							public void onClick(DialogInterface dialog,
									int which) {
								LectureFolder folder = listAdapter
										.getGroup(group);
								String newName = editText.getText().toString();
								String existName = folder.getTitle();
								if ((!existName.equals(newName))
										&& isExist(newName)) {
									Toast.makeText(LectureFolderActivity.this,
											"存在する名前です。別の名前にしてください",
											Toast.LENGTH_SHORT).show();
								} else {
									folder.setTitle(newName);
									writeListToXml();
									listAdapter.notifyDataSetChanged();
								}
							}
						}).setCancelable(true).create();
		alertDialog.setCanceledOnTouchOutside(true);
		alertDialog.show();
	}

	private static final int TOKEN_REQUEST_CODE = 1;

	@Override
	protected void onActivityResult(int requestCode, int resultCode, Intent data) {
		switch (requestCode) {
		case TOKEN_REQUEST_CODE:
			onTokenFinish();
			break;
		default:
			Log.w(TAG, "Unknown request code: " + requestCode);
		}
	}

	/**
	 * 講義画面へ遷移する準備を行い、実際に遷移させるメソッド
	 * 
	 * @param lecture
	 *            　選択された講義オブジェクトのインスタンス
	 */
	private void startTaskActivity(Lecture lecture) {
		Intent intent = new Intent(LectureFolderActivity.this,
				LectureActivity.class);
		intent.putExtra(LECTURE_KEY, lecture.getId());
		startActivity(intent);
	}

	// ダウンロードページへ遷移する
	private void startDownloadActivity() {
		Intent intent = new Intent(LectureFolderActivity.this,
				DownloadActivity.class);
		startActivity(intent);
	}

	// トーク認証ページへ遷移する
	private void startTokenAuthenticationActivity() {
		Intent intent = new Intent(LectureFolderActivity.this,
				TokenAuthenticationActivity.class);
		startActivityForResult(intent, TOKEN_REQUEST_CODE);
	}

	private void onTokenFinish() {
		if (!Token.isExist(this)) {
			finish();
		}
	}

	private boolean isExist(String folderName) {
		for (int i = 0; i < listAdapter.getGroupCount(); i++) {
			if (folderName.equals(listAdapter.getGroup(i).getTitle())) {
				return true;
			}
		}
		return false;
	}

	public void writeListToXml() {
		XmlUtility.writeXmlFile(this, listAdapter.getGroups());
	}

	public void loadXml() {
		listAdapter.clear();
		for (LectureFolder f : XmlUtility.readXmlFile(this)) {
			listAdapter.addGroup(f);
		}
		// defaultは一番下から移動できないので、size-1でいいはず
		if (!listAdapter.isEmpty()) {
			defaultFolder = listAdapter
					.getGroup(listAdapter.getGroupCount() - 1);
		}
	}

	// 課題情報の更新
	private void updateTaskInfo() {
		apiClient.getLectures(Lecture.findSelectedAll(database),
				new SimpleHttpResponseListener() {
					@Override
					public void preExec() {
						dismissProgressDialog();
					}

					@Override
					public void onSuccess(String body) {
						jsonParse(body);
					}

					@Override
					public void onFailure(Exception e) {
						showAlertDialog("サーバからデータを取得できませんでした");
						e.printStackTrace();
					}

					@Override
					public void onHttpFailure(StatusLine status, String body) {
						Log.e(TAG,
								"getLectures: returned "
										+ status.getStatusCode());
						showAlertDialog("サーバからデータを取得できませんでした");
					}
				});
	}

	// 警告ダイアログ(ネットワークエラー時に表示)
	public void showAlertDialog(String message) {
		new AlertDialog.Builder(this).setTitle("警告").setMessage(message)
				.setPositiveButton("OK", new DialogInterface.OnClickListener() {
					@Override
					public void onClick(DialogInterface dialog, int which) {
					}
				}).setCancelable(false).show();
	}

	// 課題情報付き講義リストから、課題情報を取り出してデータベースに格納する
	private void jsonParse(String data) {
		try {
			JSONArray root = new JSONArray(data);
			for (int i = 0; i < root.length(); i++) {
				JSONObject lecture = root.getJSONObject(i);
				JSONObject teacher = lecture.getJSONObject("teacher");
				Teacher t1 = new Teacher(teacher.getInt("id"),
						teacher.getString("name"),
						teacher.isNull("icon_path") ? null : teacher
								.getString("icon_path"));
				t1.save(database, this);
				Lecture l = new Lecture(lecture.getInt("id"),
						lecture.getString("name"), t1);
				JSONArray taskList = lecture.getJSONArray("tasks");
				for (int j = 0; j < taskList.length(); j++) {
					JSONObject task = taskList.getJSONObject(j);
					Task t2 = new Task(task.getInt("id"),
							task.getString("name"), l);
					t2.save(database);
					JSONObject apk = task.getJSONObject("apk");
					Apk a = new Apk(apk.getInt("id"), apk.getInt("revision"),
							apk.getString("name"), t2);
					a.save(database);
				}
			}
		} catch (JSONException e) {
			e.printStackTrace();
		}
	}

	// 講義が選択されているか判定する
	private boolean noLectures() {
		for (int i = 0; i < listAdapter.getGroupCount(); i++) {
			if (listAdapter.getChildrenCount(i) > 0)
				return false;
		}
		return true;
	}

	// プログレスダイアログ(サーバとの通信中に表示)
	public void showProgressDialog(String message) {
		pDialog = new ProgressDialog(this);
		pDialog.setTitle("通信中");
		pDialog.setMessage(message);
		pDialog.setProgressStyle(ProgressDialog.STYLE_SPINNER);
		pDialog.setCancelable(false);
		pDialog.show();
	}

	// プログレスダイアログを非表示にする
	public void dismissProgressDialog() {
		pDialog.dismiss();
	}
}
