package jp.ac.titech.sharp4k.cuten;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import jp.ac.titech.sharp4k.cuten.util.XmlUtility;

import jp.ac.titech.sharp4k.cuten.R;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.DialogInterface;
import android.content.Intent;
import android.database.sqlite.SQLiteDatabase;
import android.os.Bundle;
import android.text.InputType;
import android.util.Log;
import android.view.View;
import android.view.View.OnClickListener;
import android.widget.AdapterView;
import android.widget.AdapterView.OnItemLongClickListener;
import android.widget.BaseExpandableListAdapter;
import android.widget.Button;
import android.widget.EditText;
import android.widget.ExpandableListView;
import android.widget.ExpandableListView.OnChildClickListener;
import android.widget.ExpandableListView.OnGroupClickListener;
import android.widget.TextView;
import android.widget.Toast;

public class LectureFolderActivity extends Activity {
	public class Folder implements PandaAdapter.Group {
		private String title;
		private List<Lecture> lectures;
		private boolean isDefault;

		public Folder(String title) {
			this(title, new ArrayList<Lecture>());
		}

		public Folder(String title, List<Lecture> lectures) {
			this.title = title;
			this.lectures = lectures;
			this.isDefault = false;
		}

		public void setTitle(String title) {
			this.title = title;
		}

		public String getTitle() {
			return title;
		}

		public List<Lecture> getLectures() {
			return lectures;
		}

		public void setDefault() {
			isDefault = true;
		}

		public boolean isDefault() {
			return isDefault;
		}

		public void add(Lecture lecture) {
			getLectures().add(lecture);
		}

		public void addAll(Collection<? extends Lecture> lectures) {
			getLectures().addAll(lectures);
		}

		public Lecture get(int pos) {
			return getLectures().get(pos);
		}

		public Lecture remove(int pos) {
			return getLectures().remove(pos);
		}

		@Override
		public String getLabel() {
			return title;
		}

		@Override
		public List<? extends PandaAdapter.Child> getChildren() {
			return getLectures();
		}
	}

	/** Called when the activity is first created. */

	// フォルダー名を格納するHashMapキー
	public static final String FOLDER_TITLE = "TITLE";
	// フォルダーがデフォルト(削除できない、DL先となるフォルダー)であることを判定するためのHAshMapキー。デフォルトに設定するものに付加するので、
	// 値はつける必要は特になし。
	public static final String FOLDER_DEFAULT = "DEFAULT";
	// 各講義の名前を格納するHashMapキー
	public static final String LECTURE_LIST_TITLE = "Title";
	// 講義の下に簡単な説明を付加するさいに、値を格納するHashMapキー。現在は講師名を格納
	public static final String LECTURE_LIST_SUMMERY = "Summery";
	// 各講義のインスタンスを格納するHashMapキー。講義画面への遷移には、ここに格納されたLactureインスタンスを用いる。
	public static final String LECTURE_ID = "LectureID";

	// 画面遷移の際に引き渡すインテントに付加するキー。
	public static final String LECTURE_KEY = "LECTURE";

	// 新規作成ボタン
	Button createBtn, moveBtn, clearFlagBtn;
	// 講義リスト(のびーる)
	ExpandableListView folderList;

	// フォルダーのデータを格納するリスト。
	private List<Folder> folders;

	// 削除することのできないデフォルトのフォルダーのインスタンス。デフォルトであることの確認や、ダウンロードの際の宛先として使える？
	private Folder defaultFolder;

	// ExpandableListViewの内部を構成するアダプター。
	BaseExpandableListAdapter listAdapter;
	// 画面上部につけたテキストビュー。現在は主にデバッグよう。
	TextView t;
	// OnGroupClickListenerの拡張クラスのインスタンス。スワップができるように拡張されている。
	GroupClick gc;

	SQLiteDatabase database;

	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.main);

		folders = new ArrayList<Folder>();
		SQLHelper sqlHelper = new SQLHelper(this);
		database = sqlHelper.getReadableDatabase();
		setLayout();

		setActionListener();
	}

	/**
	 * 各ビューにリスナーをセットするメソッド
	 */
	private void setActionListener() {
		folderList.setOnGroupClickListener(gc);

		folderList.setOnChildClickListener(new OnChildClickListener() {
			@Override
			public boolean onChildClick(ExpandableListView parent, View v,
					int groupPosition, int childPosition, long id) {
				Lecture lecture = folders.get(groupPosition).get(childPosition);
				startTaskActivity(lecture);
				return false;
			}
		});

		folderList.setOnItemLongClickListener(new LongClick(gc, t));

		createBtn.setOnClickListener(new OnClickListener() {
			@Override
			public void onClick(View v) {
				showCreateDialog();
			}
		});

		moveBtn.setOnClickListener(new OnClickListener() {
			@Override
			public void onClick(View v) {
				startDownloadActivity();
			}
		});

		clearFlagBtn.setOnClickListener(new OnClickListener() {
			@Override
			public void onClick(View v) {
				gc.clearModeFlag();
			}

		});
	}

	/**
	 * 画面レイアウトから、インスタンスを確保するメソッド
	 */
	private void setLayout() {
		createBtn = (Button) findViewById(R.id.createFolderBtn);
		moveBtn = (Button) findViewById(R.id.moveBtn);
		clearFlagBtn = (Button) findViewById(R.id.clearFlagBtn);
		folderList = (ExpandableListView) findViewById(R.id.folderList);
		t = (TextView) findViewById(R.id.MessageTextView);
		gc = new GroupClick(t);
		getArrayAdapter();
	}

	/**
	 * 初回起動時、フォルダー構成のxmlが存在しないため、仮のリストを構成するメソッド こちらは、フォルダーを作成する
	 */
	private void makeDefaultList() {
		defaultFolder = new Folder("default", new ArrayList<Lecture>());
		defaultFolder.setDefault();
		folders.add(defaultFolder);
		makeDefaultChildList();
	}

	/**
	 * 初回起動時、フォルダー構成のxmlが存在しないため、仮のリストを構成するメソッド. こちらは、講義のリストを作成する。
	 */
	private void makeDefaultChildList() {
		defaultFolder.addAll(Lecture.findSelected(database));
	}

	/**
	 * expandableListViewにアダプターをセットするメソッド xmlから構成データを受け取り、新しいアダプターを構成する。
	 * いちいち読み込むのは、セーブ状態と現在の内容を一致させたいため。
	 */
	private void getArrayAdapter() {

		loadXml();
		if (folders.isEmpty()) {
			makeDefaultList();
		}

		listAdapter = new PandaAdapter(this, folders);

		Log.d("Mes", "Adapter Update");
		folderList.setAdapter(listAdapter);
	}

	/**
	 * 新規作成するフォルダーの名前を入力するダイアログを表示する。 現在は、画面上の「新規作成」ボタンを押下することで表示される。
	 */
	private void showCreateDialog() {
		final EditText editText = new EditText(this);
		editText.setInputType(InputType.TYPE_CLASS_TEXT);
		String defaultNewFolderName = "新しいフォルダ";
		String newFolderName = defaultNewFolderName;
		if (isExist(newFolderName)) {
			for (int i = 2; i <= folders.size(); i++) {
				newFolderName = defaultNewFolderName + i;
				if (!isExist(newFolderName))
					break;
			}
		}
		editText.setText(newFolderName);
		AlertDialog.Builder alertDialogBuilder = new AlertDialog.Builder(this);
		alertDialogBuilder.setTitle("新規作成");
		alertDialogBuilder.setView(editText);

		alertDialogBuilder.setPositiveButton("作成",
				new DialogInterface.OnClickListener() {

					@Override
					public void onClick(DialogInterface dialog, int which) {
						String s = editText.getText().toString();
						if (isExist(s)) {
							Toast.makeText(LectureFolderActivity.this,
									"名前がかぶります。適当な名前を入力してください。",
									Toast.LENGTH_SHORT).show();
						} else {
							Folder newFolder = new Folder(s);
							folders.add(folders.size() - 1, newFolder);
							writeListToXml();
							getArrayAdapter();
						}
					}
				});

		alertDialogBuilder.setCancelable(true);
		AlertDialog alertDialog = alertDialogBuilder.create();
		alertDialog.show();
	}

	/**
	 * スワップを選択したフォルダと、指定先のフォルダを入れ替えるメソッド
	 * 
	 * @param position1
	 *            　「スワップする」を選択したフォルダのリスト番号
	 * @param position2
	 *            　スワップ先に指定したフォルダのリスト番号
	 */
	public void swapGroup(int position1, int position2) {
		if (!(folders.get(position1).isDefault() || folders.get(position2)
				.isDefault())) {
			Folder tmp = folders.get(position1);
			folders.set(position1, folders.get(position2));
			folders.set(position2, tmp);
			listAdapter.notifyDataSetChanged();
			writeListToXml();
			getArrayAdapter();
		} else {
			Toast.makeText(LectureFolderActivity.this,
					"It is default! can not swap!", Toast.LENGTH_SHORT).show();
		}
	}

	public void mergeGroup(int position1, int position2) {
		if (position1 != position2) {
			Folder margeTo = folders.get(position2);
			margeTo.addAll(folders.get(position1).getLectures());
			folders.remove(position1);
			writeListToXml();
			getArrayAdapter();
		}
	}

	/**
	 * 講義のアイテムを、別のフォルダーに移動するメソッド
	 * 
	 * @param toGroup
	 *            　移動先のフォルダーのリスト番号
	 * @param fromGroup
	 *            　移動元のフォルダーのリスト番号
	 * @param fromChild
	 *            　移動する講義アイテムのリスト番号
	 */
	public void moveChild(int toGroup, int fromGroup, int fromChild) {
		Lecture tmp = folders.get(fromGroup).get(fromChild);
		folders.get(fromGroup).remove(fromChild);
		folders.get(toGroup).add(tmp);
		listAdapter.notifyDataSetChanged();
		writeListToXml();
		getArrayAdapter();
	}

	/**
	 * OnGroupClickListenerを拡張し、フォルダーのスワップ、講義アイテムの移動を行えるようにした内部クラス
	 * 
	 * @author yuta-sh4k
	 * 
	 */
	class GroupClick implements OnGroupClickListener {
		boolean moveMode;
		boolean mergeMode;
		int group, child;
		TextView t;

		GroupClick(TextView t) {
			moveMode = false;
			this.t = t;
		}

		@Override
		public boolean onGroupClick(ExpandableListView parent, View v,
				int groupPosition, long id) {
			if (moveMode) {
				if (child == -1) {
					swapGroup(group, groupPosition);
				} else {
					moveChild(groupPosition, group, child);
				}
				t.setText("テストでござる!");
				moveMode = false;
			} else if (mergeMode) {
				if (child == -1) {
					mergeGroup(group, groupPosition);
				}
				t.setText("マージのテストでござる！");
				mergeMode = false;
			} else {
				if (parent.expandGroup(groupPosition))
					for (int i = 0; i < folders.size(); i++) {
						if (i != groupPosition)
							parent.collapseGroup(i);
					}
				else
					parent.collapseGroup(groupPosition);
			}
			return true;
		}

		public void setMoveMode(int group, int child) {
			moveMode = true;
			mergeMode = false;
			this.group = group;
			this.child = child;
		}

		public void setMergeMode(int group, int child) {
			mergeMode = true;
			moveMode = false;
			this.group = group;
			this.child = child;
		}

		public void clearModeFlag() {
			if (mergeMode) {
				mergeMode = false;
				t.setText("操作をキャンセル");
			}
			if (moveMode) {
				moveMode = false;
				t.setText("操作をキャンセル");
			}

		}
	}

	/**
	 * OnItemLongClickListenerを拡張し、講義アイテムの移動、フォルダーのスワップ、リネーム、削除を行えるようにした内部クラス
	 * 
	 * @author yuta-sh4k
	 * 
	 */
	class LongClick implements OnItemLongClickListener {
		GroupClick gc;
		TextView t;

		LongClick(GroupClick gc, TextView t) {
			this.gc = gc;
			this.t = t;
		}

		@Override
		public boolean onItemLongClick(AdapterView<?> parent, View v,
				int position, long id) {
			ExpandableListView listView = (ExpandableListView) parent;
			long packed = listView.getExpandableListPosition(position);
			int group = ExpandableListView.getPackedPositionGroup(packed);
			int child = ExpandableListView.getPackedPositionChild(packed);

			if (ExpandableListView.getPackedPositionType(packed) == ExpandableListView.PACKED_POSITION_TYPE_CHILD) {
				t.setText("移す先のフォルダを選ぶどん!");
				gc.setMoveMode(group, child);
			} else if (ExpandableListView.getPackedPositionType(packed) == ExpandableListView.PACKED_POSITION_TYPE_GROUP) {
				showSelectDialog(group, child);

			}
			return true;
		}

	}

	/**
	 * アイテムを長押しした際に呼び出されるダイアログ。リネーム、削除、スワップを選択できる。
	 * 
	 * @param group
	 *            　選択されたフォルダーのリスト番号
	 * @param child
	 *            　選択された講義アイテムのリスト番号
	 */
	private void showSelectDialog(int group, int child) {

		final int numberOfSelectedListItem = group;
		final int finalChild = child;
		String[] SelectList = { "リネーム", "削除", "スワップ", "フォルダをマージ" };

		AlertDialog.Builder builder = new AlertDialog.Builder(
				LectureFolderActivity.this);
		builder.setItems(SelectList, new DialogInterface.OnClickListener() {

			@Override
			public void onClick(DialogInterface dialog, int which) {
				switch (which) {
				case 0: {
					showRenameDialog(numberOfSelectedListItem);
					break;
				}
				case 1: {
					deleteFolderElement(numberOfSelectedListItem);

					break;
				}
				case 2: {
					t.setText("スワップする");
					gc.setMoveMode(numberOfSelectedListItem, finalChild);
					break;
				}
				case 3: {
					if (folders.get(numberOfSelectedListItem).isDefault()) {
						Toast.makeText(LectureFolderActivity.this,
								"It is default! can not mergeFrom",
								Toast.LENGTH_SHORT).show();
					} else {
						t.setText("マージ先のフォルダを選択");
						gc.setMergeMode(numberOfSelectedListItem, finalChild);
					}
				}
				}
			}
		});

		builder.setCancelable(true);
		builder.create().show();
	}

	/**
	 * 指定した番号のフォルダーを削除し、講義アイテムをデフォルトフォルダに移動するメソッド。
	 * ただし、デフォルトに設定されたフォルダにこのメソッドが実行された場合、削除命令は取り消される。
	 * 
	 * @param numberOfSelectedListItem
	 *            　削除するフォルダのリスト番号
	 */
	private void deleteFolderElement(int numberOfSelectedListItem) {
		if (folders.get(numberOfSelectedListItem).isDefault()) {
			Toast.makeText(this, "It is default folder! can't delete.",
					Toast.LENGTH_LONG).show();
		} else {
			Folder deletedFolderElement = folders
					.remove(numberOfSelectedListItem);

			defaultFolder.addAll(deletedFolderElement.getLectures());
			writeListToXml();
			getArrayAdapter();
		}
	}

	/**
	 * 　フォルダーのリネームを行うダイアログを表示するめそっど 　基本的な構成はshowCreateDialogと同じ
	 * 
	 * @param group
	 */
	private void showRenameDialog(int group) {
		final EditText editText = new EditText(this);
		final int finalGroup = group;
		editText.setInputType(InputType.TYPE_CLASS_TEXT);
		editText.setText(folders.get(group).getTitle());
		AlertDialog.Builder alertDialogBuilder = new AlertDialog.Builder(this);
		alertDialogBuilder.setTitle("リネーム");
		alertDialogBuilder.setView(editText);

		alertDialogBuilder.setPositiveButton("リネーム",
				new DialogInterface.OnClickListener() {

					@Override
					public void onClick(DialogInterface dialog, int which) {
						Folder folder = folders.get(finalGroup);
						String newName = editText.getText().toString();
						String existName = folder.getTitle();
						if ((!existName.equals(newName)) && isExist(newName)) {
							Toast.makeText(LectureFolderActivity.this,
									"存在する名前です。別の名前にしてください", Toast.LENGTH_SHORT)
									.show();
						} else {
							folder.setTitle(newName);
							writeListToXml();
							getArrayAdapter();
						}
					}
				});

		alertDialogBuilder.setCancelable(true);
		AlertDialog alertDialog = alertDialogBuilder.create();
		alertDialog.show();
	}

	/**
	 * 講義画面へ遷移する準備を行い、実際に遷移させるメソッド
	 * 
	 * @param lecture
	 *            　選択された講義オブジェクトのインスタンス
	 */
	private void startTaskActivity(Lecture lecture) {
		Intent intent = new Intent(LectureFolderActivity.this,
				LectureActivity.class);
		intent.putExtra(LECTURE_KEY, lecture.getId());
		startActivity(intent);
	}

	// ダウンロードページへ遷移する
	private void startDownloadActivity() {
		Intent intent = new Intent(LectureFolderActivity.this,
				JsonParseTestActivity.class);
		startActivity(intent);
	}

	private boolean isExist(String folderName) {
		for (Folder folder : folders) {
			if (folderName.equals(folder.getTitle())) {
				return true;
			}
		}

		return false;
	}

	public void writeListToXml() {
		XmlUtility.writeXmlFile(this, folders);
	}

	public void loadXml() {
		folders = XmlUtility.readXmlFile(this);
		// defaultは一番下から移動できないので、size-1でいいはず
		if (!folders.isEmpty()) {
			defaultFolder = folders.get(folders.size() - 1);
		}
	}
}
