require 'test_helper'

class TasksControllerTest < ActionController::TestCase
  include Devise::TestHelpers

  setup do
    @apk = FactoryGirl.create :apk
    @task = @apk.task
    @lecture = @task.lecture
    @teacher = @lecture.teacher
    @apkfile = fixture_file_upload 'dummy.apk', Apk::MIME_TYPE
  end

  teardown do
    @apkfile.close!
  end

  test "should NOT get index" do
    assert_no_routes do
      get :index
    end
  end

  test "should NOT get new" do
    assert_no_routes do
      get :new
    end
  end

  test "should NOT create task" do
    assert_no_routes do
      post :create, task: { name: @task.name }
    end
  end

  test "should show task" do
    get :show, id: @task
    assert_response :success
    assert_match Regexp.new(Regexp.quote(@task.name)), @response.body
    assert_match Regexp.new(Regexp.quote(@apk.name)), @response.body
  end

  test "should get edit" do
    get :edit, :id => @task
    assert_response :success
  end

  test "should NOT destroy task" do
    assert_no_routes do
      delete :destroy, id: @task
    end
  end

  test "non-teacher should not get new" do
    user = FactoryGirl.create :user
    sign_in user

    get :new, :lecture_id => @lecture.id
    assert_redirected_to @lecture
  end

  test "teacher should get his new task" do
    sign_in @teacher

    get :new, :lecture_id => @lecture.id
    assert_response :success
    assert_not_nil assigns(:task)
  end

  test "teacher should not get new another teacher's task" do
    another_teacher = FactoryGirl.create :teacher
    sign_in another_teacher

    get :new, :lecture_id => @lecture.id
    assert_redirected_to @lecture
  end

  test "non-teacher should not create task" do
    user = FactoryGirl.create :user
    sign_in user

    task = FactoryGirl.build :task, :lecture => @lecture
    apk = FactoryGirl.build :apk, :task => task
    assert_difference('Task.count', 0) do
      assert_difference('Apk.count', 0) do
        create_apk apk, @apkfile
        assert_redirected_to @lecture
      end
    end
  end

  test "teacher should create his task" do
    sign_in @teacher

    task = FactoryGirl.build :task, :lecture => @lecture
    apk = FactoryGirl.build :apk, :task => task
    assert_difference('Task.count') do
      assert_difference('Apk.count') do
        create_apk apk, @apkfile
        created_apk = assigns :apk
        created_task = assigns :task
        assert_not_nil created_apk
        assert_not_nil created_task
        assert_equal task.name, created_task.name
        assert_equal apk.name, created_apk.name
        assert File.readable?(created_apk.path)
        assert_equal @lecture.id, created_task.lecture.id
      end
    end
  end

  test "teacher should not create another teacher's task" do
    another_teacher = FactoryGirl.create :teacher
    sign_in another_teacher

    task = FactoryGirl.build :task, :lecture => @lecture
    apk = FactoryGirl.build :apk, :task => task
    assert_difference('Task.count', 0) do
      assert_difference('Apk.count', 0) do
        create_apk apk, @apkfile
        assert_redirected_to @lecture
      end
    end
  end

  test "should reject empty apkfile" do
    user = FactoryGirl.create :user
    sign_in user

    task = FactoryGirl.build :task, :lecture => @lecture
    apk = FactoryGirl.build :apk, :task => task
    assert_difference('Task.count', 0) do
      assert_difference('Apk.count', 0) do
        create_apk apk, nil
      end
    end
  end

  test "should reject empty task name" do
    sign_in @teacher

    task = FactoryGirl.build :task, :name => nil, :lecture => @lecture
    apk = FactoryGirl.build :apk, :task => task
    assert_difference('Task.count', 0) do
      assert_difference('Apk.count', 0) do
        create_apk apk, @apkfile
      end
    end
  end

  test "should reject empty apk class name" do
    sign_in @teacher

    task = FactoryGirl.build :task, :lecture => @lecture
    apk = FactoryGirl.build :apk, :name => nil, :task => task
    assert_difference('Task.count', 0) do
      assert_difference('Apk.count', 0) do
        create_apk apk, @apkfile
      end
    end
  end

  test "teacher should not update another teacher's task" do
    another_teacher = FactoryGirl.create :teacher
    sign_in another_teacher

    update_apk @apk, @apkfile
    assert_redirected_to @task
    assert_not_nil flash[:alert]
  end

  test "user should not update task" do
    user = FactoryGirl.create :user
    sign_in user

    update_apk @apk, @apkfile
    assert_redirected_to @task
    assert_not_nil flash[:alert]
  end

  test "should increment revision when apkfile is given" do
    sign_in @teacher

    update_apk @apk, @apkfile
    assert_redirected_to @task
    assert_nil flash[:alert]
    created_apk = assigns :apk
    assert_equal @apk.revision+1, created_apk.revision
  end

  test "should not increment revision when apkfile is NOT given" do
    sign_in @teacher

    update_apk @apk, nil
    assert_redirected_to @task
    assert_nil flash[:alert]
    created_apk = assigns :apk
    assert_equal @apk.revision, created_apk.revision
  end

  test "should rollback when invalid input is given" do
    sign_in @teacher

    orig_task_name = @task.name
    orig_apk_name = @apk.name
    @task.name = @task.name.upcase
    @apk.name = ''
    create_apk @apk, nil
    assert_response :success
    apk = Apk.find @apk.id
    task = apk.task
    assert_equal orig_apk_name, apk.name
    assert_equal orig_task_name, task.name
  end

  def create_apk(apk, apkfile)
    task = apk.task
    lecture = task.lecture
    post :create, :lecture_id => lecture.id, :task => { :name => task.name }, :apk => { :name => apk.name }, :apkfile => { :file => apkfile }
  end

  def update_apk(apk, apkfile)
    task = apk.task
    put :update, :id => task, :task => { :name => task.name }, :apk => { :name => apk.name }, :apkfile => { :file => apkfile }
  end
end
