require 'rake/tasklib'
$LOAD_PATH.unshift File.expand_path('..', __FILE__)
require 'android'

class Rake::CutenAppTask < Rake::TaskLib
  attr_reader :sdk, :project

  def initialize(sdk)
    @sdk = sdk
    @project = sdk.project
    @before_resources = []
  end

  def define_all
    task :default => 'apk:aligned'
    define_buildconfig
    define_prepare
    define_resources
    define_compile
    define_apk
    define_push
  end

  def before_resources(&blk)
    @before_resources << blk
  end

  def define_prepare# {{{
    directory project.class_output_dir.to_s
    directory project.res_output_dir.to_s
    directory project.gen_dir.to_s
    file project.manifest_output_file.to_s => [project.output_dir] do
      cp 'AndroidManifest.xml', project.manifest_output_file.to_s
    end
    desc 'Prepare directories'
    task :prepare => [
      project.class_output_dir.to_s,
      project.res_output_dir.to_s,
      project.gen_dir.to_s,
      project.manifest_output_file.to_s,
    ]
  end# }}}

  def define_buildconfig# {{{
    desc 'Generate BuildConfig.java'
    task :buildconfig => [:prepare] do
      pkg = project.package
      project.gen_dir.join(*pkg.split('.')).join('BuildConfig.java').open('w') do |f|
        f.puts Android::BuildConfigGenerator.new(pkg, true).generate
      end
    end
  end# }}}

  def define_resources# {{{
    namespace :r do
      desc 'Generate R.java'
      task :generate => [:prepare] do
        @before_resources.each do |func|
          func.call
        end

        cmd = [
          sdk.aapt.to_s, 'package',
          '-f', '-m',
          '-M', project.manifest_output_file.to_s,
          '-S', project.res_dir.to_s,
          '-I', sdk.android_jar.to_s,
          '-J', project.gen_dir.to_s,
          '--generate-dependencies',
        ]
        sh *cmd
      end
    end
  end# }}}

  def define_compile# {{{
    desc 'Compile Java sources'
    task :javac => ['r:generate', :buildconfig] do
      cmd = [
        'javac', '-d', project.class_output_dir.to_s,
        '-classpath', sdk.classpath,
        '-sourcepath', [project.src_dir.to_s, project.gen_dir.to_s].join(':'),
        '-target', '1.5',
        '-bootclasspath', sdk.android_jar.to_s,
        '-encoding', 'UTF-8',
        '-g',
        '-source', '1.5',
      ]

      main = project.main_activity
      main_file = main && main.split('.').join(File::Separator) + '.java'
      project.src_dir.find do |path|
        if path.file? and path.extname == '.java'
          if main_file and path.to_s.end_with? main_file
            $stderr.puts "#{path.to_s}: excluding main activity"
          else
            cmd.push path.to_s
          end
        end
      end
      project.gen_dir.find do |path|
        if path.file? and path.extname == '.java'
          cmd.push path.to_s
        end
      end
      sh *cmd
    end

    desc 'Convert to dex format'
    task :dx => [:javac] do
      cmd = [
        sdk.dx.to_s, '--dex',
        '--output', project.dex_output_file.to_s,
        project.class_output_dir.to_s,
        sdk.annotations_jar.to_s
      ]
      sh *cmd
    end
  end# }}}

  def define_apk# {{{
    namespace :r do
      desc 'Crunch resources'
      task :crunch => 'r:generate' do
        cmd = [
          sdk.aapt.to_s, 'crunch',
          '-S', project.res_dir.to_s,
          '-C', project.res_output_dir.to_s,
        ]
        sh *cmd
      end

      desc 'Package resources'
      task :package => 'r:crunch' do
        cmd = [
          sdk.aapt.to_s, 'package',
          '--no-crunch', '-f', '--debug-mode', '--auto-add-overlay',
          '-M', project.manifest_output_file.to_s,
          '-S', project.res_output_dir.to_s,
          '-S', project.res_dir.to_s,
          '-I', sdk.android_jar.to_s,
          '-F', project.res_apk_file.to_s,
          '--generate-dependencies',
        ]
        if project.asset_dir.directory?
          cmd.push *['-A', project.asset_dir.to_s]
        end
        sh *cmd
      end
    end

    namespace :apk do
      desc 'Generate unaligned apk'
      task :unaligned => [:dx, 'r:package'] do
        if defined? RUBY_ENGINE and RUBY_ENGINE == 'jruby'
          require_relative 'cuten_app_jruby_helper'
          SdkLib.init sdk
          SdkLib.apkbuilder project.apk_unaligned_file.to_s, project.res_apk_file.to_s, project.dex_output_file.to_s
        else
          # XXX: apkbuilder is deprecated
          cmd = [
            sdk.apkbuilder.to_s,
            project.apk_unaligned_file.to_s,
            '-d', '-u',
            '-z', project.res_apk_file.to_s,
            '-f', project.dex_output_file.to_s,
          ]
          sh *cmd
        end
      end

      desc 'Generate apk'
      task :aligned => 'apk:unaligned' do
        cmd = [
          sdk.zipalign.to_s,
          '-f', '4',
          project.apk_unaligned_file.to_s,
          project.apk_aligned_file.to_s,
        ]
        sh *cmd
      end
    end
  end# }}}

  def define_push# {{{
    namespace :apk do
      desc 'Push apk to /sdcard'
      task :push => 'apk:aligned' do
        fname = 'MainClass.txt'
        if File.file? fname
          class_name = open(fname) { |f| f.gets.chomp }
          sh sdk.adb.to_s, 'shell', 'rm /sdcard/*.apk'
          sh sdk.adb.to_s, 'push', project.apk_aligned_file.to_s, "/sdcard/#{class_name}.apk"
        else
          abort "#{fname} not found!"
        end
      end
    end

    desc 'Clean'
    task :clean do
      rm_rf [project.output_dir.to_s, project.gen_dir.to_s]
    end
  end# }}}
end
