package jp.ac.titech.sharp4k.cuten;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.StatusLine;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.utils.URIUtils;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.params.HttpConnectionParams;
import org.apache.http.params.HttpParams;
import org.apache.http.util.EntityUtils;

import android.os.Handler;
import android.os.Looper;
import android.util.Log;

public class HttpAPIClientImpl implements HttpAPIClient {
	private static interface Proc2<T, U> {
		public void run(T arg1, U arg2);
	}

	private static final String TAG = HttpAPIClientImpl.class.getSimpleName();

	private static final ExecutorService executor = Executors
			.newFixedThreadPool(3);

	private static enum HttpMethod {
		GET, POST, PUT
	}

	public static final int CONNECTION_TIMEOUT_MSEC = 1000;
	public static final int SOCKET_TIMEOUT_MSEC = 1000;

	public static final String API_HOST = "www16307ue.sakura.ne.jp";
	public static final int API_PORT = 3001;

	private void api(final HttpMethod httpMethod, final String path,
			final List<NameValuePair> queries,
			final Proc2<HttpClient, HttpUriRequest> proc) {
		executor.execute(new Runnable() {
			@Override
			public void run() {
				final HttpClient httpClient = new DefaultHttpClient();
				final HttpParams params = httpClient.getParams();
				HttpConnectionParams.setConnectionTimeout(params,
						CONNECTION_TIMEOUT_MSEC);
				HttpConnectionParams.setSoTimeout(params, SOCKET_TIMEOUT_MSEC);
				final HttpUriRequest httpRequest = makeHttpRequest();
				proc.run(httpClient, httpRequest);
			}

			private HttpUriRequest makeHttpRequest() {
				URI uri = getFullURI();
				if (httpMethod.equals(HttpMethod.GET)) {
					return new HttpGet(uri);
				} else {
					HttpEntityEnclosingRequestBase req = httpMethod
							.equals(HttpMethod.POST) ? new HttpPost(uri)
							: new HttpPut(uri);
					try {
						req.setEntity(new UrlEncodedFormEntity(queries));
						return req;
					} catch (UnsupportedEncodingException e) {
						Log.e(TAG, "makeHttpRequest:");
						e.printStackTrace();
						return null;
					}
				}
			}

			private URI getFullURI() {
				try {
					String q = httpMethod.equals(HttpMethod.GET) ? getQuery()
							: null;
					return URIUtils.createURI("http", API_HOST, API_PORT, path,
							q, null);
				} catch (URISyntaxException ex) {
					// unreachable
					Log.d(TAG, "!!! YABAI !!!");
					return null;
				}
			}

			private String getQuery() {
				return URLEncodedUtils.format(queries, "UTF-8");
			}
		});
	}

	private void apiBuffer(HttpMethod method, String path,
			List<NameValuePair> queries, final HttpResponseListener listener) {
		api(method, path, queries, new Proc2<HttpClient, HttpUriRequest>() {
			@Override
			public void run(HttpClient httpClient, HttpUriRequest httpRequest) {
				try {
					final HttpResponse httpResponse = httpClient
							.execute(httpRequest);
					final StatusLine status = httpResponse.getStatusLine();
					final int statusCode = status.getStatusCode();
					final HttpEntity httpEntity = httpResponse.getEntity();
					final String str = EntityUtils.toString(httpEntity);
					httpEntity.consumeContent();
					postToMainThread(new Runnable() {
						@Override
						public void run() {
							switch (statusCode) {
							case HttpStatus.SC_OK:
								listener.onSuccess(str);
								break;
							case HttpStatus.SC_UNAUTHORIZED:
								listener.onUnauthorized();
								break;
							default:
								listener.onHttpFailure(status, str);
							}
						}
					});
				} catch (ClientProtocolException e) {
					postExceptionToMainThread(e);
				} catch (IOException e) {
					postExceptionToMainThread(e);
				} finally {
					httpClient.getConnectionManager().shutdown();
				}
			}

			private void postExceptionToMainThread(final Exception e) {
				postToMainThread(new Runnable() {
					@Override
					public void run() {
						listener.onFailure(e);
					}
				});
			}

			private void postToMainThread(final Runnable r) {
				new Handler(Looper.getMainLooper()).post(new Runnable() {
					@Override
					public void run() {
						listener.preExec();
						r.run();
						listener.postExec();
					}
				});
			}
		});
	}

	private void apiBuffer(HttpMethod method, String path,
			HttpResponseListener listener) {
		apiBuffer(method, path, new ArrayList<NameValuePair>(), listener);
	}

	@Override
	public void getLectures(HttpResponseListener listener) {
		apiBuffer(HttpMethod.GET, "/lectures", listener);
	}

	@Override
	public void getLectures(List<Lecture> lectures,
			HttpResponseListener listener) {
		List<NameValuePair> query = new ArrayList<NameValuePair>();
		for (Lecture l : lectures) {
			query.add(new BasicNameValuePair("ids[]", String.valueOf(l.getId())));
		}
		apiBuffer(HttpMethod.GET, "/lectures", query, listener);
	}

	@Override
	public void resetToken(String token, HttpResponseListener listener) {
		List<NameValuePair> query = new ArrayList<NameValuePair>();
		query.add(new BasicNameValuePair("auth_token", token));
		apiBuffer(HttpMethod.POST, "/users/reset_token.json", query, listener);
	}

	@Override
	public void getUsersProfile(String token, HttpResponseListener listener) {
		List<NameValuePair> query = new ArrayList<NameValuePair>();
		query.add(new BasicNameValuePair("auth_token", token));
		apiBuffer(HttpMethod.GET, "/users/profile.json", query, listener);
	}

	@Override
	public void putAchievement(String token, Achievement achievement,
			HttpResponseListener listener) {
		List<NameValuePair> query = new ArrayList<NameValuePair>();
		query.add(new BasicNameValuePair("auth_token", token));
		query.add(new BasicNameValuePair("achievement[result]", achievement
				.getResultString()));
		apiBuffer(HttpMethod.PUT, "/tasks/" + achievement.getTask().getId()
				+ "/achievement.json", query, listener);
	}

	@Override
	public void downloadApk(Apk apk, final HttpStreamListener listener) {
		api(HttpMethod.GET, "/apks/" + apk.getId() + "/download",
				new ArrayList<NameValuePair>(),
				new Proc2<HttpClient, HttpUriRequest>() {
					@Override
					public void run(HttpClient httpClient,
							HttpUriRequest httpRequest) {
						try {
							final HttpResponse httpResponse = httpClient
									.execute(httpRequest);
							final StatusLine status = httpResponse
									.getStatusLine();
							final int statusCode = status.getStatusCode();
							if (statusCode == HttpStatus.SC_OK) {
								final HttpEntity entity = httpResponse
										.getEntity();
								listener.onSuccess(entity);
								try {
									entity.consumeContent();
								} catch (IOException e) {
									// ignore
								}
								postToMainThread(new Runnable() {
									@Override
									public void run() {
										listener.postExec();
									}
								});
							} else {
								try {
									httpResponse.getEntity().consumeContent();
								} catch (IOException e) {
									// ignore
								}
								postToMainThread(new Runnable() {
									@Override
									public void run() {
										listener.onHttpFailure(status);
										listener.postExec();
									}
								});
							}
						} catch (ClientProtocolException e) {
							postExceptionToMainThread(e);
						} catch (IOException e) {
							postExceptionToMainThread(e);
						} finally {
							httpClient.getConnectionManager().shutdown();
						}
					}

					private void postExceptionToMainThread(final Exception e) {
						postToMainThread(new Runnable() {
							@Override
							public void run() {
								listener.onFailure(e);
								listener.postExec();
							}
						});
					}

					private void postToMainThread(final Runnable r) {
						new Handler(Looper.getMainLooper()).post(r);
					}
				});
	}
}
