package jp.co.epson.watch.plaWasabi.service.form;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import jp.co.epson.watch.plaWasabi.commons.WasabiRuntimeException;
import jp.co.epson.watch.plaWasabi.dto.formConfig.Action;
import jp.co.epson.watch.plaWasabi.dto.formConfig.ActionQuery;
import jp.co.epson.watch.plaWasabi.dto.formConfig.EntityMapping;
import jp.co.epson.watch.plaWasabi.dto.formConfig.ExcelForm;
import jp.co.epson.watch.plaWasabi.dto.formConfig.Form;
import jp.co.epson.watch.plaWasabi.dto.formConfig.FormConfig;
import jp.co.epson.watch.plaWasabi.dto.formConfig.PdfForm;
import jp.co.epson.watch.plaWasabi.dto.formConfig.Query;
import jp.co.epson.watch.plaWasabi.dto.formConfig.QueryParam;

import org.apache.commons.lang.StringUtils;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

/**
 * 入力フォームと D/B を連携するサービスクラス。
 * 
 * @author K.Ono
 * 
 */
public class FormService {

  private static final String CONFIG_PATH = "config";

  private static final String QUERY_PATH = "/form-config/database/queries/query";

  private static final String EM_PATH = "/form-config/database/entity-mappings/entity-mapping";

  private static final String FORM_PATH = "form-config/forms/form";

  private static final String EXCEL_FORM_PATH = "form-config/excelForms/excelForm";

  private static final String PDF_FORM_PATH = "form-config/pdfForms/pdfForm";

  private static final String ACTION_PATH = "form-config/actions/action";

  /**
   * フォーム固有名称を基に対応する設定ファイル読み込み、その情報をDTOに格納して取得する。
   * 
   * @param formName
   *          フォーム固有名称
   * @return フォーム固有の動作制御に関する設定情報
   */
  public FormConfig loadFormConfig(final String formName) {
    URL url = Thread.currentThread().getContextClassLoader().getResource(
        CONFIG_PATH + File.separator + formName + ".xml");
    InputStream is;
    try {
      is = url.openStream();
    } catch (IOException e) {
      throw new WasabiRuntimeException(e);
    }
    return this.loadFormConfig(is, formName);
  }

  /**
   * フォーム固有名称を基に対応する設定ファイル読み込み、その情報をDTOに格納して取得する。
   * 
   * @param is
   *          設定ファイルのInputStream
   * @param formName
   *          フォーム固有名称
   * @return フォーム固有の動作制御に関する設定情報
   */
  public FormConfig loadFormConfig(final InputStream is, final String formName) {
    FormConfig cfg = null;
    try {
      DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
      factory.setValidating(false);

      DocumentBuilder builder = factory.newDocumentBuilder();
      Document document = builder.parse(is);
      XPathFactory factory2 = XPathFactory.newInstance();
      XPath xpath = factory2.newXPath();

      cfg = new FormConfig();
      cfg.setId(formName);

      // queries
      cfg.setQueries(this.getQueryMap(xpath, document));

      // entity-mappings
      cfg.setEntityMappings(this.getEntityMappingMap(xpath, document));

      // forms
      cfg.setForms(this.getFormMap(xpath, document));

      // excels
      cfg.setExcelForms(this.getExcelMap(xpath, document));

      // pdfs
      cfg.setPdfForms(this.getPdfMap(xpath, document));

      // actions
      cfg.setActions(this.getActionMap(cfg, xpath, document));

    } catch (NumberFormatException e) {
      throw new WasabiRuntimeException(e);
    } catch (XPathExpressionException e) {
      // 不正なXPathの表記
      throw new WasabiRuntimeException(e);
    } catch (ParserConfigurationException e) {
      // ParserConfigurationException
      throw new WasabiRuntimeException(e);
    } catch (SAXException e) {
      // XMLの解析に失敗
      throw new WasabiRuntimeException(e);
    } catch (IOException e) {
      // XMLファイルの読み込みに失敗
      throw new WasabiRuntimeException(e);
    }

    return cfg;
  }

  private Map<String, Action> getActionMap(FormConfig conf, XPath xpath, Document document)
      throws XPathExpressionException {
    int count = Integer.parseInt(xpath.evaluate("count(" + ACTION_PATH + ")", document));
    HashMap<String, Action> res = new HashMap<String, Action>();
    for (int i = 0; i < count; i++) {
      Action a = new Action();
      a.setFormConfig(conf);
      a.setMethod(xpath.evaluate(ACTION_PATH + "[" + (i + 1) + "]/@method", document));
      a.setAplCd(xpath.evaluate(ACTION_PATH + "[" + (i + 1) + "]/aplCd", document));
      a.setAplName(xpath.evaluate(ACTION_PATH + "[" + (i + 1) + "]/aplName", document));

      String importTypeName = xpath
          .evaluate(ACTION_PATH + "[" + (i + 1) + "]/importType", document);
      if (StringUtils.isNotBlank(importTypeName)) {
        a.setImportType(Integer.parseInt(importTypeName));
      }

      String formName = xpath.evaluate(ACTION_PATH + "[" + (i + 1) + "]/form", document);
      if (StringUtils.isNotBlank(formName)) {
        a.setForm(conf.getForms().get(formName));
      }

      String deleteQueryName = xpath.evaluate(ACTION_PATH + "[" + (i + 1) + "]/deleteQuery",
          document);
      if (StringUtils.isNotBlank(deleteQueryName)) {
        a.setDeleteQuery(conf.getQueries().get(deleteQueryName));
      }

      String emName = xpath.evaluate(ACTION_PATH + "[" + (i + 1) + "]/entityMapping", document);
      if (StringUtils.isNotBlank(emName)) {
        a.setEntityMapping(conf.getEntityMappings().get(emName));
      }

      String excelName = xpath.evaluate(ACTION_PATH + "[" + (i + 1) + "]/excelForm", document);
      if (StringUtils.isNotBlank(excelName)) {
        a.setExcelForm(conf.getExcelForms().get(excelName));
      }

      String pdfName = xpath.evaluate(ACTION_PATH + "[" + (i + 1) + "]/pdfForm", document);
      if (StringUtils.isNotBlank(pdfName)) {
        a.setPdfForm(conf.getPdfForms().get(pdfName));
      }

      final String queryRootPath = ACTION_PATH + "[" + (i + 1) + "]/queries/query";
      int queryCnt = Integer.parseInt(xpath.evaluate("count(" + queryRootPath + ")", document));
      for (int q = 0; q < queryCnt; q++) {
        ActionQuery query = new ActionQuery();
        query.setAction(a);
        String queryPath = queryRootPath + "[" + (q + 1) + "]";
        String queryId = xpath.evaluate(queryPath + "/@refId", document);
        query.setId(queryId);
        query.setQuery(conf.getQueries().get(queryId));
        query.setPrimary(Boolean.parseBoolean(xpath.evaluate(queryPath + "/@primary", document)));
        query.setPrmConverter(xpath.evaluate(queryPath + "/@prmConverter", document));

        final String queryParamRootPath = queryPath + "/queryParams/queryParam";
        int entCnt = Integer
            .parseInt(xpath.evaluate("count(" + queryParamRootPath + ")", document));
        for (int j = 0; j < entCnt; j++) {
          QueryParam queryParam = new QueryParam();
          String queryParamPath = queryParamRootPath + "[" + (j + 1) + "]";
          queryParam.setId(xpath.evaluate(queryParamPath, document));
          queryParam.setCompare(xpath.evaluate(queryParamPath + "/@compare", document));
          query.getQueryParams().add(queryParam);
        }
        a.getQueryMap().put(queryId, query);
      }

      a.setForward(xpath.evaluate(ACTION_PATH + "[" + (i + 1) + "]/forward", document));

      res.put(a.getMethod(), a);
    }
    return res;
  }

  private Map<String, ExcelForm> getExcelMap(XPath xpath, Document document)
      throws XPathExpressionException {
    int count = Integer.parseInt(xpath.evaluate("count(" + EXCEL_FORM_PATH + ")", document));
    HashMap<String, ExcelForm> res = new HashMap<String, ExcelForm>();
    for (int i = 0; i < count; i++) {
      ExcelForm form = new ExcelForm();
      form.setId(xpath.evaluate(EXCEL_FORM_PATH + "[" + (i + 1) + "]/@id", document));

      final String entPath = EXCEL_FORM_PATH + "[" + (i + 1) + "]/fields/field";
      int entCnt = Integer.parseInt(xpath.evaluate("count(" + entPath + ")", document));
      for (int j = 0; j < entCnt; j++) {
        form.getFields().add(xpath.evaluate(entPath + "[" + (j + 1) + "]", document));
      }

      res.put(form.getId(), form);
    }
    return res;
  }

  private Map<String, PdfForm> getPdfMap(XPath xpath, Document document)
      throws XPathExpressionException {
    int count = Integer.parseInt(xpath.evaluate("count(" + PDF_FORM_PATH + ")", document));
    HashMap<String, PdfForm> res = new HashMap<String, PdfForm>();
    for (int i = 0; i < count; i++) {
      PdfForm form = new PdfForm();
      form.setId(xpath.evaluate(PDF_FORM_PATH + "[" + (i + 1) + "]/@id", document));
      form.setFormFile(xpath.evaluate(PDF_FORM_PATH + "[" + (i + 1) + "]/formFile", document));

      final String entPath = PDF_FORM_PATH + "[" + (i + 1) + "]/fileName/param";
      int entCnt = Integer.parseInt(xpath.evaluate("count(" + entPath + ")", document));
      for (int j = 0; j < entCnt; j++) {
        form.getFileNameParams().add(xpath.evaluate(entPath + "[" + (j + 1) + "]", document));
      }

      res.put(form.getId(), form);
    }
    return res;
  }

  private Map<String, Form> getFormMap(XPath xpath, Document document)
      throws XPathExpressionException {
    int count = Integer.parseInt(xpath.evaluate("count(" + FORM_PATH + ")", document));
    HashMap<String, Form> res = new HashMap<String, Form>();
    for (int i = 0; i < count; i++) {
      Form form = new Form();
      form.setId(xpath.evaluate(FORM_PATH + "[" + (i + 1) + "]/@id", document));
      form.setXslFile(xpath.evaluate(FORM_PATH + "[" + (i + 1) + "]/xslFile", document));
      form.setHtmliFile(xpath.evaluate(FORM_PATH + "[" + (i + 1) + "]/htmliFile", document));
      form.setHtmlrFile(xpath.evaluate(FORM_PATH + "[" + (i + 1) + "]/htmlrFile", document));

      final String ersPath = FORM_PATH + "[" + (i + 1) + "]/eraseFields/eraseField";
      int ersCnt = Integer.parseInt(xpath.evaluate("count(" + ersPath + ")", document));
      for (int j = 0; j < ersCnt; j++) {
        form.getEraseFields().add(xpath.evaluate(ersPath + "[" + (j + 1) + "]", document));
      }

      final String entPath = FORM_PATH + "[" + (i + 1) + "]/labelFields/labelField";
      int entCnt = Integer.parseInt(xpath.evaluate("count(" + entPath + ")", document));
      for (int j = 0; j < entCnt; j++) {
        form.getLabelFields().add(xpath.evaluate(entPath + "[" + (j + 1) + "]", document));
      }

      res.put(form.getId(), form);
    }
    return res;
  }

  private Map<String, EntityMapping> getEntityMappingMap(XPath xpath, Document document)
      throws XPathExpressionException {
    int count = Integer.parseInt(xpath.evaluate("count(" + EM_PATH + ")", document));
    HashMap<String, EntityMapping> res = new HashMap<String, EntityMapping>();
    for (int i = 0; i < count; i++) {
      EntityMapping em = new EntityMapping();
      em.setId(xpath.evaluate(EM_PATH + "[" + (i + 1) + "]/@id", document));

      final String entPath = EM_PATH + "[" + (i + 1) + "]/entity";
      int entCnt = Integer.parseInt(xpath.evaluate("count(" + entPath + ")", document));
      for (int j = 0; j < entCnt; j++) {
        em.getEntities().add(xpath.evaluate(entPath + "[" + (j + 1) + "]", document));
      }

      res.put(em.getId(), em);
    }
    return res;
  }

  private Map<String, Query> getQueryMap(XPath xpath, Document document)
      throws XPathExpressionException {

    int count = Integer.parseInt(xpath.evaluate("count(" + QUERY_PATH + ")", document));
    HashMap<String, Query> res = new HashMap<String, Query>();
    for (int i = 0; i < count; i++) {
      String id = xpath.evaluate(QUERY_PATH + "[" + (i + 1) + "]/@id", document);
      String query = xpath.evaluate(QUERY_PATH + "[" + (i + 1) + "]", document);
      Query q = new Query();
      q.setId(id);
      q.setQuery(query);
      res.put(id, q);
    }
    return res;
  }

}
