import React, { Component } from 'react';
import { findDOMNode } from 'react-dom';
import Tooltip from 'react-tooltip';

import Window from './Components/Window';
import WindowButtons from './Components/WindowButtons';
import WindowButton from './Components/WindowButton';
import WindowContent from './Components/WindowContent';
import Titlebar from './Components/Titlebar';
import Tabs from './Components/Tabs';
import { TabContainer, Tab, TabIcon, TabTitle, TabCloseButton } from './Components/Tab';
import TabButton from './Components/TabButton';
import TabContent from './Components/TabContent';
import Toolbar from './Components/Toolbar';
import ToolbarButton from './Components/ToolbarButton';
import ToolbarDivider from './Components/ToolbarDivider';
import { ToolbarTextBoxWrapper, ToolbarTextBox } from './Components/ToolbarTextBox';
import ContentWrapper from './Components/ContentWrapper';

import WindowMinimizeIcon from './Resources/windows/minimize.svg';
import WindowMaximizeIcon from './Resources/windows/maximize.svg';
import WindowCloseIcon from './Resources/windows/close.svg';

import BackIcon from './Resources/arrow_back.svg';
import ForwardIcon from './Resources/arrow_forward.svg';
import BackInActiveIcon from './Resources/arrow_back_inactive.svg';
import ForwardInActiveIcon from './Resources/arrow_forward_inactive.svg';
import ReloadIcon from './Resources/reload.svg';
import HomeIcon from './Resources/home.svg';
import InfomationIcon from './Resources/info.svg';
import StarIcon from './Resources/star.svg';
import StarFilledIcon from './Resources/star-filled.svg';
import AccountIcon from './Resources/account.svg';
import ShieldIcon from './Resources/shield.svg';
import FeedbackIcon from './Resources/feedback.svg';
import MoreIcon from './Resources/more.svg';

import PublicIcon from './Resources/public.svg';
import PublicInActiveIcon from './Resources/public_inactive.svg';
import AddIcon from './Resources/add.svg';
import CloseIcon from './Resources/close.svg';

import isURL from './Utils/isURL';

const { remote, ipcRenderer, shell } = window.require('electron');
const systemPreferences = remote.systemPreferences;
const Menu = remote.Menu;
const MenuItem = remote.MenuItem;
const dialog = remote.dialog;

const platform = window.require('electron-platform');
const process = window.require('process');

const Config = window.require('electron-store');
const config = new Config({
	defaults: {
		design: {
			homeButton: false,
			darkTheme: false,
			theme: 'default'
		},
		homePage: {
			defaultPage: 'my://newtab',
			defaultEngine: 'Google',
			searchEngines: [
				{
					name: 'Google',
					url: 'https://www.google.com/search?q=%s'
				},
				{
					name: 'Bing',
					url: 'https://www.bing.com/search?q=%s'
				},
				{
					name: 'Yahoo! Japan',
					url: 'https://search.yahoo.co.jp/search?p=%s'
				},
				{
					name: 'goo',
					url: 'https://search.goo.ne.jp/web.jsp?MT=%s'
				},
				{
					name: 'Google Translate',
					url: 'https://translate.google.com/?text=%s'
				},
				{
					name: 'Youtube',
					url: 'https://www.youtube.com/results?search_query=%s'
				},
				{
					name: 'Twitter',
					url: 'https://www.twitter.com/search?q=%s'
				},
				{
					name: 'GitHub',
					url: 'https://github.com/search?q=%s'
				}
			]
		},
		adBlocker: true,
		window: {
			isCustomTitlebar: true,
			isMaximized: false,
			bounds: {
				width: 1100,
				height: 680
			}
		}
	},
});

class BrowserView extends Component {
	constructor(props) {
		super(props);
		this.state = {
			barText: '',
			findText: '',
			previousText: '',
			resultString: '',
			activeMatch: 0,
			isLoading: false,
			canGoBack: false,
			canGoForward: false,
			isBookmarked: false,
			isShowing: true,
		};
	}

	componentDidMount() {
		/*
		let webView = this.refs.webView;
		let props = this.props;

		webView.addEventListener('did-finish-load', (e) => {
			document.title = webView.getTitle();
			this.setText(webView.getURL());
			this.props.updateTab(webView.getTitle(), PublicIcon, webView.getURL(), this.props.index);
			this.setState({ barText: webView.getURL() });
		}, false);

		webView.addEventListener('page-title-updated', (e) => {
			document.title = webView.getTitle();
			this.setText(webView.getURL());
			this.props.updateTab(webView.getTitle(), PublicIcon, webView.getURL(), this.props.index);
			this.setState({ barText: webView.getURL() });
		}, false);

		webView.addEventListener('page-favicon-updated', (e) => {
			if (e.favicons.length > 0) {
				this.props.updateTab(webView.getTitle(), e.favicons[0], webView.getURL(), this.props.index);
			} else {
				this.props.updateTab(webView.getTitle(), PublicIcon, webView.getURL(), this.props.index);
			}
		});

		webView.addEventListener('load-commit', (e) => {
			if (e.isMainFrame) {
				document.title = webView.getTitle();
				this.setText(webView.getURL());
				this.props.updateTab(webView.getTitle(), PublicIcon, webView.getURL(), this.props.index);
				this.setState({ barText: webView.getURL() });
			}
		});

		webView.addEventListener('found-in-page', (e, result) => {
			if (result.activeMatchOrdinal) {
				this.setState({
					activeMatch: result.activeMatchOrdinal,
					resultString: `${this.state.activeMatch}/${result.matches}`
				});
			}

			if (result.finalUpdate) {
				this.setState({ resultString: `${this.state.activeMatch}/${result.matches}` });
			}
		});

		webView.addEventListener('new-window', (e, url) => {
			this.props.addTab('新しいタブ', '', e.url);
		});
		*/

		ipcRenderer.on(`browserview-start-loading-${this.props.windowId}`, (e, args) => {
			if (args.id == this.props.index) {
				this.setState({ isLoading: true });
			}
		});

		ipcRenderer.on(`browserview-stop-loading-${this.props.windowId}`, (e, args) => {
			if (args.id == this.props.index) {
				this.setState({ isLoading: false });
			}
		});

		ipcRenderer.on(`browserview-load-${this.props.windowId}`, (e, args) => {
			if (args.id == this.props.index) {
				this.props.updateTab();
				this.setState({ isBookmarked: args.isBookmarked });
				this.setText(args.url);
			}
		});

		ipcRenderer.on(`update-navigation-state-${this.props.windowId}`, (e, args) => {
			if (args.id == this.props.index) {
				this.setState({ canGoBack: args.canGoBack, canGoForward: args.canGoForward });
			}
		});

		// console.log(webView);
	}

	handleKeyDown = (e) => {
		if (e.key != 'Enter' || this.state.barText.length == 0 || this.state.barText == '') return;

		if (isURL(this.state.barText)) {
			const pattern = /^(file:\/\/\S.*)\S*$/;

			if (pattern.test(this.state.barText)) {
				ipcRenderer.send(`browserview-loadFile-${this.props.windowId}`, { id: this.props.index, url: this.state.barText.replace('file://', '') });
			} else {
				ipcRenderer.send(`browserview-loadURL-${this.props.windowId}`, { id: this.props.index, url: this.state.barText });
			}
		} else {
			ipcRenderer.send(`browserview-loadURL-${this.props.windowId}`, { id: this.props.index, url: this.getSearchEngine().url.replace('%s', this.state.barText) });
		}
	}

	handleContextMenu = (e) => {
		const menu = Menu.buildFromTemplate([
			{
				label: '元に戻す',
				accelerator: 'CmdOrCtrl+Z',
				role: 'undo'
			},
			{
				label: 'やり直す',
				accelerator: 'CmdOrCtrl+Y',
				role: 'redo'
			},
			{ type: 'separator' },
			{
				label: '切り取り',
				accelerator: 'CmdOrCtrl+X',
				role: 'cut'
			},
			{
				label: 'コピー',
				accelerator: 'CmdOrCtrl+C',
				role: 'copy'
			},
			{
				label: '貼り付け',
				accelerator: 'CmdOrCtrl+V',
				role: 'paste'
			},
			{ type: 'separator' },
			{
				label: 'すべて選択',
				accelerator: 'CmdOrCtrl+A',
				role: 'selectAll'
			}
		]);
		menu.popup();
	}

	getSearchEngine = () => {
		for (var i = 0; i < config.get('homePage.searchEngines').length; i++) {
			if (config.get('homePage.searchEngines')[i].name == config.get('homePage.defaultEngine')) {
				return config.get('homePage.searchEngines')[i];
			}
		}
	}

	setText = (text) => {
		this.setState({ barText: text });
	}

	goBack = () => {
		ipcRenderer.send(`browserview-goBack-${this.props.windowId}`, { id: this.props.index });
	}

	goForward = () => {
		ipcRenderer.send(`browserview-goForward-${this.props.windowId}`, { id: this.props.index });
	}

	reload = () => {
		if (!this.state.isLoading) {
			ipcRenderer.send(`browserview-reload-${this.props.windowId}`, { id: this.props.index });
		} else {
			ipcRenderer.send(`browserview-stop-${this.props.windowId}`, { id: this.props.index });
		}
	}

	goHome = () => {
		ipcRenderer.send(`browserview-goHome-${this.props.windowId}`, { id: this.props.index });
	}

	bookMark = () => {
		const node = findDOMNode(this.refs.markTooltip);
		Tooltip.show(node);
		setTimeout(() => {
			Tooltip.hide(node);
		}, 1250);
		if (this.state.isBookmarked)
			ipcRenderer.send(`data-bookmark-remove-${this.props.windowId}`, { id: this.props.index, isPrivate: this.props.windowId.startsWith('private') });
		else
			ipcRenderer.send(`data-bookmark-add-${this.props.windowId}`, { id: this.props.index, isPrivate: this.props.windowId.startsWith('private') });
	}

	userMenu = async () => {
		const menu = Menu.buildFromTemplate([
			{
				label: (!this.props.windowId.startsWith('private') ? `${process.env.USERNAME} でログイン中` : 'プライベートモード'),
				enabled: false,
			},
			{ type: 'separator' },
			{
				label: '閉じる',
				click: () => { remote.getCurrentWindow().close(); }
			}
		]);
		menu.popup({
			x: remote.getCurrentWindow().getSize()[0] - 57,
			y: 65
		});
	}

	moreMenu = () => {
		const menu = Menu.buildFromTemplate([
			{
				label: '新しいタブ',
				accelerator: 'CmdOrCtrl+T',
				click: () => { this.props.addTab('https://www.google.co.jp/'); }
			},
			{
				label: '新しいウィンドウ',
				accelerator: 'CmdOrCtrl+N',
				click: () => { ipcRenderer.send(`window-add`, { isPrivate: false }); }
			},
			{
				label: 'プライベート ウィンドウを開く',
				accelerator: 'CmdOrCtrl+Shift+N',
				click: () => { ipcRenderer.send(`window-add`, { isPrivate: true }); }
			},
			{ type: 'separator' },
			{
				label: '履歴',
				click: () => { this.props.addTab('my://history'); }
			},
			{
				label: 'ダウンロード',
				click: () => { this.props.addTab('my://history'); }
			},
			{
				label: 'ブックマーク',
				click: () => { this.props.addTab('my://bookmark'); }
			},
			{ type: 'separator' },
			{
				label: '設定',
				click: () => { this.props.addTab('my://settings'); }
			},
			{
				label: 'ヘルプ',
				click: () => { this.props.addTab('my://help'); }
			},
			{ type: 'separator' },
			{
				label: '閉じる',
				click: () => { this.props.closeWindow(); }
			}
		]);
		menu.popup({
			x: remote.getCurrentWindow().getSize()[0] - 24,
			y: 65
		});
	}

	render() {
		return (
			<ContentWrapper>
				<Toolbar>
					<ToolbarButton isShowing={true} isRight={false} isMarginLeft={true} isEnabled={this.state.canGoBack} src={this.state.canGoBack ? BackIcon : BackInActiveIcon} size={24} title="前のページに戻る" onClick={() => { this.goBack(); }} />
					<ToolbarButton isShowing={true} isRight={false} isMarginLeft={false} isEnabled={this.state.canGoForward} src={this.state.canGoForward ? ForwardIcon : ForwardInActiveIcon} size={24} title="次のページに進む" onClick={() => { this.goForward(); }} />
					<ToolbarButton isShowing={true} isRight={false} isMarginLeft={false} isEnabled={true} src={!this.state.isLoading ? ReloadIcon : CloseIcon} size={24} title={!this.state.isLoading ? '再読み込み' : '読み込み中止'} onClick={() => { this.reload(); }} />
					<ToolbarButton isShowing={config.get('design.homeButton')} isRight={false} isMarginLeft={false} isEnabled={true} src={HomeIcon} size={24} title="ホームページに移動" onClick={() => { this.goHome(); }} />
					<ToolbarTextBoxWrapper buttonCount={config.get('design.homeButton') ? 6 : 5}>
						<ToolbarButton isShowing={true} isRight={false} isMarginLeft={true} isEnabled={true} src={InfomationIcon} size={24} title="このページの情報" />
						<ToolbarTextBox value={this.state.barText} onChange={(e) => { this.setState({ barText: e.target.value }); }} onKeyDown={this.handleKeyDown} onContextMenu={this.handleContextMenu} />
						<ToolbarButton data-tip={this.state.isBookmarked ? `${this.props.windowId.startsWith('private') ? 'プライベート ' : ''}ブックマークから削除しました。` : `${this.props.windowId.startsWith('private') ? 'プライベート ' : ''}ブックマークに追加しました。`} ref="markTooltip" isShowing={true} isRight={true} isMarginLeft={true} isEnabled={true} src={this.state.isBookmarked ? StarFilledIcon : StarIcon} size={24} title={this.state.isBookmarked ? 'ブックマークから削除' : 'ブックマークに追加'} onClick={() => { this.bookMark(); }}>
							<Tooltip effect="solid" place="left" event="no-event" />
						</ToolbarButton>
					</ToolbarTextBoxWrapper>
					<ToolbarButton isShowing={true} isRight={true} isMarginLeft={true} isEnabled={true} src={FeedbackIcon} size={24} title="フィードバックの送信" />
					<ToolbarDivider />
					<ToolbarButton isShowing={true} isRight={true} isMarginLeft={true} isEnabled={true} src={this.props.windowId.startsWith('private') ? ShieldIcon : AccountIcon} size={24} title={this.props.windowId.startsWith('private') ? 'プライベートモード' : process.env.USERNAME} onClick={() => { this.userMenu(); }} />
					<ToolbarButton isShowing={true} isRight={true} isMarginLeft={false} isEnabled={true} src={MoreIcon} size={24} title="メニュー" onClick={() => { this.moreMenu(); }} />
				</Toolbar>
			</ContentWrapper>
		);
	}
}

class App extends Component {
	constructor(props) {
		super(props);

		this.state = {
			tabs: [],
			current: 0
		};

		ipcRenderer.on(`browserview-get-${this.props.match.params.windowId}`, (e, args) => {
			this.setState({ tabs: args.views });
		});

		ipcRenderer.on(`browserview-set-${this.props.match.params.windowId}`, (e, args) => {
			this.setState({ current: args.id });
		});
	}

	componentDidMount = () => {
		this.addTab();
	}

	handleContextMenu = (i) => {
		let t = this;
		const menu = Menu.buildFromTemplate([
			{
				label: '新しいタブ',
				accelerator: 'CmdOrCtrl+T',
				click: function () { t.addTab('https://www.google.co.jp/'); }
			},
			{
				label: 'タブを閉じる',
				accelerator: 'CmdOrCtrl+W',
				click: function () { t.removeTab(i); }
			},
			{ type: 'separator' },
			{
				type: 'checkbox',
				label: 'タブを固定',
				checked: t.state.tabs[i].fixed,
				click: function () {
					var newTabs = t.state.tabs.concat();
					newTabs[i].fixed = !t.state.tabs[i].fixed;
					t.setState({ tabs: newTabs });
				}
			}
		]);
		menu.popup(remote.getCurrentWindow());
	}

	getTabs = () => {
		ipcRenderer.send(`browserview-get-${this.props.match.params.windowId}`, {});
	}

	addTab = (url = config.get('homePage.defaultPage')) => {
		ipcRenderer.send(`browserview-add-${this.props.match.params.windowId}`, { url, isActive: true });
		ipcRenderer.send(`browserview-get-${this.props.match.params.windowId}`, {});
		this.setState({ current: this.state.tabs.length });
	}

	removeTab = (i) => {
		ipcRenderer.send(`browserview-remove-${this.props.match.params.windowId}`, { id: i });
		ipcRenderer.send(`browserview-get-${this.props.match.params.windowId}`, {});

		if ((this.state.tabs.length - 1) > 0) {
			const id = this.state.current--;
			ipcRenderer.send(`browserview-select-${this.props.match.params.windowId}`, { id: id });
		} else {
			remote.getCurrentWindow().close();
		}
	}

	updateTab = () => {
		ipcRenderer.send(`browserview-get-${this.props.match.params.windowId}`, {});
	}

	getForegroundColor = (hexColor) => {
		var r = parseInt(hexColor.substr(1, 2), 16);
		var g = parseInt(hexColor.substr(3, 2), 16);
		var b = parseInt(hexColor.substr(5, 2), 16);

		return ((((r * 299) + (g * 587) + (b * 114)) / 1000) < 128) ? '#ffffff' : '#000000';
	}

	closeWindow = () => {
		if (this.state.tabs.length > 1) {
			const dlg = dialog.showMessageBox(remote.getCurrentWindow(), {
				type: 'info',
				buttons: ['Yes', 'No'],
				title: 'アプリ終了確認',
				message: `${this.state.tabs.length}個のタブを閉じようとしています。`,
				detail: `${this.state.tabs.length}個のタブを閉じてアプリを終了しようとしています。\n複数のタブを閉じてアプリを終了してもよろしいでしょうか？`,
				defaultId: 0,
				cancelId: 1
			});
			if (dlg === 0) {
				remote.getCurrentWindow().close();
			}
		} else {
			remote.getCurrentWindow().close();
		}
	}

	render() {
		return (
			<Window isCustomTitlebar={config.get('window.isCustomTitlebar')}>
				<Titlebar color={platform.isWin32 || platform.isDarwin ? `#${systemPreferences.getAccentColor()}` : '#353535'}>
					<Tabs isCustomTitlebar={config.get('window.isCustomTitlebar')}>
						<TabContainer>
							{this.state.tabs.map((tab, i) => {
								return (
									<Tab isActive={tab.id == this.state.current} isFixed={tab.fixed} onClick={() => { ipcRenderer.send(`browserview-select-${this.props.match.params.windowId}`, { id: tab.id }); }} onContextMenu={this.handleContextMenu.bind(this, tab.id)}>
										<TabIcon src={tab.icon != undefined ? tab.icon : PublicIcon} width={18} height={18} />
										<TabTitle isFixed={tab.fixed} title={tab.title}>{tab.title}</TabTitle>
										<TabCloseButton isActive={tab.id == this.state.current} isFixed={tab.fixed} isRight={true} src={CloseIcon} size={14} title="このタブを閉じる" onClick={() => { this.removeTab(tab.id); }} />
									</Tab>
								);
							})}
						</TabContainer>
						<TabButton isRight={true} src={AddIcon} size={24} title="新しいタブを開く" onClick={() => { this.addTab(); }} />
					</Tabs>
					<WindowButtons isCustomTitlebar={config.get('window.isCustomTitlebar')}>
						<WindowButton isClose={false} title="最小化" onClick={() => { remote.getCurrentWindow().minimize(); }}>
							<svg name="TitleBarMinimize" width="12" height="12" viewBox="0 0 12 12">
								<rect fill={this.getForegroundColor(platform.isWin32 || platform.isDarwin ? `#${systemPreferences.getAccentColor()}` : '#353535')} width="10" height="1" x="1" y="6" />
							</svg>
						</WindowButton>
						<WindowButton isClose={false} title={remote.getCurrentWindow().isMaximized() ? '元に戻す (縮小)' : '最大化'} onClick={() => { remote.getCurrentWindow().isMaximized() ? remote.getCurrentWindow().unmaximize() : remote.getCurrentWindow().maximize(); }}>
							<svg name="TitleBarMaximize" width="12" height="12" viewBox="0 0 12 12">
								<rect stroke={this.getForegroundColor(platform.isWin32 || platform.isDarwin ? `#${systemPreferences.getAccentColor()}` : '#353535')} width="9" height="9" x="1.5" y="1.5" fill="none" />
							</svg>
						</WindowButton>
						<WindowButton isClose={true} title="閉じる" onClick={() => { this.closeWindow(); }}>
							<svg name="TitleBarClose" width="12" height="12" viewBox="0 0 12 12">
								<polygon fill={this.getForegroundColor(platform.isWin32 || platform.isDarwin ? `#${systemPreferences.getAccentColor()}` : '#353535')} fill-rule="evenodd" points="11 1.576 6.583 6 11 10.424 10.424 11 6 6.583 1.576 11 1 10.424 5.417 6 1 1.576 1.576 1 6 5.417 10.424 1" />
							</svg>
						</WindowButton>
					</WindowButtons>
				</Titlebar>
				<WindowContent>
					{this.state.tabs.map((tab, i) => {
						return (
							<TabContent isActive={tab.id == this.state.current}>
								<BrowserView key={i} windowId={this.props.match.params.windowId} index={tab.id} url={tab.url}
									closeWindow={() => { this.closeWindow(); }}
									addTab={(url) => { this.addTab(url); }}
									updateTab={() => { this.updateTab(); }} />
							</TabContent>
						);
					})}
				</WindowContent>
			</Window>
		);
	}
}

export default App;