﻿/*
  Copyright 2007 Takashi Oguma

  This file is part of SendToCMD.

  SendToCMD is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  SendToCMD is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with SendToCMD; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include "Profile.h"

#include "Path.h"


namespace bearmini
{
    ///
    ///  専用 INI ファイルから文字列を取得します。
    ///  sectionName で指定されたセクションから、keyName で指定されたキーを持つ行を探し、
    ///  対応する値を取得します。
    ///  sectionName で指定されたセクションが見つからないか、keyName で指定されたキーが見つからない場合、
    ///  defaultValue で指定された値が返ります。
    ///
    ///  @param[in] sectionName     INI ファイル内のセクション名
    ///  @param[in] keyName         パラメータのキー
    ///  @param[in] defaultValue    見つからなかった時のデフォルト値
    ///
    ///  @return    取得した文字列
    ///
    std::wstring Profile::GetString(const std::wstring& sectionName, const std::wstring& keyName, const std::wstring& defaultValue)
    {
        // INI ファイルの名前を得る
        std::wstring iniFileName = getPrivateProfileName();

        // 文字列を受け取るバッファを初期化
        const int MAX_PARAM_LEN = 4096;
        wchar_t buf[MAX_PARAM_LEN] = { 0 };

        ::GetPrivateProfileStringW(sectionName.c_str(), keyName.c_str(), defaultValue.c_str(), buf, MAX_PARAM_LEN, iniFileName.c_str());

        return buf;
    }


    ///
    ///  専用 INI ファイルから整数値を取得します。
    ///  sectionName で指定されたセクションから、keyName で指定されたキーを持つ行を探し、
    ///  対応する値を取得します。
    ///  sectionName で指定されたセクションが見つからないか、keyName で指定されたキーが見つからない場合、
    ///  defaultValue で指定された値が返ります。
    ///
    ///  @param[in] sectionName     INI ファイル内のセクション名
    ///  @param[in] keyName         パラメータのキー
    ///  @param[in] defaultValue    見つからなかった時のデフォルト値
    ///
    ///  @return    取得した数値
    ///
    unsigned int Profile::GetUInt(const std::wstring& sectionName, const std::wstring& keyName, const unsigned int defaultValue)
    {
        // INI ファイルの名前を得る
        std::wstring iniFileName = getPrivateProfileName();

        return ::GetPrivateProfileIntW(sectionName.c_str(), keyName.c_str(), defaultValue, iniFileName.c_str());
    }


    ///
    ///  専用 INI ファイルから、sectionName で指定されたセクション内のキー名の一覧を取得します。
    ///
    ///  @param[in] sectionName     INI ファイル内のセクション名
    ///
    ///  @return    取得したキーの一覧
    ///
    wstring_vector Profile::GetKeyNameList(const std::wstring& sectionName)
    {
        wstring_vector result;

        // INI ファイルの名前を得る
        std::wstring iniFileName = getPrivateProfileName();

        const int MAX_LENGTH_OF_KEY_NAME_LIST = 32 * 1024;  // 32 KB
        wchar_t buf[MAX_LENGTH_OF_KEY_NAME_LIST] = { 0 };

        ::DWORD length = ::GetPrivateProfileStringW(sectionName.c_str(), 0, 0, buf, MAX_LENGTH_OF_KEY_NAME_LIST, iniFileName.c_str());

        ::DWORD offset = 0;
        while (offset < length)
        {
            if (*(buf + offset) == L'\0')
            {
                break;
            }
            std::wstring ws = buf + offset;
            result.push_back(ws);
            offset += static_cast<::DWORD>(ws.length() + 1);
        }
        return result;
    }


    ///
    ///  専用 INI ファイルのファイル名を取得します。
    ///
    std::wstring Profile::getPrivateProfileName()
    {
        // 実行ファイル名を得て、その拡張子を .ini に変更する
        wchar_t exename[BUFSIZ];
        ::GetModuleFileNameW(0, exename, BUFSIZ);

        return Path::ChangeExtension(exename, L".ini");
    }

}
