/**
 * 
 */
package jp.mydns.masahase.abaqus.weldline;

import jp.mydns.masahase.InherentStrain.ElementType;
import jp.mydns.masahase.InherentStrain.InherentStrainType;
import jp.mydns.masahase.InherentStrain.NodeType;
import jp.mydns.masahase.sdes.common.FemFunc;

/**
 * 曲線の溶接線. 円柱座標系で断面に投影. 形状関数による補間あり.
 * 
 * @author Masahiro
 */
public class CurveLine implements IWeldLine {
	/**
	 * コンストラクタ
	 * 
	 * @param pedata
	 *            固有ひずみ情報
	 * @param StartPos
	 *            溶接開始点(曲線の時計回りに見た場合の始端）
	 * @param EndPos
	 *            溶接終了点(曲線の時計回りに見た場合の終端）
	 * @param CenterPos
	 *            曲線の中心
	 * @param angle
	 * 
	 */
	public CurveLine(InherentStrainType pedata, double[] StartPos,
			double[] EndPos, double[] CenterPos, double angle) {
		double[] tmp = new double[3];
		double vec_size;
		vec_a = new double[3];
		for (int i = 0; i < 3; i++) {
			vec_a[i] = StartPos[i] - CenterPos[i];
			tmp[i] = EndPos[i] - CenterPos[i];
		}
		vec_size = 0;
		for (double t : vec_a) {
			vec_size += t * t;
		}
		vec_size = Math.sqrt(vec_size);
		for (int i = 0; i < vec_a.length; i++) {
			vec_a[i] /= vec_size;
		}
		vec_size = 0;
		for (double t : tmp) {
			vec_size += t * t;
		}
		vec_size = Math.sqrt(vec_size);
		for (int i = 0; i < tmp.length; i++) {
			tmp[i] /= vec_size;
		}
		vec_b = makeVectorProduct(vec_a, tmp);
		vec_c = makeVectorProduct(vec_a, vec_b);
		vec_size = 0;
		for (double t : vec_b) {
			vec_size += t * t;
		}
		vec_size = Math.sqrt(vec_size);
		for (int i = 0; i < tmp.length; i++) {
			vec_b[i] /= vec_size;
		}
		vec_size = 0;
		for (double t : vec_c) {
			vec_size += t * t;
		}
		vec_size = Math.sqrt(vec_size);
		for (int i = 0; i < tmp.length; i++) {
			vec_c[i] /= vec_size;
		}
		this.StartPos = StartPos.clone();
		this.CenterPos = CenterPos.clone();
		tmp = changeCoordinate(EndPos);
		EndAngle = -Math.atan2(tmp[1], tmp[0]);
		this.pedata = pedata;
		this.angle = angle / 180.0 * Math.PI;
		R = 0;
		for (int i = 0; i < StartPos.length; i++) {
			R += (StartPos[i] - CenterPos[i]) * (StartPos[i] - CenterPos[i]);
		}
		R = Math.sqrt(R);

	}

	private double[] makeVectorProduct(double[] a, double[] b) {
		if (a.length != 3 || b.length != 3) {
			return null;
		}
		double[] ret = new double[3];
		ret[0] = a[1] * b[2] - a[2] * b[1];
		ret[1] = a[2] * b[0] - a[0] * b[2];
		ret[2] = a[0] * b[1] - a[1] * b[0];
		return ret;
	}

	InherentStrainType pedata;
	double[] vec_a, vec_b, vec_c;
	double[] StartPos, CenterPos;
	double EndAngle, angle, R;

	protected double[] changeCoordinate(double[] target) {
		double[] tmp = new double[3];
		for (int i = 0; i < 3; i++) {
			tmp[i] = target[i] - CenterPos[i];
		}
		return changeCoordinate(tmp, vec_a, vec_c, vec_b);
	}

	private double[] changeCoordinate(double[] target, double[] basis_a,
			double[] basis_b, double[] basis_c) {
		double[] ret = new double[3];
		ret[0] = basis_a[0] * target[0] + basis_a[1] * target[1] + basis_a[2]
				* target[2];
		ret[1] = basis_b[0] * target[0] + basis_b[1] * target[1] + basis_b[2]
				* target[2];
		ret[2] = basis_c[0] * target[0] + basis_c[1] * target[1] + basis_c[2]
				* target[2];
		return ret;
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see jp.mydns.masahase.abaqus.weldline.IWeldLine#getPE(double, double,
	 * double)
	 */
	@Override
	public double[] getPE(double x, double y, double z) {
		double[] ret = new double[6];
		double[] tmp = { x, y, z };
		double[] coord = changeCoordinate(tmp);
		double theta = Math.atan2(coord[1], coord[0]);
		for (int i = 0; i < ret.length; i++) {
			ret[i] = 0.0;
		}
		if (-theta <= EndAngle && -theta >= 0) {
			double[] coord0 = new double[2];
			double tmp_x, tmp_y;
			tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - R;
			tmp_y = coord[2];
			coord0[0] = tmp_x * Math.cos(angle) + tmp_y * Math.sin(angle);
			coord0[1] = -tmp_x * Math.sin(angle) + tmp_y * Math.cos(angle);
			getPE_(ret, coord0);
			assert FemFunc.calcEstimateStrain(ret) < 0.1;
			ret = changeECoordinate(ret, coord);
			assert FemFunc.calcEstimateStrain(ret) < 0.1;
		}
		return ret;
	}

	/**
	 * @param ret
	 * @param coord0
	 */
	protected void getPE_(double[] ret, double[] coord0) {
		// 指示された座標を含む要素の探索
		// 範囲チェック
		if (pedata.getXMax() != null && coord0[0] > pedata.getXMax()) {
			return;
		}
		if (pedata.getYMax() != null && coord0[1] > pedata.getYMax()) {
			return;
		}
		if (pedata.getXMin() != null && coord0[0] < pedata.getXMin()) {
			return;
		}
		if (pedata.getYMin() != null && coord0[1] < pedata.getYMin()) {
			return;
		}
		// 一番近いノードの探索
		NodeType near = pedata.getNodes().getNode().get(0);
		double r0 = Math.sqrt((near.getX() - coord0[0])
				* (near.getX() - coord0[0]) + (near.getY() - coord0[1])
				* (near.getY() - coord0[1]));
		for (NodeType nd : pedata.getNodes().getNode()) {
			double r1 = Math.sqrt((nd.getX() - coord0[0])
					* (nd.getX() - coord0[0]) + (nd.getY() - coord0[1])
					* (nd.getY() - coord0[1]));
			if (r1 < r0) {
				r0 = r1;
				near = nd;
			}
		}
		// 一番近いノードを含む要素から対象の要素を絞り込む
		ElementType near_ele = null;
		double near_ele_r = 1000000;
		for (ElementType et : pedata.getElements().getElement()) {
			if (et.getNodeId().contains(Integer.valueOf(near.getId()))) {
				double[] r2 = new double[2];
				double tmp1;
				double[][] value = new double[4][2];
				for (int i = 0; i < et.getNodeId().size(); i++) {
					NodeType nd = pedata.getNodes().getNode().get(
							et.getNodeId().get(i) - 1);
					if (nd.getId() != et.getNodeId().get(i)) {
						for (NodeType nt : pedata.getNodes().getNode()) {
							if (nt.getId() == et.getNodeId().get(i)) {
								nd = nt;
								break;
							}
						}
					}
					value[i][0] = nd.getX();
					value[i][1] = nd.getY();
				}
				if (et.getType().equalsIgnoreCase("C2D4")) {
					r2 = jp.mydns.masahase.sdes.common.FemFunc
							.invertFormFunction4Square(coord0, value);
				} else if (et.getType().equalsIgnoreCase("C2D3")) {
					r2 = jp.mydns.masahase.sdes.common.FemFunc
							.invertFormFunction3Triangle(coord0, value);
				}
				tmp1 = r2[0] * r2[0] + r2[1] * r2[1];
				if (tmp1 < near_ele_r) {
					near_ele = et;
					near_ele_r = tmp1;
				}
			}
		}
		if (near_ele != null) {
			if (near_ele_r >= 4.0) {
				near_ele = null;
			}
		}
		if (near_ele == null) {
			r0 = Math.sqrt((near.getX() - coord0[0])
					* (near.getX() - coord0[0]) + (near.getY() - coord0[1])
					* (near.getY() - coord0[1]));
			if (r0 < 1.0) {
				ret[0] = near.getStrainData().getX();
				ret[1] = near.getStrainData().getY();
				ret[2] = near.getStrainData().getZ();
				ret[3] = near.getStrainData().getXy();
				ret[4] = near.getStrainData().getXz();
				ret[5] = near.getStrainData().getYz();
				assert FemFunc.calcEstimateStrain(ret) < 0.1;
			} else {
				for (int i = 0; i < 6; i++) {
					ret[i] = 0.0;
				}
			}
		}
		// 要素内での面積座標を算出し形状関数による内挿補間を行う
		if (near_ele != null) {
			if (near_ele.getType().equalsIgnoreCase("C2D4")) {
				double[][] value = new double[4][2];
				for (int i = 0; i < 4; i++) {
					NodeType nd = pedata.getNodes().getNode().get(
							near_ele.getNodeId().get(i) - 1);
					if (nd.getId() != near_ele.getNodeId().get(i)) {
						for (NodeType nt : pedata.getNodes().getNode()) {
							if (nt.getId() == near_ele.getNodeId().get(i)) {
								nd = nt;
								break;
							}
						}
					}
					value[i][0] = nd.getX();
					value[i][1] = nd.getY();
				}
				double[] u = jp.mydns.masahase.sdes.common.FemFunc
						.invertFormFunction4Square(coord0, value);
				assert u[0] * u[0] + u[1] * u[1] < 4;
				double[][] pe = new double[6][4];
				for (int i = 0; i < 4; i++) {
					NodeType nd = pedata.getNodes().getNode().get(
							near_ele.getNodeId().get(i) - 1);
					if (nd.getId() != near_ele.getNodeId().get(i)) {
						for (NodeType nt : pedata.getNodes().getNode()) {
							if (nt.getId() == near_ele.getNodeId().get(i)) {
								nd = nt;
								break;
							}
						}
					}
					pe[0][i] = nd.getStrainData().getX();
					pe[1][i] = nd.getStrainData().getY();
					pe[2][i] = nd.getStrainData().getZ();
					pe[3][i] = nd.getStrainData().getXy();
					pe[4][i] = nd.getStrainData().getXz();
					pe[5][i] = nd.getStrainData().getYz();
				}
				for (int i = 0; i < 6; i++) {
					ret[i] = jp.mydns.masahase.sdes.common.FemFunc
							.formFunction4Square(u[0], u[1], pe[i]);
				}
				if (FemFunc.calcEstimateStrain(ret) > 0.1) {
					System.out.println(u[0] * u[0] + u[1] * u[1]);
					for (int i = 0; i < pe[0].length; i++) {
						for (int j = 0; j < 6; j++) {
							System.out.print(pe[j][i]);
							System.out.print("\t");
						}
						System.out.print("\n");
					}
					for (int i = 0; i < 6; i++) {
						System.out.print(ret[i]);
						System.out.print("\t");
					}
					System.out.print("\n\n");
				}
				assert FemFunc.calcEstimateStrain(ret) < 0.1;
			} else if (near_ele.getType().equalsIgnoreCase("C2D3")) {
				double[][] value = new double[3][2];
				for (int i = 0; i < 3; i++) {
					NodeType nd = pedata.getNodes().getNode().get(
							near_ele.getNodeId().get(i) - 1);
					if (nd.getId() != near_ele.getNodeId().get(i)) {
						for (NodeType nt : pedata.getNodes().getNode()) {
							if (nt.getId() == near_ele.getNodeId().get(i)) {
								nd = nt;
								break;
							}
						}
					}
					value[i][0] = nd.getX();
					value[i][1] = nd.getY();
				}
				double[] u = jp.mydns.masahase.sdes.common.FemFunc
						.invertFormFunction3Triangle(coord0, value);
				double[][] pe = new double[6][3];
				for (int i = 0; i < 3; i++) {
					NodeType nd = pedata.getNodes().getNode().get(
							near_ele.getNodeId().get(i) - 1);
					if (nd.getId() != near_ele.getNodeId().get(i)) {
						for (NodeType nt : pedata.getNodes().getNode()) {
							if (nt.getId() == near_ele.getNodeId().get(i)) {
								nd = nt;
								break;
							}
						}
					}
					pe[0][i] = nd.getStrainData().getX();
					pe[1][i] = nd.getStrainData().getY();
					pe[2][i] = nd.getStrainData().getZ();
					pe[3][i] = nd.getStrainData().getXy();
					pe[4][i] = nd.getStrainData().getXz();
					pe[5][i] = nd.getStrainData().getYz();
				}
				for (int i = 0; i < 6; i++) {
					ret[i] = jp.mydns.masahase.sdes.common.FemFunc
							.formFunction3Triangle(u[0], u[1], pe[i]);
				}
				if (FemFunc.calcEstimateStrain(ret) > 0.1) {
					for (int i = 0; i < pe[0].length; i++) {
						for (int j = 0; j < 6; j++) {
							System.out.print(pe[j][i]);
							System.out.print("\t");
						}
						System.out.print("\n");
					}
					for (int i = 0; i < 6; i++) {
						System.out.print(ret[i]);
						System.out.print("\t");
					}
					System.out.print("\n\n");
				}
				assert FemFunc.calcEstimateStrain(ret) < 0.1;
			} else {
				throw new RuntimeException("There is no support element type:"
						+ near_ele.getType() + "Elemid:" + near_ele.getId());
			}
		}
	}

	/**
	 * ひずみの座標変換. ひずみテンソルe<sub>ij</sub>が既知の時、x<sub>i</sub>=g<sub>ij</sub>*x
	 * '<sub>j</sub>となるようg<sub>ij</sub>を置くと
	 * e'<sub>ij</sub>=g<sub>il</sub>*g<sub>jm</sub>*e<sub>lm</sub>
	 * 
	 * @param data
	 *            double[6]
	 * @param coord
	 *            double[3]
	 * @return double[6]
	 */
	protected double[] changeECoordinate(double[] data, double[] coord) {
		double[] ret = new double[6];
		double[] tmp1 = new double[6];
		double[] tmp2 = new double[6];
		double[] tmp3 = new double[6];
		double[] tmp4 = new double[6];
		double g11, g12, g13, g21, g22, g23, g31, g32, g33;
		double r = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]);
		double cost = coord[0] / r;
		double sint = coord[1] / r;
		double cosp = Math.cos(angle);
		double sinp = Math.sin(angle);

		g11 = 1;
		g12 = 0;
		g13 = 0;
		g21 = 0;
		g22 = 0;
		g23 = 1;
		g31 = 0;
		g32 = 1;
		g33 = 0;
		tmp1[0] = g11 * g11 * data[0] + g12 * g12 * data[1] + g13 * g13
				* data[2] + g11 * g12 * data[3] + g11 * g13 * data[4] + g12
				* g13 * data[5];
		tmp1[1] = g21 * g21 * data[0] + g22 * g22 * data[1] + g23 * g23
				* data[2] + g21 * g22 * data[3] + g21 * g23 * data[4] + g22
				* g23 * data[5];
		tmp1[2] = g31 * g31 * data[0] + g32 * g32 * data[1] + g33 * g33
				* data[2] + g31 * g32 * data[3] + g31 * g33 * data[4] + g32
				* g33 * data[5];
		tmp1[3] = g11 * g21 * data[0] + g12 * g22 * data[1] + g13 * g23
				* data[2] + (g11 * g22 + g21 * g12) * data[3]
				+ (g11 * g23 + g21 * g13) * data[4] + (g12 * g23 + g22 * g13)
				* data[5];
		tmp1[4] = g11 * g31 * data[0] + g12 * g32 * data[1] + g13 * g33
				* data[2] + (g11 * g32 + g31 * g12) * data[3]
				+ (g11 * g33 + g31 * g13) * data[4] + (g12 * g33 + g32 * g13)
				* data[5];
		tmp1[5] = g21 * g31 * data[0] + g22 * g32 * data[1] + g23 * g33
				* data[2] + (g21 * g32 + g31 * g22) * data[3]
				+ (g21 * g33 + g31 * g23) * data[4] + (g22 * g33 + g32 * g23)
				* data[5];

		g11 = cosp;
		g12 = sinp;
		g13 = 0;
		g21 = -sinp;
		g22 = cosp;
		g23 = 0;
		g31 = 0;
		g32 = 0;
		g33 = 1;
		tmp2[0] = g11 * g11 * tmp1[0] + g12 * g12 * tmp1[1] + g13 * g13
				* tmp1[2] + g11 * g12 * tmp1[3] + g11 * g13 * tmp1[4] + g12
				* g13 * tmp1[5];
		tmp2[1] = g21 * g21 * tmp1[0] + g22 * g22 * tmp1[1] + g23 * g23
				* tmp1[2] + g21 * g22 * tmp1[3] + g21 * g23 * tmp1[4] + g22
				* g23 * tmp1[5];
		tmp2[2] = g31 * g31 * tmp1[0] + g32 * g32 * tmp1[1] + g33 * g33
				* tmp1[2] + g31 * g32 * tmp1[3] + g31 * g33 * tmp1[4] + g32
				* g33 * tmp1[5];
		tmp2[3] = g11 * g21 * tmp1[0] + g12 * g22 * tmp1[1] + g13 * g23
				* tmp1[2] + (g11 * g22 + g21 * g12) * tmp1[3]
				+ (g11 * g23 + g21 * g13) * tmp1[4] + (g12 * g23 + g22 * g13)
				* tmp1[5];
		tmp2[4] = g11 * g31 * tmp1[0] + g12 * g32 * tmp1[1] + g13 * g33
				* tmp1[2] + (g11 * g32 + g31 * g12) * tmp1[3]
				+ (g11 * g33 + g31 * g13) * tmp1[4] + (g12 * g33 + g32 * g13)
				* tmp1[5];
		tmp2[5] = g21 * g31 * tmp1[0] + g22 * g32 * tmp1[1] + g23 * g33
				* tmp1[2] + (g21 * g32 + g31 * g22) * tmp1[3]
				+ (g21 * g33 + g31 * g23) * tmp1[4] + (g22 * g33 + g32 * g23)
				* tmp1[5];

		g11 = 1;
		g12 = 0;
		g13 = 0;
		g21 = 0;
		g22 = 0;
		g23 = 1;
		g31 = 0;
		g32 = 1;
		g33 = 0;
		tmp3[0] = g11 * g11 * tmp2[0] + g12 * g12 * tmp2[1] + g13 * g13
				* tmp2[2] + g11 * g12 * tmp2[3] + g11 * g13 * tmp2[4] + g12
				* g13 * tmp2[5];
		tmp3[1] = g21 * g21 * tmp2[0] + g22 * g22 * tmp2[1] + g23 * g23
				* tmp2[2] + g21 * g22 * tmp2[3] + g21 * g23 * tmp2[4] + g22
				* g23 * tmp2[5];
		tmp3[2] = g31 * g31 * tmp2[0] + g32 * g32 * tmp2[1] + g33 * g33
				* tmp2[2] + g31 * g32 * tmp2[3] + g31 * g33 * tmp2[4] + g32
				* g33 * tmp2[5];
		tmp3[3] = g11 * g21 * tmp2[0] + g12 * g22 * tmp2[1] + g13 * g23
				* tmp2[2] + (g11 * g22 + g21 * g12) * tmp2[3]
				+ (g11 * g23 + g21 * g13) * tmp2[4] + (g12 * g23 + g22 * g13)
				* tmp2[5];
		tmp3[4] = g11 * g31 * tmp2[0] + g12 * g32 * tmp2[1] + g13 * g33
				* tmp2[2] + (g11 * g32 + g31 * g12) * tmp2[3]
				+ (g11 * g33 + g31 * g13) * tmp2[4] + (g12 * g33 + g32 * g13)
				* tmp2[5];
		tmp3[5] = g21 * g31 * tmp2[0] + g22 * g32 * tmp2[1] + g23 * g33
				* tmp2[2] + (g21 * g32 + g31 * g22) * tmp2[3]
				+ (g21 * g33 + g31 * g23) * tmp2[4] + (g22 * g33 + g32 * g23)
				* tmp2[5];

		g11 = cost;
		g12 = -sint;
		g13 = 0;
		g21 = sint;
		g22 = cost;
		g23 = 0;
		g31 = 0;
		g32 = 0;
		g33 = 1;
		tmp4[0] = g11 * g11 * tmp3[0] + g12 * g12 * tmp3[1] + g13 * g13
				* tmp3[2] + g11 * g12 * tmp3[3] + g11 * g13 * tmp3[4] + g12
				* g13 * tmp3[5];
		tmp4[1] = g21 * g21 * tmp3[0] + g22 * g22 * tmp3[1] + g23 * g23
				* tmp3[2] + g21 * g22 * tmp3[3] + g21 * g23 * tmp3[4] + g22
				* g23 * tmp3[5];
		tmp4[2] = g31 * g31 * tmp3[0] + g32 * g32 * tmp3[1] + g33 * g33
				* tmp3[2] + g31 * g32 * tmp3[3] + g31 * g33 * tmp3[4] + g32
				* g33 * tmp3[5];
		tmp4[3] = g11 * g21 * tmp3[0] + g12 * g22 * tmp3[1] + g13 * g23
				* tmp3[2] + (g11 * g22 + g21 * g12) * tmp3[3]
				+ (g11 * g23 + g21 * g13) * tmp3[4] + (g12 * g23 + g22 * g13)
				* tmp3[5];
		tmp4[4] = g11 * g31 * tmp3[0] + g12 * g32 * tmp3[1] + g13 * g33
				* tmp3[2] + (g11 * g32 + g31 * g12) * tmp3[3]
				+ (g11 * g33 + g31 * g13) * tmp3[4] + (g12 * g33 + g32 * g13)
				* tmp3[5];
		tmp4[5] = g21 * g31 * tmp3[0] + g22 * g32 * tmp3[1] + g23 * g33
				* tmp3[2] + (g21 * g32 + g31 * g22) * tmp3[3]
				+ (g21 * g33 + g31 * g23) * tmp3[4] + (g22 * g33 + g32 * g23)
				* tmp3[5];

		g11 = vec_a[0];
		g12 = vec_a[1];
		g13 = vec_a[2];
		g21 = vec_c[0];
		g22 = vec_c[1];
		g23 = vec_c[2];
		g31 = vec_b[0];
		g32 = vec_b[1];
		g33 = vec_b[2];
		ret[0] = g11 * g11 * tmp4[0] + g12 * g12 * tmp4[1] + g13 * g13
				* tmp4[2] + g11 * g12 * tmp4[3] + g11 * g13 * tmp4[4] + g12
				* g13 * tmp4[5];
		ret[1] = g21 * g21 * tmp4[0] + g22 * g22 * tmp4[1] + g23 * g23
				* tmp4[2] + g21 * g22 * tmp4[3] + g21 * g23 * tmp4[4] + g22
				* g23 * tmp4[5];
		ret[2] = g31 * g31 * tmp4[0] + g32 * g32 * tmp4[1] + g33 * g33
				* tmp4[2] + g31 * g32 * tmp4[3] + g31 * g33 * tmp4[4] + g32
				* g33 * tmp4[5];
		ret[3] = g11 * g21 * tmp4[0] + g12 * g22 * tmp4[1] + g13 * g23
				* tmp4[2] + (g11 * g22 + g21 * g12) * tmp4[3]
				+ (g11 * g23 + g21 * g13) * tmp4[4] + (g12 * g23 + g22 * g13)
				* tmp4[5];
		ret[4] = g11 * g31 * tmp4[0] + g12 * g32 * tmp4[1] + g13 * g33
				* tmp4[2] + (g11 * g32 + g31 * g12) * tmp4[3]
				+ (g11 * g33 + g31 * g13) * tmp4[4] + (g12 * g33 + g32 * g13)
				* tmp4[5];
		ret[5] = g21 * g31 * tmp4[0] + g22 * g32 * tmp4[1] + g23 * g33
				* tmp4[2] + (g21 * g32 + g31 * g22) * tmp4[3]
				+ (g21 * g33 + g31 * g23) * tmp4[4] + (g22 * g33 + g32 * g23)
				* tmp4[5];

		return ret;
	}

	// for debug
	public static void main(String[] args) {
		double[] start, end, ref, coord, tmp;
		double theta, tmp_x, tmp_y;
		start = new double[3];
		end = new double[3];
		ref = new double[3];
		tmp = new double[3];
		System.out.println("1)座標変換の確認(x,y,z)->(x,y,theta)");
		System.out.println("中心は全て(0,0,0)");
		System.out.println("1-1)開始点(10.,0,0)、終了点(0,10.,0)、回転角0");
		start[0] = 10.0;
		start[1] = 0;
		start[2] = 0;
		end[0] = 0;
		end[1] = 10.0;
		end[2] = 0;
		ref[0] = 0;
		ref[1] = 0;
		ref[2] = 0;
		CurveLine cl1 = new CurveLine(null, start, end, ref, 0);
		tmp[0] = 10;
		tmp[1] = 0;
		tmp[2] = 0;
		coord = cl1.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl1.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl1.angle) + tmp_y * Math.sin(cl1.angle);
		coord[1] = -tmp_x * Math.sin(cl1.angle) + tmp_y * Math.cos(cl1.angle);
		System.out.printf("(10,0,0)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[2] = 10;
		coord = cl1.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl1.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl1.angle) + tmp_y * Math.sin(cl1.angle);
		coord[1] = -tmp_x * Math.sin(cl1.angle) + tmp_y * Math.cos(cl1.angle);
		System.out.printf("(10,0,10)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[2] = 0;
		tmp[0] = 9;
		coord = cl1.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl1.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl1.angle) + tmp_y * Math.sin(cl1.angle);
		coord[1] = -tmp_x * Math.sin(cl1.angle) + tmp_y * Math.cos(cl1.angle);
		System.out.printf("(9,0,0)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[1] = 10;
		tmp[2] = 0;
		tmp[0] = 0;
		coord = cl1.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl1.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl1.angle) + tmp_y * Math.sin(cl1.angle);
		coord[1] = -tmp_x * Math.sin(cl1.angle) + tmp_y * Math.cos(cl1.angle);
		System.out.printf("(0,10,0)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[2] = 10;
		coord = cl1.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl1.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl1.angle) + tmp_y * Math.sin(cl1.angle);
		coord[1] = -tmp_x * Math.sin(cl1.angle) + tmp_y * Math.cos(cl1.angle);
		System.out.printf("(0,10,10)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[2] = -10;
		coord = cl1.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl1.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl1.angle) + tmp_y * Math.sin(cl1.angle);
		coord[1] = -tmp_x * Math.sin(cl1.angle) + tmp_y * Math.cos(cl1.angle);
		System.out
				.printf("(0,10,-10)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		System.out.println("1-2)開始点(10.,0,0)、終了点(0,10.,0)、回転角90°");
		start[0] = 10.0;
		start[1] = 0;
		start[2] = 0;
		end[0] = 0;
		end[1] = 10.0;
		end[2] = 0;
		ref[0] = 0;
		ref[1] = 0;
		ref[2] = 0;
		CurveLine cl2 = new CurveLine(null, start, end, ref, 90);
		tmp[0] = 10;
		tmp[1] = 0;
		tmp[2] = 0;
		coord = cl2.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl2.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl2.angle) + tmp_y * Math.sin(cl2.angle);
		coord[1] = -tmp_x * Math.sin(cl2.angle) + tmp_y * Math.cos(cl2.angle);
		System.out.printf("(10,0,0)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[2] = 10;
		coord = cl2.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl2.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl2.angle) + tmp_y * Math.sin(cl2.angle);
		coord[1] = -tmp_x * Math.sin(cl2.angle) + tmp_y * Math.cos(cl2.angle);
		System.out.printf("(10,0,10)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[2] = 0;
		tmp[0] = 9;
		coord = cl2.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl2.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl2.angle) + tmp_y * Math.sin(cl2.angle);
		coord[1] = -tmp_x * Math.sin(cl2.angle) + tmp_y * Math.cos(cl2.angle);
		System.out.printf("(9,0,0)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[1] = 10;
		tmp[2] = 0;
		tmp[0] = 0;
		coord = cl2.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl2.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl2.angle) + tmp_y * Math.sin(cl2.angle);
		coord[1] = -tmp_x * Math.sin(cl2.angle) + tmp_y * Math.cos(cl2.angle);
		System.out.printf("(0,10,0)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[2] = 10;
		coord = cl2.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl2.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl2.angle) + tmp_y * Math.sin(cl2.angle);
		coord[1] = -tmp_x * Math.sin(cl2.angle) + tmp_y * Math.cos(cl2.angle);
		System.out.printf("(0,10,10)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[2] = -10;
		coord = cl2.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl2.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl2.angle) + tmp_y * Math.sin(cl2.angle);
		coord[1] = -tmp_x * Math.sin(cl2.angle) + tmp_y * Math.cos(cl2.angle);
		System.out
				.printf("(0,10,-10)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		System.out.println("1-3)開始点(10.,0,0)、終了点(0,10.,0)、回転角-90°");
		start[0] = 10.0;
		start[1] = 0;
		start[2] = 0;
		end[0] = 0;
		end[1] = 10.0;
		end[2] = 0;
		ref[0] = 0;
		ref[1] = 0;
		ref[2] = 0;
		CurveLine cl3 = new CurveLine(null, start, end, ref, -90);
		tmp[0] = 10;
		tmp[1] = 0;
		tmp[2] = 0;
		coord = cl3.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl3.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl3.angle) + tmp_y * Math.sin(cl3.angle);
		coord[1] = -tmp_x * Math.sin(cl3.angle) + tmp_y * Math.cos(cl3.angle);
		System.out.printf("(10,0,0)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[2] = 10;
		coord = cl3.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl3.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl3.angle) + tmp_y * Math.sin(cl3.angle);
		coord[1] = -tmp_x * Math.sin(cl3.angle) + tmp_y * Math.cos(cl3.angle);
		System.out.printf("(10,0,10)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[2] = 0;
		tmp[0] = 9;
		coord = cl3.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl3.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl3.angle) + tmp_y * Math.sin(cl3.angle);
		coord[1] = -tmp_x * Math.sin(cl3.angle) + tmp_y * Math.cos(cl3.angle);
		System.out.printf("(9,0,0)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[1] = 10;
		tmp[2] = 0;
		tmp[0] = 0;
		coord = cl3.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl3.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl3.angle) + tmp_y * Math.sin(cl3.angle);
		coord[1] = -tmp_x * Math.sin(cl3.angle) + tmp_y * Math.cos(cl3.angle);
		System.out.printf("(0,10,0)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[2] = 10;
		coord = cl3.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl3.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl3.angle) + tmp_y * Math.sin(cl3.angle);
		coord[1] = -tmp_x * Math.sin(cl3.angle) + tmp_y * Math.cos(cl3.angle);
		System.out.printf("(0,10,10)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[2] = -10;
		coord = cl3.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl3.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl3.angle) + tmp_y * Math.sin(cl3.angle);
		coord[1] = -tmp_x * Math.sin(cl3.angle) + tmp_y * Math.cos(cl3.angle);
		System.out
				.printf("(0,10,-10)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		System.out.println("1-4)開始点(-10,0,0)、終了点(0,0,-10)、回転角0°");
		start[0] = -10;
		start[1] = 0;
		start[2] = 0;
		end[0] = 0;
		end[1] = 0;
		end[2] = -10;
		ref[0] = 0;
		ref[1] = 0;
		ref[2] = 0;
		CurveLine cl4 = new CurveLine(null, start, end, ref, 0);
		tmp[0] = -10;
		tmp[1] = 0;
		tmp[2] = 0;
		coord = cl4.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl4.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl4.angle) + tmp_y * Math.sin(cl4.angle);
		coord[1] = -tmp_x * Math.sin(cl4.angle) + tmp_y * Math.cos(cl4.angle);
		System.out.printf("(-10,0,0)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[1] = 10;
		coord = cl4.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl4.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl4.angle) + tmp_y * Math.sin(cl4.angle);
		coord[1] = -tmp_x * Math.sin(cl4.angle) + tmp_y * Math.cos(cl4.angle);
		System.out
				.printf("(-10,10,0)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[1] = -10;
		coord = cl4.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl4.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl4.angle) + tmp_y * Math.sin(cl4.angle);
		coord[1] = -tmp_x * Math.sin(cl4.angle) + tmp_y * Math.cos(cl4.angle);
		System.out.printf("(-10,-10,0)->(%f,%f,%f)\n", coord[0], coord[1],
				theta);
		tmp[1] = 0;
		tmp[2] = -10;
		tmp[0] = 0;
		coord = cl4.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl4.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl4.angle) + tmp_y * Math.sin(cl4.angle);
		coord[1] = -tmp_x * Math.sin(cl4.angle) + tmp_y * Math.cos(cl4.angle);
		System.out.printf("(0,0,-10)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[1] = 10;
		coord = cl4.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl4.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl4.angle) + tmp_y * Math.sin(cl4.angle);
		coord[1] = -tmp_x * Math.sin(cl4.angle) + tmp_y * Math.cos(cl4.angle);
		System.out
				.printf("(0,10,-10)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[1] = -10;
		coord = cl4.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl4.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl4.angle) + tmp_y * Math.sin(cl4.angle);
		coord[1] = -tmp_x * Math.sin(cl4.angle) + tmp_y * Math.cos(cl4.angle);
		System.out.printf("(0,-10,-10)->(%f,%f,%f)\n", coord[0], coord[1],
				theta);
		System.out.println("1-5)開始点(-10,0,0)、終了点(0,0,-10)、回転角90°");
		start[0] = -10;
		start[1] = 0;
		start[2] = 0;
		end[0] = 0;
		end[1] = 0;
		end[2] = -10;
		ref[0] = 0;
		ref[1] = 0;
		ref[2] = 0;
		CurveLine cl5 = new CurveLine(null, start, end, ref, 90);
		tmp[0] = -10;
		tmp[1] = 0;
		tmp[2] = 0;
		coord = cl5.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl5.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl5.angle) + tmp_y * Math.sin(cl5.angle);
		coord[1] = -tmp_x * Math.sin(cl5.angle) + tmp_y * Math.cos(cl5.angle);
		System.out.printf("(-10,0,0)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[1] = 10;
		coord = cl5.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl5.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl5.angle) + tmp_y * Math.sin(cl5.angle);
		coord[1] = -tmp_x * Math.sin(cl5.angle) + tmp_y * Math.cos(cl5.angle);
		System.out
				.printf("(-10,10,0)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[1] = -10;
		coord = cl5.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl5.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl5.angle) + tmp_y * Math.sin(cl5.angle);
		coord[1] = -tmp_x * Math.sin(cl5.angle) + tmp_y * Math.cos(cl5.angle);
		System.out.printf("(-10,-10,0)->(%f,%f,%f)\n", coord[0], coord[1],
				theta);
		tmp[1] = 0;
		tmp[2] = -10;
		tmp[0] = 0;
		coord = cl5.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl5.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl5.angle) + tmp_y * Math.sin(cl5.angle);
		coord[1] = -tmp_x * Math.sin(cl5.angle) + tmp_y * Math.cos(cl5.angle);
		System.out.printf("(0,0,-10)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[1] = 10;
		coord = cl5.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl5.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl5.angle) + tmp_y * Math.sin(cl5.angle);
		coord[1] = -tmp_x * Math.sin(cl5.angle) + tmp_y * Math.cos(cl5.angle);
		System.out
				.printf("(0,10,-10)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[1] = -10;
		coord = cl5.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl5.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl5.angle) + tmp_y * Math.sin(cl5.angle);
		coord[1] = -tmp_x * Math.sin(cl5.angle) + tmp_y * Math.cos(cl5.angle);
		System.out.printf("(0,-10,-10)->(%f,%f,%f)\n", coord[0], coord[1],
				theta);
		System.out.println("1-5)開始点(-10,0,0)、終了点(0,0,-10)、回転角180°");
		start[0] = -10;
		start[1] = 0;
		start[2] = 0;
		end[0] = 0;
		end[1] = 0;
		end[2] = -10;
		ref[0] = 0;
		ref[1] = 0;
		ref[2] = 0;
		CurveLine cl6 = new CurveLine(null, start, end, ref, 180);
		tmp[0] = -10;
		tmp[1] = 0;
		tmp[2] = 0;
		coord = cl6.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl6.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl6.angle) + tmp_y * Math.sin(cl6.angle);
		coord[1] = -tmp_x * Math.sin(cl6.angle) + tmp_y * Math.cos(cl6.angle);
		System.out.printf("(-10,0,0)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[1] = 10;
		coord = cl6.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl6.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl6.angle) + tmp_y * Math.sin(cl6.angle);
		coord[1] = -tmp_x * Math.sin(cl6.angle) + tmp_y * Math.cos(cl6.angle);
		System.out
				.printf("(-10,10,0)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[1] = -10;
		coord = cl6.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl6.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl6.angle) + tmp_y * Math.sin(cl6.angle);
		coord[1] = -tmp_x * Math.sin(cl6.angle) + tmp_y * Math.cos(cl6.angle);
		System.out.printf("(-10,-10,0)->(%f,%f,%f)\n", coord[0], coord[1],
				theta);
		tmp[1] = 0;
		tmp[2] = 0;
		tmp[0] = -9;
		coord = cl6.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl6.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl6.angle) + tmp_y * Math.sin(cl6.angle);
		coord[1] = -tmp_x * Math.sin(cl6.angle) + tmp_y * Math.cos(cl6.angle);
		System.out.printf("(-9,0,0)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		tmp[0] = -11;
		coord = cl6.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl6.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl6.angle) + tmp_y * Math.sin(cl6.angle);
		coord[1] = -tmp_x * Math.sin(cl6.angle) + tmp_y * Math.cos(cl6.angle);
		System.out.printf("(-11,0,0)->(%f,%f,%f)\n", coord[0], coord[1], theta);
		System.out.println("1-6)開始点(0,10,0)、終了点(-10,0,0)、回転角0°");
		start[0] = 0;
		start[1] = 10;
		start[2] = 0;
		end[0] = -10;
		end[1] = 0;
		end[2] = 0;
		ref[0] = 0;
		ref[1] = 0;
		ref[2] = 0;
		CurveLine cl7 = new CurveLine(null, start, end, ref, 0);
		tmp[0] = 0;
		tmp[1] = 10;
		tmp[2] = -1;
		coord = cl7.changeCoordinate(tmp);
		theta = Math.atan2(coord[1], coord[0]);
		tmp_x = Math.sqrt(coord[0] * coord[0] + coord[1] * coord[1]) - cl7.R;
		tmp_y = coord[2];
		coord[0] = tmp_x * Math.cos(cl7.angle) + tmp_y * Math.sin(cl7.angle);
		coord[1] = -tmp_x * Math.sin(cl7.angle) + tmp_y * Math.cos(cl7.angle);
		System.out.printf("(0,10,-1)->(%f,%f,%f)\n", coord[0], coord[1], theta);

		tmp[0] = 10;
		tmp[1] = 0;
		tmp[2] = 0;

		System.out.println("2)ひずみの座標変換の確認");
		System.out.println("溶接線方向が0.1");
		System.out.println("板幅方向が0.2");
		System.out.println("板厚方向が0.3");
		System.out.println("いづれも開始点に対応する座標変換");
		double[] tmp_e = new double[6];
		double[] tmp_e2;
		double tmp_e1 = FemFunc.calcEstimateStrain(tmp_e);
		tmp_e[0] = 0.2;// 2
		tmp_e[1] = 0.1;
		tmp_e[2] = 0.3;// 3
		tmp_e[3] = 0.4;
		tmp_e[4] = 0.5;
		tmp_e[5] = 0.6;
		System.out.println("2-1)開始点(10.,0,0)、終了点(0,10.,0)、回転角0\t(2,1,3)");
		tmp_e2 = cl1.changeECoordinate(tmp_e, cl1.changeCoordinate(tmp));
		System.out.printf("(%f,%f,%f)\n", tmp_e2[0], tmp_e2[1], tmp_e2[2]);
		// System.out.printf("Error:%f%%\n",Math.abs((tmp_e1-FemFunc.calcEstimateStrain(tmp_e2)/tmp_e1))*100);
		System.out.println("2-2)開始点(10.,0,0)、終了点(0,10.,0)、回転角90\t(3,1,2)");
		tmp_e2 = cl2.changeECoordinate(tmp_e, cl2.changeCoordinate(tmp));
		System.out.printf("(%f,%f,%f)\n", tmp_e2[0], tmp_e2[1], tmp_e2[2]);
		// System.out.printf("Error:%f%%\n",Math.abs((tmp_e1-FemFunc.calcEstimateStrain(tmp_e2)/tmp_e1))*100);
		System.out.println("2-3)開始点(10.,0,0)、終了点(0,10.,0)、回転角-90\t(3,1,2)");
		tmp_e2 = cl3.changeECoordinate(tmp_e, cl3.changeCoordinate(tmp));
		System.out.printf("(%f,%f,%f)\n", tmp_e2[0], tmp_e2[1], tmp_e2[2]);
		// System.out.printf("Error:%f%%\n",Math.abs((tmp_e1-FemFunc.calcEstimateStrain(tmp_e2)/tmp_e1))*100);
		System.out.println("2-4)開始点(-10,0,0)、終了点(0,0,-10)、回転角0\t(2,3,1)");
		tmp_e2 = cl4.changeECoordinate(tmp_e, cl4
				.changeCoordinate(cl4.StartPos));
		System.out.printf("(%f,%f,%f)\n", tmp_e2[0], tmp_e2[1], tmp_e2[2]);
		// System.out.printf("Error:%f%%\n",Math.abs((tmp_e1-FemFunc.calcEstimateStrain(tmp_e2)/tmp_e1))*100);
	}
}
