#include <pthread.h>
#include <semaphore.h> 


#include "ring.h"
#include "ad_ring.h"
#include "mes_print.h"

// Ring Buffer読み込みポインタ個数
#define	AD_RING_READ_NUM	2
// Ring Buffer読み込みポインター　1sec記録用
#define	AD_RING_READ_PTR1	0
// Ring Buffer読み込みポインター　1min記録用
#define	AD_RING_READ_PTR2	1

static AdData	ad_ring_buf[AD_RING_NUM];

// ミューテックス
static pthread_mutex_t	mutex_ad_ring = PTHREAD_MUTEX_INITIALIZER;

// Ring buffer
static RING_T	*ring;

int ad_ring_init(void)
{
pthread_mutex_lock(&mutex_ad_ring);
	ring = ring_create(AD_RING_READ_NUM);
	if (ring == NULL) {
		PERRF("ERROR ring_create() == NULL\n");
pthread_mutex_unlock(&mutex_ad_ring);
		return -1;
	}
	ring_init(ring, AD_RING_NUM);
pthread_mutex_unlock(&mutex_ad_ring);
	return 0;
}
void ad_ring_clear_sec(void)
{
	ring_clear(ring, AD_RING_READ_PTR1);
}
void ad_ring_clear_high(void)
{
	ring_clear(ring, AD_RING_READ_PTR2);
}
void ad_ring_clear_latest(void)
{
	ring_latest_set(ring, -1);
}

// 読み出し位置　1secデータ記録で使用
int ad_ring_read_get_sec(void)
{
	return ring_read_get(ring, AD_RING_READ_PTR1);
}
void ad_ring_read_set_sec(int i)
{
	ring_read_set(ring, AD_RING_READ_PTR1, i);
}
void ad_ring_read_plus_sec(void)
{
	ring_read_plus(ring, AD_RING_READ_PTR1);
}
// 読み出し位置　高速データ記録で使用
int ad_ring_read_get_high(void)
{
	return ring_read_get(ring, AD_RING_READ_PTR2);
}
void ad_ring_read_set_high(int i)
{
	ring_read_set(ring, AD_RING_READ_PTR2, i);
}
void ad_ring_read_plus_high(void)
{
	ring_read_plus(ring, AD_RING_READ_PTR2);
}

// 書き込み位置　AD受信で使用
int ad_ring_write_get(void)
{
	return ring_write_get(ring);
}
void ad_ring_write_plus(void)
{
	ring_write_plus(ring);
}
// 読み込んでいないデータ数
int ad_ring_num_get_sec(void)
{
	return ring_num_get(ring, AD_RING_READ_PTR1);
}
// 読み込んでいないデータ数
int ad_ring_num_get_high(void)
{
	return ring_num_get(ring, AD_RING_READ_PTR2);
}
// 最新データ位置　表示
int ad_ring_latest_get(void)
{
	return ring_latest_get(ring);
}
void ad_ring_latest_set(int i)
{
	ring_latest_set(ring, i);
}
// データ取得
AdData* ad_ring_get(int ptr)
{
	AdData*	p;
pthread_mutex_lock(&mutex_ad_ring);
	p = &ad_ring_buf[ptr];
pthread_mutex_unlock(&mutex_ad_ring);
	return p;
}
/*
	バッファにデータコピー
*/
void ad_ring_set(AdData *data)
{
	int	i;
	
pthread_mutex_lock(&mutex_ad_ring);
	i = ring_write_get(ring);
	ad_ring_buf[i] = *data;
	// 最新データ位置セット
	ring_latest_set(ring, i);
pthread_mutex_unlock(&mutex_ad_ring);
}

/*
	パケットバッファフル？
	1=Full
	0=not Full
*/
int ad_ring_is_full(void)
{
	return ring_is_full(ring, AD_RING_READ_PTR1);
}


/**** CUnit test
*/
#ifdef CUNIT
static void test_ad_ring(void)
{
	ad_ring_init();

	// CLEAR
	ad_ring_clear_sec();
	CU_ASSERT(ad_ring_read_get_sec() == 0);
	ad_ring_clear_high();
	CU_ASSERT(ad_ring_read_get_high() == 0);
	ad_ring_clear_latest();
	CU_ASSERT(ad_ring_latest_get() == -1);

	// PLUS
	ad_ring_read_plus_sec();
	CU_ASSERT(ad_ring_read_get_sec() == 1);
	ad_ring_read_plus_high();
	CU_ASSERT(ad_ring_read_get_high() == 1);
	ad_ring_write_plus();
	CU_ASSERT(ad_ring_write_get() == 1);

	// 境界値でPLUS
	ad_ring_read_set_sec(AD_RING_NUM - 1);
	ad_ring_read_plus_sec();
	CU_ASSERT(ad_ring_read_get_sec() == 0);

	ad_ring_read_set_high(AD_RING_NUM - 1);
	ad_ring_read_plus_high();
	CU_ASSERT(ad_ring_read_get_high() == 0);

	ring->write = AD_RING_NUM - 1;
	ad_ring_write_plus();
	CU_ASSERT(ad_ring_write_get() == 0);
	
	//
	ring->write = 1;
	ring->read[AD_RING_READ_PTR1] = 0;
	CU_ASSERT(ad_ring_num_get_sec() == 1);

	ring->write = 0;
	ring->read[AD_RING_READ_PTR1] = AD_RING_NUM - 1;
	CU_ASSERT(ad_ring_num_get_sec() == 1);

	ring->write = 1;
	ring->read[AD_RING_READ_PTR2] = 0;
	CU_ASSERT(ad_ring_num_get_high() == 1);

	ring->write = 0;
	ring->read[AD_RING_READ_PTR2] = AD_RING_NUM - 1;
	CU_ASSERT(ad_ring_num_get_high() == 1);

}

void ad_ring_test(CU_pSuite test_suite)
{
	CU_add_test(test_suite, "test_ad_ring", test_ad_ring);

}
#endif
