/*
 * File:   main.c
 * sciLogger サブCPU
 * PIC24Fj64GA004
 * 8MHz

 **** Ver1.1
 * 2016/2
 * IDE MPLABXで使えるようにするため
 * xc16-gccのオプション, Option Categories: Optimization, Do not override 'inline'　チェックを入れる
 * _CONFIG2変更 定数定義が変わっているので
 *
 * Created on February 9, 2010, 10:53 AM
 */

#include <stdio.h>
#include <stdlib.h>
#include <p24FJ64GA004.h>
#include <libpic30.h>
#include <string.h>
#include "myuart.h"
#include "myspi.h"
#include "debug_print.h"
#include "ublox.h"
#include "myad.h"
#include "delay.h"
#include "ring.h"
#include "byte2var.h"
#include "myint.h"
#include "mysts.h"
#include "util.h"

_CONFIG1( JTAGEN_OFF & GCP_OFF & GWRP_OFF & COE_OFF & FWDTEN_OFF & ICS_PGx2 )
//_CONFIG2( FCKSM_CSDCMD & OSCIOFNC_ON & POSCMOD_OFF & FNOSC_FRC)
//#pragma config FCKSM = CSDCMD, OSCIOFNC = ON, POSCMOD = OFF, FNOSC = FRC, FPBDIV = DIV_8
_CONFIG2(POSCMOD_NONE & IOL1WAY_OFF & OSCIOFNC_ON & FCKSM_CSDCMD & FCKSM_CSECMD & FNOSC_FRC)  // RC 8MHz


//#define CPU_CLOCK        10000000                         // クロック[ Hz ]
//#define CPU_PLL          8                                // PLL
//#define CLOCK_FREQUENCY  ( ( CPU_CLOCK * CPU_PLL ) / 4 )  // 動作周波数[ Hz ]
//#define FOSC 4000000L
// UART2 Debug Port
#define BAUDRATE2         19200L
// UART1 GPS
#define BAUDRATE1         9600L

// SPI受信データの先頭文字　送信コマンドでも使用
#define	SPI_HEAD_CHAR	'$'

/*** ADバッファ *************************************************
 * 50Hzデータ
 * 1秒平均データ
 * タイムスタンプ
 */
#define ADBUF_BYTE  3   // AD 1CH 1sample dataのbyte数
#define ADBUF_LEN   (sizeof(ADbufType)+1)   // SPI送信1packetの長さ byte $含まず　DMA問題のために+1byteしている

#define RESDATA_LEN 17  // コマンドに対する返答データ　最大長さ

// 送信データの先頭からのオフセット adbuf内のオフセット　$含まない
#define	ADBUF_OFS_SUM		962 // checksum

// SPI送信にそのまま使われるので注意！
// 全体のサイズが2byteの整数倍になるようにすること　word align
typedef struct {
    unsigned char   rescode;    // コマンドに対する返答 ACK/NAK/NUL
    unsigned char   resdata[RESDATA_LEN];   // コマンドに対する返答データ
    UbloxNavTimeUtc t;  // タイムスタンプ
    long data1sec[AD_CHNUM];  // 1sec平均値
    unsigned char data[AD_CHNUM][ADBUF_BYTE*AD_SAMPLE];  // AD_SAMPLE[Hz]のデータ
    unsigned char suma;
    unsigned char sumb;
} ADbufType;
//} __attribute__((__packed__)) ADbufType;
static ADbufType   adbuf[RING_NUM];

#define adbuf_get(buf_no)   (&adbuf[buf_no])
#define adbuf_data_get(buf_no)   (adbuf[buf_no].data)
#define adbuf_1sec_get(buf_no)   (adbuf[buf_no].data1sec)
#define adbuf_time_get(buf_no)   (&(adbuf[buf_no].t))

inline void adbuf_write(unsigned char buf_no, unsigned char ch, int cnt, long *data)
{
    unsigned char    *ptr = (unsigned char*)data;
    unsigned char   i;

    for(i = 0; i < ADBUF_BYTE; i++) {
        adbuf[buf_no].data[ch][ADBUF_BYTE*cnt + i] = *(ptr++);
    }
}
// 1
inline void adbuf_1sec_write(char buf_no, long *data)
{
    my_memcpy(adbuf_1sec_get(buf_no), data, sizeof(long) * AD_CHNUM);
/*    char ch;
    for(ch = 0; ch < AD_CHNUM; ch++)
        adbuf_1sec[buf_no][ch] = data[ch];
*/
}
inline void adbuf_time_write(char buf_no, UbloxNavTimeUtc *t)
{
//    my_memcpy(&adbuf_t[buf_no], t, sizeof(UbloxNavTimeUtc));
    my_memcpy(adbuf_time_get(buf_no), t, sizeof(UbloxNavTimeUtc));
}
inline void adbuf_sum_write(char buf_no, unsigned char *suma, unsigned char *sumb)
{
    adbuf[buf_no].suma = *suma;
    adbuf[buf_no].sumb = *sumb;
}

/**** SPI CMD受信バッファ ********************************************
 */
#define SPI_RX_LEN   20  // Overoから受信するコマンド長　$含む
static char spi_rx_buf[SPI_RX_LEN];
static char spi_rx_cnt; // 受信文字数カウント 0〜SPI_RX_LEN

// 受信コマンド内のオフセット位置
#define CMD_OFS_CMDCODE 1
#define CMD_OFS_CMDDATA 2

// コマンドコード
#define CMD_CODE_GAIN   1

// 返答コード
#define RES_CODE_ACK    1
#define RES_CODE_NAK    2
#define RES_CODE_NUL    0

// 状態
#define CMD_STS_NONE    0   // 有効なコマンド無し
#define CMD_STS_RCVD    1   // 有効なコマンド受け取った
static char spi_cmd_sts;    // コマンド受信状態

#define spi_cmd_sts_set(i)  INTERRUPT_PROTECT(spi_cmd_sts = i)
#define spi_cmd_sts_get()   spi_cmd_sts


// ACK送信
void spi_res_send(unsigned char res_code, unsigned char *res_data, int data_len)
{
    ADbufType *ad;

    INTERRUPT_PROTECT(
        ad = adbuf_get(ring_read1_get());
        ad->rescode = res_code;
        if (res_data == NULL || data_len <= 0) {
            my_memset(ad->resdata, 0, RESDATA_LEN);
        } else {
            my_memcpy(ad->resdata, res_data, data_len);
        }
    );
}
#define spi_res_send_ack()  spi_res_send(RES_CODE_ACK, NULL, 0)
#define spi_res_send_nak()  spi_res_send(RES_CODE_NAK, NULL, 0)


// コマンドに対応する処理
void spi_cmd_proc(void)
{
    int i;

    switch(spi_rx_buf[CMD_OFS_CMDCODE]) {
        case CMD_CODE_GAIN:
            for(i = 0; i < AD_CHNUM; i++) {
                pga_gain_set(i, spi_rx_buf[CMD_OFS_CMDDATA + i]);
            }
            // ACK送信
            spi_res_send_ack();
            break;
        default:
            PDEBUG("spi_cmd_porc(): unknown CMD CODE\r\n");
            // NACK送信
            spi_res_send_nak();
            break;
    }
}

/**** ublox ************************************************
 */
static UbloxNavTimeUtc gpsNow;
static void ublox_rcv(void)
{
    unsigned char class, id;
    unsigned int    len;
    unsigned char   payload[128];
//    char    sz[128];
    UbloxNavTimeUtc *gps = &gpsNow;

    ublox_rcv_pkt(&class, &id, &len, payload);
//    sprintf(sz, "CLASS=%02X ID=%02X LEN=%d\r\n", class, id, len);
//    PDEBUG(sz);
//    ublox_debug_pkt_out(payload, len);
    if (class == UBX_CLS_NAV && id == UBX_ID_NAV_TIMEUTC && len == UBX_LEN_NAV_TIMEUTC) {
        ublox_decode_nav_timeutc(payload, gps);
/*        sprintf(sz, "%04u/%02d/%02d %02d:%02d:%02d.%09ld %6lums %6luns %02X\r\n",
            gps->year, gps->month, gps->day, gps->hour, gps->min, gps->sec, gps->nano,
            gps->tow, gps->tacc, gps->valid
        );
        PDEBUG(sz);
*/
    }
}
/**** INT1/SPI1 ***********************************************/
/*
 * INT1 Int及びSPI1 IntでADのデータを取得する
 */
#define ADSTS_IDLE  0
#define ADSTS_H 1
#define ADSTS_M 2
#define ADSTS_L 3


static char adint_ch;   // 取得中のCH
static char adint_cnt = 0;  // 0〜AD_SAMPLE-1
static char adint_sts = ADSTS_IDLE;
static long adint_add[AD_CHNUM];    // 1秒平均用加算

/**** 1PPS ********************************************/
static char pps_intf;

#define pps_intf_clear()  pps_intf = 0
#define pps_intf_set()  pps_intf = 1
#define pps_is_intf_enable()    (pps_intf != 0)

/**** DataOut SPI2 *********************************************/
#define SPI2CSpin  (PORTCbits.RC8)

// SPI2 未送信データバイト数
static int  dataout_cnt;
#define dataout_cnt_set(i)  dataout_cnt = (i)
#define dataout_cnt_get()  dataout_cnt
#define dataout_cnt_minus()  (dataout_cnt--)

// 送信位置ポインタ
static char    *dout_ptr;


/**** DRDY OUT ****************************************/
#define DRDY_OUT    PORTCbits.RC9


/*
 * main()
 */
int main(int argc, char** argv) {
    unsigned int i;
    // UART2 DebugPort
    const double Baudrate2 = ( double )FCY / ( 16 * BAUDRATE2 ) - 1;
    // ボーレートの小数点以下を四捨五入する
    unsigned int baudrate2 = ( unsigned int )( Baudrate2 + 0.5 );
    // UART1 GPS
    const double Baudrate1 = ( double )FCY / ( 16 * BAUDRATE1 ) - 1;
    // ボーレートの小数点以下を四捨五入する
    unsigned int baudrate1 = ( unsigned int )( Baudrate1 + 0.5 );
    char    sz[128];
    UbloxNavTimeUtc *gps = &gpsNow;
    ADbufType   *ad;
    
    /**** 割り込み禁止 ****/
    SET_CPU_IPL( 7 );
    __builtin_write_OSCCONL(OSCCON & 0xbf); //clear the bit 6 of OSCCONL to unlock Pin Re-map
    // 割り込みネストenable
    int_nest_enable();

    // UART1ピン割り当て
    RPINR18bits.U1RXR = 6;	// UART1 RX to RP6
    RPOR2bits.RP5R =  3;       	// UART1 TX(3) to RP5
    // UART2ピン割り当て
    RPINR19bits.U2RXR = 21;	// UART2 RX to RP21
    RPOR10bits.RP20R = 5;       // UART2 TX(5) to RP20
    // PIC-RDYピン割り当て
    RPOR12bits.RP25R = 0;       // RC9

    // DataOut SPI2 pin config
    RPINR22bits.SDI2R = 13;     // SPI2 RX to RP13
    RPINR22bits.SCK2R = 23;     // SPI2 CLK to RP23
    RPINR23bits.SS2R = 24;      // SPI2 SS to RP24
    RPOR11bits.RP22R = 10;      // SPI1 SDO2(10) to RP22
    // AD SPI1 pin config
    RPINR20bits.SDI1R = 18;     // SPI1 RX to RP18
    RPOR8bits.RP16R = 8;        // SPI1 SCK1(8) to RP16
    RPOR8bits.RP17R = 7;        // SPI1 SDO1(7) to RP17
    // TRISx 1=input 0=output
    TRISA = 0x0000;             // PortA RA7,8,9 output
    //        5432109876543210
    TRISB = 0b0011000011000000; // PortB RB13(RP13),RB12,RB7,RB6(RP6) input
    TRISC = 0x01A4;             // ポートC RC8(RP24),RC7(RP23),RC5(RP21),RC2(RP18) input
    CLKDIV = 0;                 // CPU Peripheral Clock Ratio=1:1
    AD1PCFG = 0xFFFF;           // AN1-12input pin = Digital mode

    // INT1(RP12 RB12) AD -DRDY0 setup
    RPINR0bits.INT1R = 12;  // RP12(RB12)
    int1_int_disable();
    int1_level_set(4);    // Int priority
    int1_edge_neg(); // Negative Edge
    // INT0(RB7) GPS +PPS setup
    int0_int_disable();
    int0_level_set(4);    // INT Priority
    int0_edge_pos(); // Positive Edge
    // ChangeNotification INT, CN20(RC8), SPI2(Overo) CS0
    cnint_int_disable();
    CNEN2bits.CN20IE = 1;
    CNPU2bits.CN20PUE = 1;  // Pullup ON
    cnint_level_set(4);

    adint_cnt = 0;
    ad_cs_dis();
    sts_set(STS_NOSYNC);
    pps_intf_clear();
    DRDY_OUT = 1;
    spi_cmd_sts = CMD_STS_NONE;
    
    // GPS UART1初期設定　BRGH=Standard mode
    // 9600bps 8bit nonparity 1stop nonflowcnt.
    //                           5432109876543210
    unsigned int U1MODEvalue = 0b1000100000000000;
    unsigned int U1STAvalue =  0b0000010000000000;
//    OpenUART1( U1MODEvalue, U1STAvalue, baudrate1);
    uart1_init( U1MODEvalue, U1STAvalue, baudrate1);
    // DEBUG UART2初期設定　BRGH=Standard mode
    // 9600bps 8bit nonparity 1stop nonflowcnt.
    //                           5432109876543210
    unsigned int U2MODEvalue = 0b1000100000000000;
    unsigned int U2STAvalue =  0b0000010000000000;
    uart2_init( U2MODEvalue, U2STAvalue, baudrate2);


    spi1_init();
    spi2_init();
    spi1_int_disable();
    spi2_int_disable();
    spi2_putc(SPI_HEAD_CHAR); // 一番最初のデータ送信のためにセットしておく

    PDEBUG("START\r\n");
//    sprintf(sz, "sizeof(ADBUF)=%d\r\n", ADBUF_LEN);
//    PDEBUG(sz);

    // GPS UART1 受信割り込み許可
    uart1_rx_int_enable();
    uart1_set_int_level(4);  // 割り込みレベル　デフォルト4
    // DebugOut UART2 受信割り込み許可
    uart2_rx_int_enable();
    uart2_set_int_level(4);  // 割り込みレベル　デフォルト4
PDEBUG("*");

    //**** 割り込み許可 **************************************************
    // CPU割り込み優先レベル デフォルト0　ペリフェラルはデフォルト4
//    SRbits.IPL = 2;
    SET_CPU_IPL( 2 );
PDEBUG("*");

    // ublox起動待ち
    delay_ms(100);
    // UBXプロトコルのみにする
    ublox_rcvbuf_clear();
    ublox_send_cfg_prt(BAUDRATE1);
    ublox_rcv_ack();
    // TimePulse2にAD用クロック出力する
    ublox_rcvbuf_clear();
    ublox_send_cfg_tp5_timepulse2();
    ublox_rcv_ack();
PDEBUG("*");
    // TimePulse1をGPS時刻fixしなくても出力する
    // pulse幅 unlocked=500msec locked=100msec
    ublox_rcvbuf_clear();
    ublox_send_cfg_tp5_timepulse1();
    ublox_rcv_ack();
PDEBUG("*");
    // NAV-TIMEUTC出力させる
    ublox_send_cfg_msg();
    ublox_rcv_ack();
PDEBUG("*");

    // AD初期化
    ad_reset_dis();
    ad_start_dis();
    ad_cs_dis();
    //
    ad_reset(); // -AD RESET
    // WAIT 2^16 AD system clocks
    delay_ms(100);
PDEBUG("*");
    //
    ad_cs_init();
    ad_init();
    pga_init();
    
    ring_init();
PDEBUG("*");
    // AD SPI1　割り込み設定のみ　許可しない
    // INT1ハンドラ内で有効にされる
    spi1_intf_clear();
    spi1_set_int_level(4);
    // DataOut SPI2　割り込み許可しない CSのCN INTで許可するので
    spi2_intf_clear();
    spi2_set_int_level(4);
PDEBUG("*");

    /**** ペリフェラル割り込み許可 INT1/INT0/CN ************************/
    // INT1割り込み許可
    int1_intf_clear();
    int1_int_enable();    // Int enable
PDEBUG("*");
    // INT0割り込み許可
    int0_intf_clear();
    int0_int_enable();    // Int enable
PDEBUG("*");
    // ChangeNotification INT許可 SPI2(Overo) CS
    cnint_intf_clear();
    cnint_int_enable();
PDEBUG("*\r\n");

    while(1) {
        // GPSから受信してデコード
        ublox_rcv();
        if (gpsNow.valid == 0x07 && sts_get() == STS_NOSYNC) {
            // 時刻有効
            sts_set(STS_SYNCNOW0);
        }
        // DebugOutに1secデータ出力
        if (ring_read2_get() != ring_write_get()) {
            // タイムスタンプget
            gps = adbuf_time_get(ring_read2_get());
            sprintf(sz, "%04u/%02d/%02d %02d:%02d:%02d.%09ld,%6lums,%6luns,%02X",
                gps->year, gps->month, gps->day, gps->hour, gps->min, gps->sec, gps->nano,
                gps->tow, gps->tacc, gps->valid
            );
            PDEBUG(sz);

#if 1
            // AD data get
            char ch;
            long    *lp;
            lp = adbuf_1sec_get(ring_read2_get());
            for(ch=0; ch<AD_CHNUM; ch++) {
                sprintf(sz, ",%+7ld", lp[ch]);
                PDEBUG(sz);
            }
            PDEBUG("\r\n");
#endif
#if 0
            ad = adbuf_get(ring_read2_get());
            sprintf(sz, "%02X %02X\r\n", ad->sumb, ad->suma);
            PDEBUG(sz);
#endif
            ring_read2_plus();
        }
        // SPIからコマンド受信
        if (spi_cmd_sts_get() == CMD_STS_RCVD) {
#ifdef DEBUG_PRINT
            PDEBUG("CMD RCV:");
            for(i = 0; i < SPI_RX_LEN; i++) {
                sprintf(sz, " %02X", spi_rx_buf[i]);
                PDEBUG(sz);
            }
            PDEBUG("\r\n");
#endif
            // コマンドに対応する処理
            spi_cmd_proc();
            // コマンド受信再開
            spi_cmd_sts_set(CMD_STS_NONE);
        }

    }
    
    return (EXIT_SUCCESS);
}
/*
 * INT0 GPS 1PPS割り込みハンドラ
 * 
 *
 */
void __attribute__((interrupt, no_auto_psv, shadow)) _INT0Interrupt(void)
{
    // AD STARTピン=H
    ad_start_ena();

    int0_intf_clear();
    pps_intf_set();
    /****AD同期!!!!!!!!!!!!!!!*/
    // AD個数カウンタ=SAMP_FREQ-1
    adint_cnt = AD_SAMPLE - 1;

    // AD START信号による同期
    if (sts_get() == STS_SYNCNOW0) {
        // AD STARTピン=L
        ad_start_dis();
        sts_set(STS_SYNCNOW1);
    } else if (sts_get() == STS_SYNCNOW1) {
        // AD STARTピン=H　関数先頭で行っているのでここではやらない
        sts_set(STS_SYNCWAIT);
    } else if (sts_get() == STS_SYNCWAIT) {
        sts_set(STS_SYNCED);
    }
}

/*
 * INT1(RP12) AD DRDY0割り込みハンドラ
 * Negativ Edge
 * SPI1受信開始してSPI1割り込み有効にする
 *
 */
void __attribute__((interrupt, no_auto_psv)) _INT1Interrupt(void)
//void _ISR _INT1Interrupt(void)
{
    int1_intf_clear();
    if (!ad_is_drdy0_enable()) {
        return;
    }

    adint_ch = 0;
    adint_sts = ADSTS_H;
    if (adint_cnt == 0) {
        my_memset(adint_add, 0, sizeof(adint_add));
    }

    ad_cs(adint_ch);    // CSx=L
    spi1_rx_overrun_clear();
    // 前に受信したデータをクリア
    spi1_getc();
    // SPI割り込みON
    spi1_intf_clear();
    spi1_int_enable();
    // 送信
    spi1_putc(0);

}
/*
 * AD SPI1 Interrupt Handler
 * ADから受信したデータをADバッファに格納する
 * 6CHぶん受信したらSPI1割り込みをOFF
 * 
 */
//void _ISR _SPI1Interrupt(void)
void __attribute__((interrupt, auto_psv)) _SPI1Interrupt(void)
{
    static unsigned char in[3];
    unsigned char c;
    long    l;
//int i;
//unsigned char   *ptr;

    spi1_intf_clear();
    c = spi1_getc();
    spi1_rx_overrun_clear();

    switch(adint_sts) {
        case ADSTS_H:
            in[2] = c;
            adint_sts = ADSTS_M;
            spi1_putc(0);
            break;
        case ADSTS_M:
            in[1] = c;
            adint_sts = ADSTS_L;
            spi1_putc(0);
            break;
        case ADSTS_L:
            // CSx=H
            ad_cs_dis();
            in[0] = c;
            l = byte3_to_long(in);
            // バッファに書きこみ
            adbuf_write(ring_write_get(), adint_ch, adint_cnt, &l);
            // 1秒平均用加算
            adint_add[adint_ch] += l;
            // CH進める
            adint_ch++;
            if (adint_ch >= AD_CHNUM) {
                // 全CHデータgetした
                adint_cnt++;
                if (adint_cnt == AD_SAMPLE-1) {
                    // ADバッファにタイムスタンプ付ける
                    adbuf_time_write(ring_write_get(), &gpsNow);
                } else if (adint_cnt >= AD_SAMPLE) {
                    // 1secぶんのデータgetした
                    // DRDY->H DRDY=Lの時でもDRDYのエッジを作るため
                    DRDY_OUT = 1;
                    // 1sec平均する
                    char ch;
                    for(ch = 0; ch < AD_CHNUM; ch++) {
                        adint_add[ch] /= adint_cnt;
                    }
                    // 1secバッファに書きこみ
                    adbuf_1sec_write(ring_write_get(), adint_add);
#if 0
ptr = (unsigned char*)adbuf_get(ring_write_get());
ptr += 62;
            for(i = 0; i < 900; i++) {
                suma += *ptr++;
                sumb += suma;
            }
#endif
                    // バッファ書き込み位置＋１
                    ring_write_plus();
                    // 次のADバッファをゼロクリア
                    my_memset(adbuf_get(ring_write_get()), 0, ADBUF_LEN);
                    adint_cnt = 0;
                    // DRDY->L
// TODO: waitカウンタで制限
                    DRDY_OUT = 0;
                }
                spi1_int_disable(); // SPI割り込みOFF
                adint_sts = ADSTS_IDLE;
            } else {
                // 次のCHへ
                ad_cs(adint_ch);    // CSx=L
                // 送信
                spi1_putc(0);
                adint_sts = ADSTS_H;
            }

            break;
        default:
            spi1_int_disable(); // SPI割り込みOFF
            adint_sts = ADSTS_IDLE;
            break;
    }

}
static unsigned char    suma;
static unsigned char    sumb;

/*
 * DataOut SPI2 Interrupt Handler
 * TX FIFOに1byteの空きが出来ると発生
 */
void __attribute__((interrupt, no_auto_psv)) _SPI2Interrupt(void)
{
    unsigned char   rx;

    spi2_intf_clear();
    // 送信 AD DATA
    while(SPI2_TBF == 0) {
        // checksum
        if ((ADBUF_LEN - dataout_cnt_get()) == ADBUF_OFS_SUM) {
            // checksum書き込み
            adbuf_sum_write(ring_read1_get(), &suma, &sumb);
        }
        suma += (unsigned char)*dout_ptr;
        sumb += suma;

        if (dataout_cnt_get() <= 0) break;
        spi2_putc(*dout_ptr++);
        dataout_cnt_minus();
    }
    // 受信 CMD
    while(SPI2_SRXMPT == 0) {
        rx = spi2_getc();
        if (spi_cmd_sts == CMD_STS_NONE && spi_rx_cnt < SPI_RX_LEN) {
            spi_rx_buf[spi_rx_cnt] = rx;
            spi_rx_cnt++;
            if (spi_rx_cnt >= SPI_RX_LEN) {
                if (spi_rx_buf[CMD_OFS_CMDCODE] != 0) {
                    spi_cmd_sts = CMD_STS_RCVD;
                }
            }
        }
    }

    // オーバーランエラーならばクリアする
    if (SPI2STATbits.SPIROV == 1) SPI2STATbits.SPIROV = 0;

}
/*
 * Overo SPI2 CS Input Change Notification
 * Interrupt Handler
 *
 */
void __attribute__((interrupt, no_auto_psv)) _CNInterrupt(void)
{
    cnint_intf_clear();
    if (SPI2CSpin == 1) {
        // CS=H
        // SPI2割り込みOFF
        spi2_int_disable();
        // SPI2をリセットしてTX/RX FIFOクリア
        SPI2STATbits.SPIEN = 0;
        asm("nop");
        SPI2STATbits.SPIEN = 1;
        asm("nop");
        SPI2BUF;    // RBF->0
        // $->TX FIFO
        spi2_putc(SPI_HEAD_CHAR);
        // リングバッファ読み込み位置+1
        ring_read1_plus();
        // 送信データがリングバッファにあるとき
        if (ring_num1_get() > 0) {
            // DRDY->L
            DRDY_OUT = 0;
        }
    } else {
        // CS=L
        // リングバッファ読み込み位置get
        dout_ptr = (char*)adbuf_get(ring_read1_get());
        // SPI送信の一番最初
        dataout_cnt_set(ADBUF_LEN);
        suma = 0;
        sumb = 0;
        // SPI2 TX FIFOに送信データ書き込み
        while(SPI2_TBF == 0) {
            suma += (unsigned char)*dout_ptr;
            sumb += suma;
            if (dataout_cnt_get() <= 0) break;
            spi2_putc(*dout_ptr++);
            dataout_cnt_minus();
        }
        if (spi_cmd_sts == CMD_STS_NONE) {
            // SPI受信文字数カウンタクリア
            spi_rx_cnt = 0;
        }
        // SPI2割り込みON
        spi2_int_enable();
        // DRDY=H
        DRDY_OUT = 1;
    }
}
