﻿using System;
using System.IO;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using SCFiler2.ItemInterface;

namespace SCFiler2.FileSystem {
	class ClipboardManager {

		/// <summary>
		/// エクスプローラとの受け渡しに使うための、クリップボード操作の種類
		/// </summary>
		public enum EffectType : int {
			None = 0,
			Copy = 1,
			Cut = 2,
			Link = 4,	//LinkはSCFilerではしばらく対応しない予定
		}

		/// <summary>
		/// クリップボードの中にファイルがあるかどうか
		/// </summary>
		/// <returns>ファイルがクリップボードの中にあったらtrue</returns>
		public static bool IsConatinFiles() {
			if (Clipboard.ContainsFileDropList()) {
				return true;
			} else {
				return false;
			}
		}

		/// <summary>
		/// クリップボードの中のアイテムをFilerItemのリスト形式で取得
		/// </summary>
		/// <returns>IFilerItemのリスト。有効なファイルが入っていなかったらnull</returns>
		public static List<IFilerItem> GetFilesInClipboard() {
			IDataObject obj = Clipboard.GetDataObject();
			if (obj.GetDataPresent(DataFormats.FileDrop)) {
				string[] files = (string[])obj.GetData(DataFormats.FileDrop);

				return FilerItem.GetFileListFromStringArray(files);
			} else {
				return null;
			}
		}

		/// <summary>
		/// クリップボードに入っているファイルが"コピー"か"切り取り"かを判断する
		/// </summary>
		/// <returns>EffectType</returns>
		public static EffectType GetDropEffect() {
			IDataObject obj = Clipboard.GetDataObject();
			object effect = obj.GetData("Preferred DropEffect");
			if (effect == null) {
				return EffectType.None;
			}
			if (effect.GetType() == typeof(MemoryStream)) {
				MemoryStream stream = (MemoryStream)effect;
				EffectType type = (EffectType)BitConverter.ToInt32(stream.ToArray(), 0);
				if ((type & EffectType.Copy) == EffectType.Copy) {
					return EffectType.Copy;
				} else if ((type & EffectType.Cut) == EffectType.Cut) {
					return EffectType.Cut;
				} else {
					return type;
				}
			} else {
				return EffectType.None;
			}
		}

		/// <summary>
		/// クリップボードに"コピー"や"切り取り"を指定してコピー
		/// </summary>
		/// <param name="effect">"コピー"や"切り取り"などの操作の種類</param>
		/// <param name="itemsToSend">クリップボードに送るアイテムのリスト</param>
		public static void SendToClipboard(EffectType effect, List<IFilerItem> itemsToSend) {
			//クリップボードに渡すためにstring[]へ変換
			string[] files = new string[itemsToSend.Count];
			for (int i = 0; i < itemsToSend.Count; i++) {
				files[i] = itemsToSend[i].FullName;
			}

			//クリップボードに渡すDataObjectの作成
			IDataObject objToSend = new DataObject(DataFormats.FileDrop, files);
			MemoryStream effectData = new MemoryStream();
			byte[] effectByte = new byte[] { (byte)effect, 0, 0, 0 };
			effectData.Write(effectByte, 0, effectByte.Length);
			effectData.SetLength(effectByte.Length);
			objToSend.SetData("Preferred DropEffect", effectData);
			Clipboard.SetDataObject(objToSend);
		}

	}
}
