#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2010, mshio <mshio@users.sourceforge.jp>
#
# This program is free software: you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import fontforge
import optparse
import os
import sys
import time
import xml.parsers.expat

COPYRIGHT = 'Copyright: (c) 2008-%d mshio <mshio@users.sourceforge.jp>'

class SvgCustomizer:
  '''
  FontForge が書き出した SVG ファイルの内容を書き換え、
  ウェブブラウザで表示できるようにするためのクラスです。

  インスタンスを生成した後、execute メソッドを呼び出してください。

  This class is for converting SVG files that are generated by FontForge
  to ones that can be displayed by a web browser.

  After getting an instance, call its execute method.
  '''

  def __init__(self, fill_color, comment):
    '''
    コンストラクタ

    引数：
      fill_color -- SVG のパスを塗りつぶすための色。文字列
      comment    -- コピーライトコメントを付けるかどうか。True/False

    Constructor

    Arguments:
      fill_color -- color name for filling svg pathes.
      comment    -- True if adding copyright comment.
    '''
    self.fill_color = fill_color
    self.with_comment = comment

  def execute(self, glyph, file):
    '''
    FontForge が書き出した SVG ファイルを解析し、目的の形式に書き換えます。

    引数：
      glyph -- FontForge のもつグリフオブジェクト
      file  -- FontForge の書き出した SVG ファイルのファイル名。文字列

    Parses the SVG file that is output by FontForge, and outputs a new 
    SVG file that can be displayed by a web browser.

    Arguments:
      glyph -- glyph object
      file  -- SVG file name
    '''
    self.setup(glyph)
    try:
      c = self.get_contents(file)
      self.out = open(file, 'w')
      self.parser.Parse(c, 1)
    except:
      print sys.exc_info()[0]
    self.out.close()

  def setup(self, glyph):
    '''内部にもつ XML パーサーのセットアップを行います。'''
    self.glyph = glyph
    self.parser = xml.parsers.expat.ParserCreate()
    self.parser.XmlDeclHandler = self.start_xml_declaration
    self.parser.StartDoctypeDeclHandler = self.start_doctype_declaration
    self.parser.StartElementHandler = self.start_element
    self.parser.EndElementHandler = self.end_element

  def get_contents(self, file):
    '''指定したファイルの内容を読み込んで返します。'''
    h = open(file, 'r')
    c = h.read()
    h.close()
    return c

  def start_xml_declaration(self, version, encoding, standalone):
    '''XML 宣言部分のパースと書き出しを行います。'''
    self.out.write('<?xml')
    if version:
      self.out.write(' version="%s"' % version)
    if encoding:
      self.out.write(' encoding="%s"' % encoding)
    if standalone != -1:
      val = 'yes' if standalone == 1 else 'no'
      self.out.write(' standalone="%s"' % val)
    self.out.write('?>')

  def start_doctype_declaration(self, doctype, sys_id, pub_id, has_subset):
    '''DOCTYPE 部分のパースと書き出しを行います。'''
    self.out.write('<!DOCTYPE %s' % doctype)
    if pub_id:
      self.out.write(' PUBLIC "%s"' % pub_id)
    if sys_id:
      self.out.write(' "%s"' % sys_id)
    self.out.write('>')

  def start_element(self, name, attrs):
    '''
    各要素の開始部分を読み込み、その部分の書き出しを行います。
    要素名が 'svg' の場合、xmlns 属性を付加し、さらに viewBox の値を変更し、
    またさらにコピーライトの出力が指示されている場合は、svg 要素の後に
    metadata 要素を追加し、そこにコピーライトを出力します。
    要素名が 'path' の場合は、fill 属性の有無をチェックし、
    あれば、その値を指定の値に変更します。
    '''
    self.out.write('<%s' % name)
    is_svg = name == 'svg'
    is_path = name == 'path'
    if is_svg:
      self.out.write(' xmlns="http://www.w3.org/2000/svg"')
    for k in attrs.keys():
      if is_svg and k == 'viewBox':
        org = attrs[k].split()
        w = self.glyph.width - int(org[0])
        # here, using magic numbers at the values of top and height.
        # if using this script for some other fonts,
        # you might have to change these values.
        val = '%s -100 %d 1200' % (org[0], w)
        self.out.write(' %s="%s"' % (k, val))
      elif is_path and k == 'fill':
        self.out.write(' %s="%s"' % (k, self.fill_color))
      else:
        self.out.write(' %s="%s"' % (k, attrs[k]))
    self.out.write('>')
    # output copyright string
    if is_svg and self.with_comment:
      c = COPYRIGHT % time.localtime()[0]
      self.out.write('<metadata><![CDATA[ %s ]]></metadata>' % c)

  def end_element(self, name):
    '''各要素の終了部分を読み込み、その部分の書き出しを行います。'''
    self.out.write('</%s>' % name)

#
# functions
#
def make_svg(customizer, glyph, output_dir):
  '''
  SVG ファイルの生成と書き換えを行います。

  まず FontForge の export 機能で SVG ファイルを生成し、
  次いで SvgCustomizer でその内容を書き換えます。

  引数:
    customizer -- SvgCustomizer のインスタンス
    glyph      -- 書き出す対象のグリフオブジェクト
    output_dir -- 出力先のディレクトリ

  Exports a SVG file from the specified glyph, and
  outputs new one that can be displayed with a web browser.

  Arguments:
    customizer -- an instance of SvgCustomizer
    glyph      -- target glyph object
    output_dir -- directory for SVG files that will be outputed by this script
  '''
  path = '%s/%04x.svg' % (output_dir, glyph.unicode)
  print path
  glyph.export(path, 1)
  customizer.execute(glyph, path)


if __name__ == '__main__':

  class InvalidArgumentError(Exception):
    '''コマンドライン引数が正しくない場合に生成される例外です。'''
    def __init__(self, value):
      self.value = value

  def parse_args():
    '''
    コマンドライン引数の処理をします。
    第一引数はフォントファイル、第二引数は保存先のディレクトリです。
    オプションは下記のとおりです。

     -f, --fill :
            SVG のパスを塗りつぶす色を指定します。
            省略すると、black が指定されたことになります。
     -c, --copyright :
            コピーライト（内容は固定）を含める場合に指定します。
            コピーライトは、metadata のデータとして挿入されます。
     -g, --glpyh :
            フォントが持つ全グリフではなく、特定の文字のみの
            SVG ファイルを出力したい場合に指定します。
            指定は、16 進数の文字コード（Unicode）で行います。

    Processes the arguments of command line.
    The first of them is font file path, and the second of them is path of
    directory in which the svg files will be stored.
    The options are like this:

     -f, --fill:
            The color name with which fill the svg pathes.
            Default is 'black.'
     -c, --copyright:
            When this option is specified, the copyright is written as 
            'metadata' in each svg files.
     -g, --glyph:
            Character code (hex). When this option is specified, 
            this script will output only a SVG file of the specified character.
            When not, it will output ones of all glyphs.
    '''
    usage = 'usage: %prog [-f color] [-c] fontfile directory'
    p = optparse.OptionParser(usage=usage)
    p.add_option('-f', '--fill', dest='color', default='black',
                 help='name of color that is used when filling the svg pathes')
    p.add_option('-c', '--copyright', dest='copyright', 
                 action='store_true', default=False,
                 help="write mshio's copyright in each output files")
    p.add_option('-g', '--glyph', dest='code', default=None,
                 help='char code that wants to be output')
    (opt, args) = p.parse_args()

    if len(args) != 2:
      p.error('incorrect number of arguments')
      raise InvalidArgumentError

    return (args[0], args[1], opt.color, opt.copyright, opt.code)


  try:
    (font_path, output_dir, color_name, comment, code) = parse_args()
  except InvalidArgumentError:
    sys.exit(1)

  char_code = int(code, 16) if code else None

  customizer = SvgCustomizer(color_name, comment)
  font = fontforge.open(font_path)
  if char_code:
    make_svg(customizer, font[char_code], output_dir)
  else:
    for g in font:
      if g[0] != '.':
        glyph = font[g]
        if glyph.unicode > 0:
          make_svg(customizer, glyph, output_dir)
