
#include "utils/option_parser.hpp"
#include "utils/file_path.hpp"
#include "utils/number_format.hpp"

#include <fstream>
#include <vector>
#include <string>

#include "basic_type.hpp"

#include "ines_file.hpp"
#include "nes_memory_mapper.hpp"
#include "instruction.hpp"
#include "reference.hpp"

#include "utils/hex_form.hpp"

#include "front_end.hpp"
#include "display_module.hpp"

/**
   Function of sau decompiler
   1. show nes info
   2. show hex dump
   3. separate interactive code data
   4. show assembly
   5. show cico 
*/

/* This is application command line front end.
   
 */
class ApplicationEntry
{
	enum OptionFlagID {
		FLAG_UNDEF=0,
        LOAD_CDL_FILE,
		DISP_INFO_NES,
        EDIT_CODE_DATA,
		DISP_ASSEMBLY,
		DISP_CODE,
		DISP_HELP,
        DISP_VERBOSE,
		FLAG_MAX,
	};
	bool m_flags[FLAG_MAX];
	std::vector<FilePath> m_files;
	FilePath m_CDLfile;
	Word m_start_address;
public:
    const char* usageMessage()
    {
        static char mes[] = "NAME\n"
            "\tsau - sau-decompiler for NES\n"
            "\n"
            "SYNOPSIS\n"
            "\tsau [-i|-e|-a|-c] [-vh]\n"
            "\t    [-l CDLfile] nesfile\n"
            "\n"
            "OPTIONS\n"
            "\t-l CDLfile log to edit or disassemble.\n"
            "\t-i Show NES File infomations\n"
            "\t   (header,origin,interrupt vector)\n"
            "\t-e Code Data Log edit mode\n"
            "\t-a Show disassembled mnemonic\n"
            "\t-c Show decompiled code\n"
            "\t-h Show help\n"
            "\t-v Verbose mode\n"
            ;
        return mes;
    }
    
	ApplicationEntry(int argc, char** argv)
            :m_CDLfile("")
	{
		for (int i=0; i<FLAG_MAX; ++i) {
			m_flags[i]=false;
		}
		m_start_address =0x8000;
		CharacterOptionParser opt("l:ieacvh?");
		opt.parse(argc,argv);
		
		for (auto i=opt.begin(); i!=opt.end(); ++i) {
			switch (i->first) {
            case 'l':
                m_flags[LOAD_CDL_FILE] = true;
                m_CDLfile = i->second;
                break;
			case 'i': m_flags[DISP_INFO_NES] = true; break;
			case 'e': m_flags[EDIT_CODE_DATA] = true; break;
			case 'a': m_flags[DISP_ASSEMBLY] = true; break;
            case 'c': m_flags[DISP_CODE] = true; break;
            case 'v': m_flags[DISP_VERBOSE] = true; break;
			case 'h': m_flags[DISP_HELP] = true; break;
			case '?': m_flags[DISP_HELP] = true; break;
			default: break;
			}
		}
		if (opt.rest_arguments.empty()) {
            m_flags[DISP_HELP] = true;
		} 
		if (opt.bad()) {
            std::cerr << opt.invalidOptionWarningMessage()
                      << std::endl;
		}
		for (std::vector<std::string>::iterator
                 i = opt.rest_arguments.begin();
			 i != opt.rest_arguments.end();
			 ++i) {
			FilePath n(*i);
            if (*i == "?") {
                m_flags[DISP_HELP] = true;
            } else if (n.exists()) {
				m_files.push_back(n);
			} else {
				std::cerr << "File '" << n.to_s() << "' is not found.";
            }
		}
	}
    
	/** execute main flow of decompilation.*/
	void run()
	{
        if (m_flags[DISP_HELP]){
            std::cerr << usageMessage() << std::endl;
        }
        for (unsigned i =0; i<m_files.size(); ++i) {
            if (m_files.size()>1){
                std::cout << m_files[i].fileName()
                          << std::endl;
            }
            processFile(m_files[i]);
        }
	}
	void verbose(std::string mes, bool endline=true)
    {
        if (m_flags[DISP_VERBOSE]) {
            std::cerr << mes;
            if (endline) {
                std::cerr << std::endl;
            }
        }
    }
	void processFile(FilePath file_name)
	{
		// phase 1: load nes image
        verbose("phase 1: load nes image");
		INesFile nes;
		nes.load(file_name);
        
		// phase 2: determine bank origin.
        verbose("phase 2: determine bank origin.");
		std::vector<Word> origins
            = OriginScanner::scanOrigin(nes, 10);
		NesMemoryMapper mapper(nes, origins);
		
        if (m_flags[DISP_INFO_NES]) {
			displayINesFile(std::cout, nes,
                            mapper.interrupts());
			displayOrigins(std::cout, origins);
		}
        if (m_files.size()>1){
            verbose("skipped because of multi-file mode");
            return;
        }
        // phase 2.1: separate code and data.
        verbose("phase 3: separate code and data.");
        
        PByteSequence cdflags;
        if (m_flags[LOAD_CDL_FILE]) {
            cdflags = CodeDataModule::load(m_CDLfile);
            if (cdflags->size() != nes.prgSize()){
                std::cerr << "CDL size is not matched with NES file." << std::endl;
                return;
            }
        } else {
            cdflags = CodeDataModule::parse(mapper);
            verbose("Parsed Code Data flags size: ",false);
            verbose(HexForm(cdflags->size()).to_s());
            assert(cdflags->size() == nes.prgSize());
        }
        if (m_flags[EDIT_CODE_DATA]) {
            CodeDataModule::edit(cdflags,
                                 nes.prgBanks(), origins,
                                 m_CDLfile);
        }
		// result 3.1: reference table.
        if (m_flags[DISP_CODE]){
            displayCodeDataStats(std::cout, *cdflags);
            displayCodeDataInfo(std::cout, *cdflags, origins);
        }
		// result 2.1: raw assembly
		if (m_flags[DISP_ASSEMBLY]) {
			displayRawAssembly(nes.prgBanks(), origins);
		}
        verbose("finished normally.");
	}
};

int main(int argc,char** argv)
{
	ApplicationEntry ent(argc,argv);
	ent.run();
	return 0;
}
	

	
