#pragma once

#include "basic_type.hpp"
#include <vector>
#include <map>
#include <iostream>
#include <boost/shared_ptr.hpp>
#include "opcode.hpp"

class Instruction;
typedef boost::shared_ptr<Instruction> PInstruction;
typedef std::vector<PInstruction> InstructionSequence;
typedef std::map<MemoryID, PInstruction> InstructionMap;

class InstructionHelper
{
public:
    static bool isRomAccess(const Instruction& instr);
    static bool isTransfer(const Instruction& instr);
};

class Instruction
{
    enum Constants {
        VALID_SIGNATURE=0x1234,
    };
	std::vector<Byte> m_bytes;
	const Word m_address;
    Opcode m_op;
	Word m_operand;
    MemoryID m_operand_id;
    Word m_signature;
public:
	static Instruction getInvalid() { return Instruction(0,0,0,0); }
	/** <<constructor>> by three bytes.*/
	Instruction(Word address, Byte byte, Byte low, Byte high)
			:m_address(address),
			 m_op(OpcodeHelper::decode(byte)),
             m_operand_id(0x0ABC),
             m_signature(VALID_SIGNATURE)
	{
		init(byte, low, high);
	}
	/** <<constructor>> by byte sequence*/
	Instruction(Word address, const ByteSequence& bank, unsigned idx)
			:m_address(address),
			 m_op(OpcodeHelper::decode(
                          (idx<bank.size()) ? bank[idx] : 0)),
             m_operand_id(0x0ABC),
             m_signature(VALID_SIGNATURE)
	{
		initFromVector(bank,idx);
	}
	/** <<copy>> */
	Instruction(const Instruction& that)
			:m_address(that.m_address),
			 m_op(that.m_op),
             m_operand_id(that.m_operand_id),
             m_signature(VALID_SIGNATURE)
	{
		initFromVector(that.m_bytes, 0);
	}
    /** pseudo machine instruction.*/
    static PInstruction createInstruction(
            Word address,OpcodeType type, AddressingMode mode,
            Word operand)
    {
        PInstruction ptr(new Instruction(
                                 address, type, mode, operand));
        return ptr;
    }
private:
    Instruction(Word address,OpcodeType type,
                AddressingMode mode, Word operand)
            :m_address(address),
             m_op(type, mode),
             m_operand(operand),
             m_signature(VALID_SIGNATURE)
    {
    }
public:
	/*query*/
	Word address()const { inv(); return m_address; }
	OpcodeType type()const { inv(); return m_op.type(); }
	AddressingMode mode()const { inv(); return m_op.mode(); }
	Word operand()const { inv(); return m_operand; }
    MemoryID operandID()const { return m_operand_id; }
    void setOperandID(MemoryID id) { m_operand_id = id; }
    
	const char* mnemonic()const
    {
        return OpcodeHelper::mnemonic(type());
    }
	unsigned bytelength()const {
        inv();
		if (type() == opcode::BRK) {
			return 2;
		} else {
			return OpcodeHelper::bytelength(mode());
		}
	}
	Word next()const { inv(); return m_address+bytelength(); }
    const ByteSequence& bytes()const { return m_bytes; }
	void display(std::ostream& os)const;
	friend std::ostream& operator<<(std::ostream& os,
                                    const Instruction& rh)
	{
		rh.display(os);
		return os;
	}
private:
    /** invariant assertion.*/
    void inv() const { assert(m_signature == VALID_SIGNATURE); }
    /** if index is out of bank range, get 0 as data. */
	void initFromVector(const std::vector<Byte>& bank, unsigned idx)
	{
		Byte byte, low, high;
		unsigned size = bank.size();
		byte = (idx<size) ? bank[idx] : 0;
		low = (idx+1<size)? bank[idx+1] : 0;
		high = (idx+2<size)? bank[idx+2] : 0;
		init(byte, low, high);
	}
	void init(Byte byte, Byte low, Byte high)
	{
		m_bytes.push_back(byte);
		switch (mode()) {
		case opcode::ABS: case opcode::ABS_X: case opcode::ABS_Y:
		case opcode::IND:
			m_operand = bit8::lowHigh(low,high);
			m_bytes.push_back(low);
			m_bytes.push_back(high);
			break;
		case opcode::IMM: case opcode::ZERO:
		case opcode::ZERO_X: case opcode::ZERO_Y:
		case opcode::PRE_IND: case opcode::POST_IND:
			m_operand = low;
			m_bytes.push_back(low);
			break;
		case opcode::REL:
			if (low < 0x80) {
				m_operand = next() + low;
			} else {
				m_operand = next() - (0x100 - low);
			}
			m_bytes.push_back(low);
			break;
		default:
			if (type() == opcode::UND) {
				m_operand = byte;
			} else if (type() == opcode::BRK) {
				m_operand = low;
				m_bytes.push_back(low);
                //m_operand = 0;
			} else {
				m_operand = 0;
			}
			break;
		}
	}
};
