#pragma once

#include <iterator>				// for std::iterator
#include <boost/shared_ptr.hpp>		 // for boost::shared_ptr

#include <iostream>                  // for std::cout
#include <string>					 // for std::string
#include <algorithm>                 // for std::for_each
#include <vector>
#include <map>
#include <set>

//typedef std::string T;
//typedef unsigned TL;
template<typename T, typename TL>
class Graph
{
public:
	class Vertex;
	class Edge;
	
	typedef boost::shared_ptr<Edge> PEdge;
	
	typedef std::map<T, PEdge>  EdgeSubMap;
	typedef std::map<T, EdgeSubMap> EdgeMap;
	
	typedef std::vector<PEdge> EdgeList;
	
	typedef boost::shared_ptr<Vertex> PVertex;
	
	typedef std::map<T, PVertex> VertexMap;
	typedef typename VertexMap::iterator vert_iterator;
	typedef typename VertexMap::const_iterator cvert_iterator;
	
	typedef std::pair<T, unsigned> pair;
	typedef std::vector<pair> Path; //depth and value.
	
	/** This label provides iteration of vertices.
	 *  first is the value of vertex.
	 *  second is the depth level.
	 */
	class iterator
		: public std::iterator<std::forward_iterator_tag, T>
	{
	public:
		unsigned m_index;
		Path m_path; //depth and value.
		iterator(Path path) :m_index(0), m_path(path) { }
        iterator() :m_index(0) { }
		iterator(const iterator& that)
				:m_index(that.m_index), m_path(that.m_path)
		{ }
        iterator& operator=(const iterator& that)
        {
            m_index = that.m_index;
            m_path.clear();
            m_path = that.m_path;
            m_path.resize(that.m_path.size());
            for (unsigned i=0; i<that.m_path.size(); ++i) {
                m_path[i] = that.m_path[i];
            }
            return *this;
        }
		pair& operator*() { dereferencable(); return m_path[m_index]; }
		pair* operator->() { dereferencable(); return &m_path[m_index]; }
		bool operator==(const iterator& rh) { return remain() == rh.remain();}
		bool operator!=(const iterator& that) { return !(*this == that); }
		iterator& operator++() { ++m_index; return *this; }
	private:
		// invariant
		void dereferencable()const { assert(m_index < m_path.size()); }
		unsigned remain()const { return m_path.size() - m_index; }
	};
	
	/** This value represents the directed edge between two vertices.
	 *
	 * Edge is possesed by vertex,
	 * so if an end point vertex is destructed,
	 * the edge becomes undefined.
	 * copyable? No.
	 * changeable? value.
	 */
	class Edge
	{
		enum constant{
			VALID_SIGN = 0x1234, INVALID_SIGN,
		};
		
		Vertex* m_tail;
		Vertex* m_head;
		TL m_label;				// This can be used weight or type of edge.
		
		unsigned m_signature;
		Edge():m_signature(INVALID_SIGN) {}
	public:
		/* tail -> head */
		Edge(Vertex* tail, Vertex* head, TL label) 
				:m_tail(tail),m_head(head), m_label(label),
				 m_signature(VALID_SIGN) { }
		/** TL must have default label.*/
		Edge(Vertex* tail, Vertex* head)
				:m_tail(tail),m_head(head),
				 m_signature(VALID_SIGN) { }
		
		/** This is tail vertex of this edge.*/
		Vertex* tail()const { invariant(); return m_tail; }
		/** This is head vertex of this edge.*/
		Vertex* head()const { invariant(); return m_head; }
		/** This is getter method for edge label.*/
		TL label()const { invariant(); return m_label; }
		/** This setter method is used to assign the edge label.*/
		void setLabel(TL label) { invariant(); m_label = label; }
		/** <<Identity>> Although the labels differs, they are same edge.*/
		bool operator==(const Edge& rh)const
		{
			if (isValid() && rh.isValid()){
				return m_head == rh.m_head && m_tail == rh.m_tail;
			} else if (!isValid() && !rh.isValid()){
				return true;
			} else {
				return false;
			}
		}
		/** get invalid edge.*/
		static PEdge getInvalid() {
			static PEdge invalid(new Edge());
			return invalid;
		}
		/** check legal signature.*/
		bool isValid()const { return m_signature == VALID_SIGN; }
	private:
		/** This method is set to check invariant assertions.*/
		void invariant()const { assert(isValid()); }
	};
	
	/** This container allocate data in the graph.
	 *  copyable? No.
	 */
	class Vertex
	{
	private:
		enum constant{
			VALID_SIGN = 0x1234, INVALID_SIGN,
		};
		T m_value;
		EdgeList m_in_edges;
		EdgeList m_out_edges;
		unsigned m_signature;

		/** default is invalid.*/
		Vertex() :m_signature(INVALID_SIGN) {}
	public:
		/** This constructer is used to set entity to vertex */
		Vertex(const T& value):m_value(value),m_signature(VALID_SIGN) {}
		/** get contained value.*/
		T value()const { invariant(); return m_value; }
		/** <<getting>> edges*/
		const EdgeList& inEdges()const { return m_in_edges;}
		const EdgeList& outEdges()const { return m_out_edges;}
		/** add edge. */
		void add(PEdge e)
		{
			invariant();
			if (e->tail() == this) { addEdgeTo(e, m_out_edges); } 
			if (e->head() == this) { addEdgeTo(e, m_in_edges); } 
		}
		/** if v->u, then remove the edge.*/
		void remove(PEdge e)
		{
			invariant();
			if (e->tail() == this) { removeEdgeFrom(e, m_out_edges); } 
			if (e->head() == this) { removeEdgeFrom(e, m_in_edges); } 
		}
		/** clear both edge lists.*/
		void clear() { invariant(); m_in_edges.clear(); m_out_edges.clear(); }
		/** This query method represents whether this has been constructed.*/
		bool isValid()const { return m_signature == VALID_SIGN; }
		/** This class method provides singleton invalid. */
		static PVertex getInvalid()
		{
			static PVertex invalid(new Vertex);
			return invalid;
		}
	private:
		void addEdgeTo(PEdge e, EdgeList& list)
		{
			typename EdgeList::iterator i;
			i = std::find(list.begin(), list.end(), e);
			if (i == list.end()) { 
				list.push_back(e);
			}
		}
		/* called from remove: to get rid of the edge from the list.*/
		void removeEdgeFrom(PEdge e, EdgeList& list)
		{
			typename EdgeList::iterator i;
			i = std::remove(list.begin(), list.end(), e);
			list.erase(i, list.end());
		}

		/** This method is set to check invariant assertions.*/
		void invariant()const { assert(isValid()); }
	};
	
	VertexMap m_vertices;
	EdgeMap m_edges;
public:
	
	Graph() {}
	~Graph() {}
    /** <<copy>> */
    Graph(const Graph& that)
            :m_vertices(that.m_vertices), m_edges(that.m_edges)
    { }
	unsigned size()const { return m_vertices.size(); }
	
	bool empty()const { return m_vertices.empty(); }
	void clear()
	{
		while (!empty()) {
			removeVertex(m_vertices.begin()->second->value());
		}
	}
	
	/** O(1): check whether vertex exists or not.*/
	bool hasVertex(T v)const {return m_vertices.find(v) != m_vertices.end();}
	
	/** O(1): This method is used to create new vertex into the graph.*/
	void addVertex(T v)
	{
		if (!hasVertex(v)) {
			PVertex new_vertex(new Vertex(v));
			m_vertices[v] = new_vertex;
		}
		assert(hasVertex(v));
	}
	/** O(n): This method is used to remove vertex and its edges.*/
	void removeVertex(T v)
	{
		typename VertexMap::iterator iter = m_vertices.find(v);
		if (iter != m_vertices.end()) {
			PVertex vertex = iter->second;
			// This order is important:
			//  remove from edge map
			//  -> remove from Vertices' edge list.
			//  -> remove from vertex map.
			m_edges[v].clear();
			for (typename EdgeMap::iterator i=m_edges.begin();
				 i!=m_edges.end(); ++i) {
				for (typename EdgeSubMap::iterator j=i->second.begin();
					 j!=i->second.end(); ++j) {
					if (j->first == v) {
						i->second.erase(j);
						break;
					}
				}
			}
			removeEdgesFromOpposite(vertex.get(), vertex->inEdges());
			removeEdgesFromOpposite(vertex.get(), vertex->outEdges());
			vertex->clear();
			m_vertices.erase(iter);
		}
		assert(m_vertices.find(v) == m_vertices.end());
	}
	
	/** neighbour list of vertex.*/
	std::vector<T> predecessors(T v)const
	{
		PVertex vert = getVertex(v);
		const EdgeList& edges=vert->inEdges();
		std::vector<T> result;
		typename EdgeList::const_iterator i;
		
		for (i=edges.begin(); i!=edges.end();++i) { 
			result.push_back((*i)->tail()->value());
		}
		return result;
	}
	/** <<query>> but it generate vector by call.*/
	std::vector<T> successors(T v)const
	{
		PVertex vert = getVertex(v);
		const EdgeList& edges=vert->outEdges();
		std::vector<T> result;
		typename EdgeList::const_iterator i;
		
		for (i=edges.begin(); i!=edges.end();++i) {
			result.push_back((*i)->head()->value());
		}
		return result;
	}
	
	unsigned edgeSize()const
	{
		unsigned result=0;
		for (typename EdgeMap::const_iterator i=m_edges.begin();
			 i != m_edges.end(); ++i) {
			result += i->second.size();
		}
		return result;
	}
	/** This query method checks whether the directed edge exists.*/
	bool hasEdge(T tail, T head)const
	{
		typename EdgeMap::const_iterator i = m_edges.find(tail);
		if (i != m_edges.end()) {
			if (i->second.find(head) != i->second.end()) {
				return true;
			}
		}
		return false;
	}
    TL label(T tail, T head)const
    {
		typename EdgeMap::const_iterator i = m_edges.find(tail);
		if (i != m_edges.end()) {
            typename EdgeSubMap::const_iterator j;
            j = i->second.find(head);
			if (j != i->second.end()) {
				return j->second->label();
			}
		}
		return TL();
        
    }
	/** add edge, default label.*/
	// To get default value of TL, we call default constructor explicitly.
	void addEdge(T tail, T head)
	{
		addLabeledEdge(tail, head, TL());
	}
	/** add edge, and vertices if necessary.*/
	void addLabeledEdge(T tail, T head, TL label)
	{
		if (hasEdge(tail, head)) { return; }
		if (!hasVertex(tail)) { addVertex(tail); }
		if (!hasVertex(head)){ addVertex(head); }
		
		PVertex v = getVertex(tail);
		PVertex u = getVertex(head);
		
		PEdge e(new Edge(v.get(),u.get(),label));
		m_edges[tail][head]=e;
		v->add(e);
		u->add(e);
	}
	/** This method is used to remove the edge from the end points and list.*/
	void removeEdge(T tail, T head)
	{
		typename EdgeMap::iterator i = m_edges.find(tail);
        
		if (i != m_edges.end()) {
			typename EdgeSubMap::iterator j = i->second.find(head);
			if (j != i->second.end()) {
                PEdge edge = j->second;
                edge->tail()->remove(edge);
                edge->head()->remove(edge);
				i->second.erase(j);
			}
		}
	}

	/** <<iterate>>*/
	template<typename FUNC>
	void foreach(FUNC& f)
	{
		for (vert_iterator i=m_vertices.begin(); i!=m_vertices.end(); ++i) {
			f(i->first);
		}
	}
	
	/** <<iterate>> f:(T, T)*/
	template<typename FUNC>
	void foreachEdge(FUNC& f)
	{
		for (typename EdgeMap::iterator i=m_edges.begin();
			 i!=m_edges.end(); ++i) {
			for (typename EdgeSubMap::iterator j=i->second.begin();
				 j!=i->second.end(); ++j) {
				f(i->first,j->first);
			}
		}
	}
	/** <<iterate>> f:(T, T, Label)*/
	template<typename FUNC>
	void foreachLabeledEdge(FUNC& f)
	{
		for (typename EdgeMap::iterator i=m_edges.begin();
			 i!=m_edges.end(); ++i) {
			for (typename EdgeSubMap::iterator j=i->second.begin();
				 j!=i->second.end(); ++j) {
				f(i->first,j->first, j->second->label());
			}
		}
	}
	iterator begin()const
	{
		Path path;
		for (cvert_iterator i=m_vertices.begin(); i!=m_vertices.end(); ++i) {
            path.push_back(pair(i->first, 0));// all node is top level.
		}
        
		assert(path.size() == size());
		return iterator(path); 
	}
	iterator end()const { return iterator(Path()); }
	iterator find(T v)
	{
		Path path;
		for (typename VertexMap::const_iterator i=m_vertices.find(v);
			 i!=m_vertices.end(); ++i) {
			pair node(i->first, 0);
			path.push_back(node);
		}
		iterator i(path);
		return i;
	}
	void insert(Graph::iterator i)
	{
		if (i.isValid()) { addVertex(i->first); }
	}
	void remove(Graph::iterator i)
	{
		if (i.isValid()) { removeVertex(i->first); }
	}
	void swap(Graph& that)
	{
		m_vertices.swap(that.m_vertices);
		m_edges.swap(that.m_edges);
	}
	void copy(const Graph& that)
	{
		clear();
		for (typename EdgeList::const_iterator i=that.m_edges.begin();
			 i != that.m_edges.begin(); ++i) {
			PEdge e=*i;
			addEdge(e->head()->value(), e->tail()->value());
		}
	}

	/** <<traverse>> */
	iterator ascendant()const { return begin();}
	
	/** <<traverse>> depth first pre-order traversal; child is first*/
	iterator pre(T v)const
	{
		Path path;
		std::set<T> visited;
		collectPathPreorder(path, visited, v, 0);
		assert(path.size() <= size());
		return iterator(path);
	}
	/** <<traverse>> depth first post-order traversal;
		like control
	*/
	iterator post(T v)const
	{
		Path path;
		std::set<T> visited;
		collectPathPostorder(path, visited, v, 0);
		assert(path.size() <= size());
        for (typename Path::const_iterator i=path.begin();
             i != path.end(); ++i) {
            assert(hasVertex(i->first));
        }
		return iterator(path);
	}
	/** <<traverse>> depth first reversed post-order traversal;last is first*/
	iterator rpost(T v)const
	{
		Path path;
		std::set<T> visited;
		collectPathPostorder(path, visited, v, 0);
		assert(path.size() <= size());

		unsigned size=path.size();
		Path rpath(size);
		for (unsigned i=0; i<size; ++i) {
			rpath[i]=path[size-1-i];
		}
		return iterator(rpath);
	}
	
	
	/** <<traverse>> breadth first traversal*/
	iterator breadthFirst(T v)const
	{
		Path path;
		std::set<T> visited;
		path.push_back(pair(v,0));
		visited.insert(v);
		
		collectPathBreadthFirst(path, visited, v, 1);
		assert(path.size() <= size());
		return iterator(path);
	}
	
private:
	/** O(1): This method is get vertex pointer by element.*/
	PVertex getVertex(T v)const
	{
		typename VertexMap::const_iterator i = m_vertices.find(v);
		if (i != m_vertices.end()){
			return i->second;
		} else {
			return Vertex::getInvalid();
		}
	}
	enum TraversalType { PREORDER, POSTORDER, };
	void collectPathDepthFirst(Path& path, typename std::set<T>& visited,
							   T v, unsigned depth, TraversalType t)const
	{
		if (visited.find(v) != visited.end()
            || !hasVertex(v) ){
			return;
		}
		visited.insert(v);
		
		if (t == POSTORDER) {
			path.push_back(pair(v,depth));
		}
		std::vector<T> succ=successors(v);
		for (unsigned i=0; i<succ.size(); ++i) {
			collectPathDepthFirst(path, visited, succ[i], depth+1, t);
		}
		if (t == PREORDER) {
			path.push_back(pair(v,depth));
		}
	}
	void collectPathPreorder(Path& path, typename std::set<T>& visited,
							 T v, unsigned depth)const
	{
		collectPathDepthFirst(path, visited, v, depth, PREORDER);
	}
	void collectPathPostorder(Path& path, typename std::set<T>& visited,
							  T v, unsigned depth)const
	{
		collectPathDepthFirst(path, visited, v, depth, POSTORDER);
	}
	void collectPathBreadthFirst(Path& path, std::set<T>& visited,
								 T v, unsigned depth)const
	{
		std::vector<T> succ=successors(v);
		std::vector<T> next;
		for (unsigned i=0; i<succ.size(); ++i) {
			if (visited.find(succ[i]) == visited.end()){
				path.push_back(pair(succ[i],depth));
				visited.insert(succ[i]);
				next.push_back(succ[i]);
			}
		}
		for (unsigned i=0; i<next.size(); ++i) {
			collectPathBreadthFirst(path, visited, next[i], depth+1);
		}
	}
	/** called from clear*/
	void removeEdgesFromOpposite(Vertex* v, const EdgeList& list)
	{
		typename EdgeList::const_iterator i;
		for (i=list.begin(); i!=list.end();++i){
			PEdge e=*i;
			Vertex* a=e->head();
			Vertex* b=e->tail();
				
			if (a != v) { a->remove(e); }
			else if (b != v) { b->remove(e); }
		}
	}
};

