
#include "display_module.hpp"

#include "nes_memory_mapper.hpp"
#include "disassembler.hpp"
#include "code_data_parser.hpp"

void displayINesFile(std::ostream& os, 
                     const INesFile& nes,
					 const std::vector<Word>& interrupts)
{
	Disassembler dis(os);
	dis.writeINesFile(nes);
	dis.writeInterruptVector(interrupts);
}
void displayOrigins(std::ostream& os, const std::vector<Word>& origins)
{
	for (unsigned i=0; i<origins.size(); ++i) {
		os << "; bank " << HexForm(i).width(2) 
           << ":" << HexForm(origins[i])
           << std::endl;
	}
}

void displayRawAssembly(const std::vector<PByteSequence>& banks,
						const std::vector<Word>& origins)
{
	Disassembler dis(std::cout);
		
	for (unsigned i=0; i<banks.size(); ++i) {
		dis.writeBankOrigin(i, origins[i]);
		dis.writeBankLiterally(origins[i], *(banks[i]));
	}
}

void displayRawAssemblyRegeon(
        std::ostream& os,
        const std::vector<PByteSequence>& banks,
        const std::vector<Word>& origins,
        Byte bank_num,
        Word begin,
        Word end)
{
    unsigned begin_index=begin&0x1FFF;
    unsigned end_index=end&0x1FFF;
        
	Disassembler dis(os);
		
    dis.writeBankRegeon(origins[bank_num],
                        *(banks[bank_num]),
                        begin_index,end_index);

}

void displayBankAddress(std::ostream& os,
                        Byte bank_num, Word addr)
{
    os << "(" << HexForm(bank_num).width(2)
       << ")"
       << HexForm(addr);
}

void displayCodeDataFlag(std::ostream& os,Byte byte)
{
    Byte f=byte & 0x33;
    if (f == cdp::BLANK) {
        os << " ";
    } else if (f == (cdp::CODE | cdp::IND_CODE)) {
        os << "C";
    } else if (f == (cdp::CODE | cdp::DATA)) {
        os << "I";
    } else if (f == cdp::CODE) {
        os << "c";
    } else if (f == (cdp::DATA | cdp::IND_DATA)) {
        os << ":";
    } else if (f == cdp::DATA) {
        os << ".";
    } else {
        Byte high = (f >> 2) & 0x0C;
        Byte low = f & 0x03;
        os << HexForm(low+high).width(1);
    }
}
unsigned addressToCDLIndex(Byte bank_num, Word addr)
{
    return (addr & 0x1FFF) + bank_num*nes::BANK_SIZE;
}

void displayCodeDataRegeon(std::ostream& os,
                           const ByteSequence& bytes,
                           const std::vector<Word>& origins,
                           Byte bank_num,
                           Word begin,
                           Word end)
{
    unsigned col_length=32;
    Word org = origins[bank_num];
    
    unsigned begin_index = addressToCDLIndex(
            bank_num,begin);
    unsigned end_index = addressToCDLIndex(
            bank_num,end);
    
    unsigned begin_index2 =
        begin_index - begin_index % col_length;
    if (begin_index2 < bank_num * nes::BANK_SIZE){
        begin_index2= bank_num * nes::BANK_SIZE;
    }
    unsigned end_index2 =
        (end_index) - (end_index)%col_length + col_length;
    if (end_index2 > unsigned(
                (bank_num+1) * nes::BANK_SIZE)){
        end_index2= (bank_num+1) * nes::BANK_SIZE;
    }
    
    for (unsigned i=begin_index2; i<end_index2; ++i){
        Word addr = (i & 0x1FFF)+org;
        
        if (i%col_length==0) {
            displayBankAddress(os,bank_num,addr);
            os << ": ";
        }
        if (i==begin_index){
            os << "[";
        }else if (i==end_index) {
            os << "]";
        } else {
            displayCodeDataFlag(os,bytes[i]);
        }
        if (i%col_length==col_length-1) {
            os << std::endl;
        }
    }
}
void displayCodeDataInfo(std::ostream& os,
                         const ByteSequence& bytes,
                         const std::vector<Word>& origins)
{
    unsigned col_length=32;
    
    for (unsigned i=0; i<bytes.size(); ++i){
        
        Byte bank_num = i / nes::BANK_SIZE;
        assert(bank_num < origins.size());
        Word org = origins[bank_num];
        Word addr = (i & 0x1FFF)+org;
        
        if (i%col_length==0) {
            displayBankAddress(os,bank_num,addr);
            os << ": ";
        }
        displayCodeDataFlag(os,bytes[i]);
        
        if (i%col_length==col_length-1) {
            os << std::endl;
        }
    }
}
void displayCodeDataStats(std::ostream& os,
                          const ByteSequence& bytes)
{
    unsigned blank=0;
    unsigned code=0;
    unsigned data=0;
    unsigned size=bytes.size();
    
    for (unsigned i=0; i<bytes.size(); ++i){
        Byte f=bytes[i] & 0x33;
        if (f == cdp::BLANK) {
            blank++;
        }
        if (f & (cdp::CODE|cdp::IND_CODE)) {
            code++;
        }
        if (f & (cdp::DATA|cdp::IND_DATA)) {
            data++;
        }
    }
    double code_rate = code*100.0 / size;
    double data_rate = data*100.0 / size;
    double blank_rate = blank*100.0 / size;
    
    os << "code rate: " << code_rate << std::endl;
    os << "data rate: " << data_rate << std::endl;
    os << "blank rate: " << blank_rate << std::endl;
}
