#! /usr/bin/env ruby -Ku
#
# 2008-04-11 katoy
#
# EDINET の XBRL ファイル群のファイル名、ファイル内容を変更して、複写する。
#  jpfr-ssr-X99006-000-2008-09-30-00-2008-12-26*.* のファイルを
#  jpfr-ssr-X99006-000-2008-09-30-01-2008-12-26*.* と 提出回数の部分だけを変更する
#  ときなどに利用できる。
#  (ファイル名だけでなく、
#   ファイル内の     "jpfr-ssr-X99006-000-2008-09-30-00-2008-12-26"
#   の文字列もすべて "jpfr-ssr-X99006-000-2008-09-30-01-2008-12-26"
#   に置換される)
#  ただし、ファイル内容置換は単純な文字置換であることに注意すること。
#

def put_usage
  $stderr.puts "Usage: ruby #{File.basename($0)} srcString dstString"
  $stderr.puts " example: ruby jpfr-ssr-X99006-000-2008-09-30-02-2008-12-26 zzz-ssr-X99999-000-2008-09-30-00-2008-12-26"
  $stderr.puts "     Copy ./jpfr-ssr-X99006-000-2008-09-30-02-2008-12-26*.* to"
  $stderr.puts "          ./zzz-jpfr-ssr-X99999-000-2008-09-30-00-2008-12-26*.*, "
  $stderr.puts "     And replace file contents."
end

class XbrlCopy

  # EDINET 規約の ファイル名中の "jpfr-asr-X99001-000-2008-03-31-01-2008-06-27" の書式から
  # xmlns 宣言、prefix を生成する。
  def parse(edinetStr)
    info = {}

    # 0123456789 0123456789 0123456789 0123456789 012
    # jpfr-asr-X99001-000-2008-03-31-01-2008-06-27"
    prefix = edinetStr[0..18]
    kind  = edinetStr[5..7]
    code  = edinetStr[9..18]
    start = edinetStr[20..29]
    count = edinetStr[31..32]
    date  = edinetStr[34..-1]

    info[:xmlns] =
      %Q<xmlns:#{prefix}="http://info.edinet-fsa.go.jp/jp/fr/gaap/#{code}/#{kind}/#{start}/#{count}/#{date}">
    info[:prefix_colon] = prefix + ':'

    info
  end

  def copy_with_replace(srcFile, destFile, srcStr, destStr)
    src_info = parse(srcStr)
    dest_info = parse(destStr)
    open(destFile, 'w') {|dest|
      open(srcFile) {|src|
        while line = src.gets
          line.gsub!(srcStr, destStr)
          line.gsub!(src_info[:prefix_colon], dest_info[:prefix_colon])
          line.gsub!(src_info[:xmlns], dest_info[:xmlns])
          dest.puts line
        end
      }
    }
  end

  def copy_all(from_str, to_str)
    r = Regexp.new(from_str)
    Dir.foreach('.') do |srcFile|
      next if (r =~ srcFile) == nil  # skip non-EDINET Files
      destFile = srcFile.gsub(from_str, to_str)
      copy_with_replace(srcFile, destFile, from_str, to_str)
    end
  end

end

if ARGV.size != 2
  put_usage
  exit 1
end

from_edinet_str = ARGV[0]
to_edinet_str = ARGV[1]
app = XbrlCopy.new
app.copy_all(from_edinet_str, to_edinet_str)
