/*
 * Copyright (c) 2008, AIST.
 * All rights reserved. This program is made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * National Institute of Advanced Industrial Science and Technology (AIST)
 */
/* $Id: shmcport.c,v 1.3 2008/06/06 09:00:02 yoshi Exp $ */
#ifdef USE_SHMC
#include <RtORB/corba.h>
#include <RtORB/shmc.h>
#include <RtORB/sockport.h>

#define SHMC_CLOSED SOCK_CLOSED
#define SHMC_SERVER SOCK_SERVER
#define SHMC_SERVICE SOCK_SERVICE
#define SHMC_CLIENT SOCK_CLIENT

#define SHMC_LOCK()
#define SHMC_UNLOCK()

typedef struct {
  int idx;
  int type;
  void *arg;
  Shmc * shmc;
  int (*connection_proc)();
  int (*command_proc)();
  int (*disconnect_proc)();
} shmc_profile;

static shmc_profile ShmcProfile[SHMC_MAX_CONN] = { {0} };
static ShmcSet shmcSets[SHMC_MAX_CONN] = { {0} };
static int nShmcSets = 0;

#define ARY_SIZE(ARY) (int)(sizeof(ARY)/sizeof(ARY[0]))

void init_ShmcProfile()
{
  int i;
  memset(&ShmcProfile, 0x00, sizeof(ShmcProfile));
  for (i=0; i<ARY_SIZE(ShmcProfile); i++) {
    ShmcProfile[i].idx = -1;
  }
}

static shmc_profile *get_empty_profile()
{
  int i;
  shmc_profile *p = NULL;
  for (i=0; i<ARY_SIZE(ShmcProfile); i++) {
    p = &ShmcProfile[i];
    if (p->shmc == NULL) {
      p->idx = i;
      return p;
    }
  }
  return NULL;
}

static shmc_profile * find_profile(Shmc *h);

void set_ShmcProfile(Shmc *h, int type, 
		     int (*connection_proc)(),
		     int (*disconnect_proc)(),
		     int (*command_proc)()
		     )
{
  shmc_profile *prof = find_profile(h);
  if (!prof) {
    prof = get_empty_profile();
    prof->shmc = h;
  }
  
  prof->type = type;
  prof->connection_proc = connection_proc;
  prof->command_proc = command_proc;
  prof->disconnect_proc = disconnect_proc;
}

void set_ShmcProfile_arg(Shmc *h, void *arg)
{
  shmc_profile *prof = find_profile(h);
  if (!prof) {
    prof = get_empty_profile();
    prof->shmc = h;
  }
  prof->arg = arg;
}

void *get_ShmcProfile_arg(Shmc *h)
{
  shmc_profile *prof = find_profile(h);
  if (!prof) { return NULL; }
  return prof->arg;
}
	
static shmc_profile * find_profile(Shmc *h)
{
  int i;
  if (!h) { return NULL; }
  
  for (i=0; i<ARY_SIZE(ShmcProfile); i++) {
    shmc_profile *prof = &ShmcProfile[i];
    if (prof->shmc == h) {
      prof->idx = i;
      return prof;
    }
  }
  return NULL;
}

static shmc_profile *copy_profile(Shmc *shmc, shmc_profile *orig)
{
  shmc_profile *target = get_empty_profile();
  int idx = target->idx;
  memcpy(target, orig, sizeof(*orig));
  target->shmc = shmc;
  target->idx = idx;
  return target;
}

static Shmc *next_active_shmc(int nSets, ShmcSet *sets, int *currIdx)
{
  int i = *currIdx >= 0? *currIdx + 1 : 0;
  for (; i<nSets; i++) {
    if (SHMC_ISSET(sets, i)) {
      *currIdx = i;
      return SHMC_GET(sets, i);
    }
  }
  return NULL;
}

static int get_Servers_from_ShmcProfile(Shmc **servers, int *n)
{
  int i;
  int cnt = 0;

  SHMC_LOCK();

  for (i=0; i<ARY_SIZE(ShmcProfile); i++) {
    if (ShmcProfile[i].shmc == NULL) { continue; }
    if (ShmcProfile[i].type == SHMC_SERVER ||
	ShmcProfile[i].type == SHMC_SERVICE) {
      servers[cnt] = ShmcProfile[i].shmc;
      cnt++;
    }
  }
  *n = cnt;

  SHMC_UNLOCK();

  return cnt;
}

void init_shmc_servers()
{
  int i;
  Shmc *servers[SHMC_MAX_CONN];
  int n;

  get_Servers_from_ShmcProfile(servers, &n);

  SHMC_LOCK();

  for (i=0; i<ARY_SIZE(shmcSets); i++) {
    SHMC_CLR(shmcSets, i);
  }

  for(i=0; i<n; i++) {
    SHMC_SET(shmcSets, i, servers[i]);
  }
  nShmcSets = n;

  SHMC_UNLOCK();
#ifdef DEBUG1
  fprintf(stderr, "init_shmc_server : server num = %d\n", n);
#endif
}

static int select_shmc_port(int nSets, ShmcSet *sets, struct timeval timeout);

void select_shmc_servers(struct timeval timeout)
{
  init_shmc_servers();

  select_shmc_port(nShmcSets, shmcSets, timeout);
}

static int select_shmc_port(int nSets, ShmcSet *sets, struct timeval timeout)
{
  int res, stat;
  int activeIdx = -1;
  int (*func)();
  GIOP_ConnectionHandler h;

  res = stat = shmcSelect(nSets, sets, 0, NULL, &timeout);
#ifdef DEBUG1
  fprintf(stderr, "shmc %d/%d port active\n", res, nSets);
#endif
  while (stat > 0) {
    Shmc *active = next_active_shmc(nSets, sets, &activeIdx); /* OK */

    if (active == NULL) {
      fprintf(stderr, "Invalid shmc port is active");
      return -1;
    }

    shmc_profile *prof = find_profile(active); /* OK */
    switch(prof->type) {
    case SHMC_SERVER:
      {
	Shmc *new_;
#ifdef DEBUG1
	fprintf(stderr, "Request shmc connection %#x\n", active);

#endif
	new_= shmcAccept(active);
	if (new_ != NULL) {
	  shmc_profile *new_prof = copy_profile(new_, prof); /* OK */
	  SHMC_SET(sets, new_prof->idx, new_);

	  new_prof->type = SHMC_SERVICE;
#ifdef DEBUG1
	  fprintf(stderr, "Request shmc connection %#x\n", active);
#endif
	}
	func = prof->connection_proc;
	if (func) {
	  (*func)(active, prof->arg);
	}
	//SHMC_CLR(sets, prof->idx);
	stat--;
      }
      break;
    case SHMC_CLIENT: 
    case SHMC_SERVICE:
#ifdef DEBUG1
      fprintf(stderr, "shmcport responding...\n");
#endif
      h.type = CONNECTION_TYPE_SHMC;
      h.shmc = active;

      func = prof->command_proc;
      if (func && (*func)(&h) == -1) {
	func = prof->disconnect_proc;
	if (func) {
	  (*func)(active, prof->arg);
	}
#ifdef DEBUG1
	fprintf(stderr, "shmc : Request Close Connection fd=%d\n", active_port);
#endif
	shmcClose(active);
        SHMC_CLR(sets, prof->idx);
	prof->type = SHMC_CLOSED;
	prof->shmc = NULL;
      }
      //      SHMC_CLR(sets, prof->idx);
      stat--;
      break;
    default:
      fprintf(stderr, "shmc already closed...\n");
      stat--;
      break;
    }
  }
    
  return res;
}


#endif

