/*
 * Copyright (c) 2008, AIST.
 * All rights reserved. This program is made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * National Institute of Advanced Industrial Science and Technology (AIST)
 */
/*
 * Shmc: An attempting to replace the UNIX domain socket,
 * with semaphore and sharedmemory.
 *
 * $Id: shmc.c,v 1.17 2008/06/06 10:20:17 yoshi Exp $
 */
#include <unistd.h>
#include <stdlib.h>
#include <malloc.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/shm.h>

#define SHMC_LIB
#include <shmc.h>

#define MAX_CONN     	SHMC_MAX_CONN
#define SEND_SIZE	SHMC_SEND_SIZE
#define RECV_SIZE	SHMC_RECV_SIZE

//
#define SHMB_SIZE   (sizeof(int)*8)   // bind function
#define SEM_SIZE   ((MAX_CONN+1)*2+4)
#define SHMS_SIZE   (sizeof(void*)*4)                     // select function
#define SHMD_SIZE   (sizeof(int)*8+SEND_SIZE+RECV_SIZE)   // send/recv
#define SHMC_SIZE   (SHMD_SIZE+(SHMS_SIZE)*2)

#define SHMC_INIT     0
#define SHMC_OPEN     1
#define SHMC_BIND     2
#define SHMC_ACCEPT   4
#define SHMC_CONNECT  8
#define SHMC_CLOSE   16

#define SEL_R    0x01
#define SEL_W    0x02

#define CONNECT_SEMSET_ID  MAX_CONN*2
#define ACCEPT_SEMSET_ID  (MAX_CONN*2+1)
#define AC_SEMSET_ID      (MAX_CONN*2+2)
#define BASE_SEMSET_ID    (MAX_CONN*2+3)

/*
 * Structure for managing each Connection
 */
typedef struct ShmcConn{
  int   conn_id;      // id-number of each connection(for internal use)
  // int  semset_id;
  int  s_select_semset;  // for setting
  int  r_select_semset;  // for reference
  int  send_sem_id;  // semaphore-id for sending(index to the semaphore-set)
  int  recv_sem_id;  // semaphore-id for receiving(index to the semaphore-set)
  int  sendb_sem_id; // semaphore-id for blocking send
  int  recvb_sem_id; // semaphore-id for blocking recv
  int  shm_id;      // sharedmemory-id
  char *shmp;       // the head pointer of the mapped sharedmemory area
  int  *s_sem_id;
  int  *s_semset_id;
  int  *s_flags;
  int  *r_sem_id;
  int  *r_semset_id;
  int  *r_flags;
  int  *status;     // the pointer of status of this connection.
  char *send;       // pointer of sending area
  int  *send_head;
  int  *send_length;
  char *recv;       // pointer of receiving area
  int  *recv_head;
  int  *recv_length;
} ShmcConn;

//
typedef union {
  int val;
  unsigned short *buf;
} semun;

//extern int semtimedop(int sem_id, struct sembuf *semops, int id, struct timespec *timeout);


//
static int wait_sem_n_wtm(int sem_id, int semset_id, int n, struct timespec *timeout)
{
  struct sembuf semops[1];
  semops[0].sem_num = semset_id;
  semops[0].sem_op = n*(-1);
  semops[0].sem_flg = 0;

  while(1) {
    // int r = semop(sem_id, semops, (unsigned)1);
    // int r = semtimedop(sem_id, semops, (unsigned)1, timeout);
    int r;
    if (timeout != NULL) 
      r = semtimedop(sem_id, semops, (unsigned)1, timeout);
    else
      r = semop(sem_id, semops, (unsigned)1);
    if (r<0) {
      if (errno==EINTR)
	continue;         // must update the timeout 
#ifdef SHMC_DEBUG_TRACE
      perror("wait_sem_n_wtm");
#endif
      return -1;
    } else
      return 0;
  }
}

//
static int wait_sem_n(int sem_id, int semset_id, int n)
{
  return wait_sem_n_wtm(sem_id, semset_id, n, NULL);
}

//
static int reset_sem_n(int sem_id, int semset_id, int n)
{
  semun semval;
  semval.val = n;
  int r = semctl(sem_id, semset_id, SETVAL, semval);
  if (r<0) {
#ifdef SHMC_DEBUG_TRACE
      perror("reset_sem_n");
#endif
    return -1;
  }
  return 0;
}

//
static int get_shm_nattach(int shmid) {
  struct shmid_ds ds;

  int r = shmctl(shmid, IPC_STAT, &ds);
  if (r<0) {
#ifdef SHMC_DEBUG_TRACE
      perror("get_shm_nattach");
#endif
    return -1;
  }
  return ds.shm_nattch;
}

//
static void closeBase(Shmc *shmc);

/*
 *
 */
Shmc *shmcOpen(const char *key)
{
  Shmc *base = (Shmc *)malloc(sizeof(Shmc));
  if (base==NULL) {
    // ENOMEM
#ifdef SHMC_DEBUG_TRACE
    perror("shmcOpen");
#endif
    return NULL;
  }
  base->app_key = strdup(key);
  base->shmc_key = ftok(key, MAX_CONN*2+1);
  base->type = 0;
  base->close = closeBase;
  base->sem_id = 0;
  base->sem_id_b = 0;
  base->sem_id_s = 0;
  base->sem_id_w = 0;
  // base->shm_id = 0;
  // base->shmp = NULL;
  base->connect = NULL;
  base->accept = NULL;
  return base;
}

//
static void closeBase(Shmc *shmc)
{
#ifdef SHMC_DEBUG_TRACE
  fprintf(stderr, "closeBase: start\n");
#endif
  //
  reset_sem_n(shmc->sem_id, ACCEPT_SEMSET_ID, 1);
  reset_sem_n(shmc->sem_id, CONNECT_SEMSET_ID, 1);
  reset_sem_n(shmc->sem_id, AC_SEMSET_ID, 1);
  reset_sem_n(shmc->sem_id, BASE_SEMSET_ID, 1);

  if (shmc->accept) {
    // timedwait_sem_n(shmc->sem_id, ACCEPT_SEMSET_ID, 1);
    Shmc **accept = (Shmc **)shmc->accept;
    int i;
    for (i = 0; i < MAX_CONN; i++) {
      if (accept[i] != NULL) {
	// if (accept[i]->shmp!=NULL)
	//   shmdt(accept[i]->shmp);
	// if (accept[i]->shm_id>=0)
	//   shmctl(accept[i]->shm_id, IPC_RMID, NULL);
	// if (accept[i]->sem_id>=0)
	//  semctl(accept[i]->sem_id, 0, IPC_RMID);
	free (accept[i]);
	accept[i] = NULL;
      }
    }
    // reset_sem_n(shmc->sem_id, ACCEPT_SEMSET_ID, 1);
    free(shmc->accept);
  }
  if (shmc->sem_id > 0)
    semctl(shmc->sem_id, 0, IPC_RMID);
  if (shmc->sem_id_b > 0)
    semctl(shmc->sem_id_b, 0, IPC_RMID);
  if (shmc->sem_id_s > 0)
    semctl(shmc->sem_id_s, 0, IPC_RMID);
  if (shmc->sem_id_w > 0)
    semctl(shmc->sem_id_w, 0, IPC_RMID);
  //
  if (shmc->connect != NULL) {
    if (shmc->connect->shm_id > 0) {
      shmctl(shmc->connect->shm_id, IPC_RMID, NULL);
    }
    free(shmc->connect);
  }
  
  if (shmc->app_key) {
    free(shmc->app_key);
  }
  free(shmc);
#ifdef SHMC_DEBUG_TRACE
  fprintf(stderr, "closeBase: end\n");
#endif
}

//
static void wakeup(Shmc *shmc, ShmcConn *conn) {
    if (reset_sem_n(shmc->sem_id, conn->send_sem_id, 1) < 0) {
#ifdef SHMC_DEBUG_TRACE
      perror("wakeup");
#endif
    }
    if (reset_sem_n(shmc->sem_id_b, conn->sendb_sem_id, 5) < 0) {
#ifdef SHMC_DEBUG_TRACE
      perror("wakeup");
#endif
    }
    if (reset_sem_n(shmc->sem_id, conn->recv_sem_id, 1) < 0) {
#ifdef SHMC_DEBUG_TRACE
      perror("wakeup");
#endif
    }
    if (reset_sem_n(shmc->sem_id_b, conn->recvb_sem_id, 5) < 0) {
#ifdef SHMC_DEBUG_TRACE
      perror("wakeup");
#endif
    }
}

//
static int is_selected(Shmc *shmc, int flag);
//
static void closeAcceptSide(Shmc *shmc)
{
  ShmcConn *conn = shmc->connect;

  if (conn!=NULL) {
    //
    *(conn->status) = SHMC_CLOSE;
    is_selected(shmc, SEL_W);
    is_selected(shmc, SEL_R);
    //
#ifdef SHMC_DEBUG_TRACE
    fprintf(stderr, "closeAcceptSide: conn_id is %d\n",conn->conn_id);
#endif
    if (conn->shmp!=NULL)
      shmdt(conn->shmp);
    if (conn->shm_id>=0) {
#ifdef SHMC_DEBUG_TRACE
      fprintf(stderr, "closeAcceptSide: shm_id is %d\n",conn->shm_id);
#endif
      shmctl(conn->shm_id, IPC_RMID, NULL);
    }
  }
  // free(((ShmcConn **)shmc->accept)[conn->conn_id]);
  // timedwait_sem_n(shmc->sem_id, ACCEPT_SEMSET_ID, 1);
  wait_sem_n(shmc->sem_id, ACCEPT_SEMSET_ID, 1);

  ((ShmcConn **)shmc->accept)[conn->conn_id] = NULL;
  reset_sem_n(shmc->sem_id, ACCEPT_SEMSET_ID, 1);
  //
  wakeup(shmc, conn);

  free(conn);
  //
  free(shmc);
#ifdef SHMC_DEBUG_TRACE
  fprintf(stderr, "closeAcceptSide: end\n");
#endif
}

//
static void closeConnectSide(Shmc *shmc)
{
  ShmcConn *conn = shmc->connect;
  if (conn!=NULL) {
#ifdef SHMC_DEBUG_TRACE
    fprintf(stderr, "closeConnectSide: conn_id is %d\n",conn->conn_id);
#endif
    //
    if (wait_sem_n(shmc->sem_id, conn->send_sem_id, 1) < 0) {
      // error
#ifdef SHMC_DEBUG_TRACE
      perror("closeConnectSide");
#endif
    }
    *(conn->status) = SHMC_CLOSE;
    if (reset_sem_n(shmc->sem_id, conn->send_sem_id, 1) < 0) {
      return;
#ifdef SHMC_DEBUG_TRACE
      perror("closeConnectSide");
#endif
    }
    is_selected(shmc, SEL_W);
    is_selected(shmc, SEL_R);

    //
    if (conn->shmp!=NULL)
      shmdt(conn->shmp);
    if (conn->shm_id>=0) {
      if (get_shm_nattach(conn->shm_id)<1) {
#ifdef SHMC_DEBUG_TRACE
	fprintf(stderr, "closeConnectSide: shm_id is %d\n",conn->shm_id);
#endif
	shmctl(conn->shm_id, IPC_RMID, NULL);
      }
    }
    wakeup(shmc, conn);

    free(conn);
  }
  //
  free(shmc);
#ifdef SHMC_DEBUG_TRACE
  fprintf(stderr, "closeConnectSide: end\n");
#endif
}

/*
 *
 */
void shmcClose(Shmc *shmc)
{
  if (shmc->close != NULL)
    shmc->close(shmc);
}

/*
 *
 */
int shmcBind(Shmc *shmc)
{
  if (shmc==NULL)
    return -1;  // error

  // generate a key
  key_t shmc_key = shmc->shmc_key;
  // get a new semaphore set
#ifdef NO_IPC_EXCL
  shmc->sem_id = semget(shmc_key, SEM_SIZE, IPC_CREAT|0666);
  shmc->sem_id_b = semget(ftok(shmc->app_key, MAX_CONN*2+2), SEM_SIZE, IPC_CREAT|0666);
  shmc->sem_id_s = semget(ftok(shmc->app_key, MAX_CONN*2+3), SEM_SIZE, IPC_CREAT|0666);
  shmc->sem_id_w = semget(ftok(shmc->app_key, MAX_CONN*2+4), SEM_SIZE, IPC_CREAT|0666);
#else
  shmc->sem_id = semget(shmc_key, SEM_SIZE, IPC_CREAT|IPC_EXCL|0666);
  shmc->sem_id_b = semget(ftok(shmc->app_key, MAX_CONN*2+2), SEM_SIZE, IPC_CREAT|IPC_EXCL|0666);
  shmc->sem_id_s = semget(ftok(shmc->app_key, MAX_CONN*2+3), SEM_SIZE, IPC_CREAT|IPC_EXCL|0666);
  shmc->sem_id_w = semget(ftok(shmc->app_key, MAX_CONN*2+4), SEM_SIZE, IPC_CREAT|IPC_EXCL|0666);
#endif
  if(shmc->sem_id<0||shmc->sem_id_b<0||shmc->sem_id_s<0||shmc->sem_id_w<0) {
    // error  EBUSY
#ifdef SHMC_DEBUG_TRACE
    perror("shmcBind: semget");
#endif
    return -1;
  }

  // set semval of all semaphore set to 0
  semun semval;
  unsigned short semvar_ar[SEM_SIZE];
  semval.buf = semvar_ar;
  int i;
  for (i=0; i<SEM_SIZE; i++)
    semval.buf[i] = 0;
  semctl(shmc->sem_id, 0, SETALL, semval);

  ShmcConn *conn = (ShmcConn*)malloc(sizeof(ShmcConn));
  shmc->connect = conn;
  if (shmc->connect==NULL) {
    // error  ENOMEM
#ifdef SHMC_DEBUG_TRACE
    perror("shmcBind: shmc->connect");
#endif
    return -1;
  }
  //
  //
  conn->s_select_semset = MAX_CONN; 
  conn->r_select_semset = MAX_CONN+1;

  // get sheredmemory for nagotiation of Accept and Connect
#ifdef NO_IPC_EXCL
  int shm_id = shmget(shmc_key, SHMB_SIZE, IPC_CREAT|0666);
#else
  int shm_id = shmget(shmc_key, SHMB_SIZE, IPC_CREAT|IPC_EXCL|0666);
#endif

  if(shm_id < 0) {
    // error  EBUSY
#ifdef SHMC_DEBUG_TRACE
    perror("shmcBind 2");
#endif
    return -1;
  }
  //
  {
    int r;
    r = get_shm_nattach(shm_id);
#ifdef SHMC_DEBUG_TRACE
    fprintf(stderr, "shmcBind: get_shm_nattach is %d\n", r);
#endif
    if (get_shm_nattach(shm_id)>=1) {
      shmc->sem_id = 0;
      shmc->sem_id_b = 0;
      shmc->sem_id_s = 0;
      shmc->sem_id_w = 0;
      errno = EINVAL;
      return -1;
    }
  }


  conn->shm_id = shm_id;
  //
  conn->shmp = (char *)shmat(shm_id, NULL, 0);
  int *shmp = (int *)(conn->shmp);
  for (i=0; i<SHMB_SIZE; i++) {
    shmp[i] = 0;
  }
  conn->s_sem_id = shmp+2;
  conn->s_semset_id = shmp+3;
  conn->s_flags = shmp+4;
  conn->r_sem_id = shmp+5;
  conn->r_semset_id = shmp+6;
  conn->r_flags = shmp+7;

  // for managing each connection
  shmc->accept = (void **)malloc(MAX_CONN*sizeof(ShmcConn *));
  if (shmc->accept==NULL) {
    // error  ENOMEM
#ifdef SHMC_DEBUG_TRACE
    perror("shmcBind: shmc->accept");
#endif
    return -1;
  }
  for (i=0; i<MAX_CONN; i++)
    shmc->accept[i] = NULL;
  //
  reset_sem_n(shmc->sem_id_s, conn->s_select_semset, 1);
  reset_sem_n(shmc->sem_id_s, conn->r_select_semset, 1);
  reset_sem_n(shmc->sem_id_w, conn->s_select_semset, 1);
  reset_sem_n(shmc->sem_id_w, conn->r_select_semset, 1);

  //
  reset_sem_n(shmc->sem_id, CONNECT_SEMSET_ID, 1);
  // reset_sem_n(shmc->sem_id, AC_SEMSET_ID, 1);
  reset_sem_n(shmc->sem_id, AC_SEMSET_ID, 0);
  reset_sem_n(shmc->sem_id, ACCEPT_SEMSET_ID, 1);
  reset_sem_n(shmc->sem_id, BASE_SEMSET_ID, 1);      // for select function
  shmc->type = SHMC_BIND;
  return 0;
}

//
static Shmc *newShmc(Shmc *shmc) {
  Shmc *nshmc = (Shmc*)malloc(sizeof(Shmc));
  if (shmc!=NULL) {
    nshmc->app_key = shmc->app_key;
    // nshmc->shmc_key = shmc->shmc_key;
    // nshmc->type ...
    nshmc->sem_id = shmc->sem_id;
    nshmc->sem_id_b = shmc->sem_id_b;
    nshmc->sem_id_s = shmc->sem_id_s;
    nshmc->sem_id_w = shmc->sem_id_w;
    // nshmc->shm_id = shmc->shm_id;
    // nshmc->shmp = shmc->shmp;
    nshmc->accept = shmc->accept;
  }
  return nshmc;
}

/*
 *
 */
Shmc *shmcAccept(Shmc *shmc) 
{
  int r;
  int conn_id;

  r = reset_sem_n(shmc->sem_id, AC_SEMSET_ID, 1);
  if (r<0) {
    // error
#ifdef SHMC_DEBUG_TRACE
    perror("shmcAccept");
#endif
    return NULL;
  }

  // wait until anyone call Connect
  r = wait_sem_n(shmc->sem_id, AC_SEMSET_ID, 2);
  if (r<0) {
    // error
#ifdef SHMC_DEBUG_TRACE
    perror("shmcAccept");
#endif
    return NULL;
  }

  // regist a new connection 
  for (conn_id=0; conn_id<MAX_CONN; conn_id++) {
    // search an empty cell
    if (shmc->accept[conn_id]==NULL) {
      // allocate for new connection
      break;
    }
  }
  if (conn_id >= MAX_CONN) {
    reset_sem_n(shmc->sem_id, AC_SEMSET_ID, 1);
    // error EMFILE
    errno = EMFILE;
#ifdef SHMC_DEBUG_TRACE
    perror("shmcAccept");
#endif
    return NULL;
  }

  Shmc *nshmc = newShmc(shmc);
  nshmc->type = SHMC_ACCEPT;
  if (nshmc==NULL) {
    reset_sem_n(shmc->sem_id, AC_SEMSET_ID, 1);
    // ENOMEM
#ifdef SHMC_DEBUG_TRACE
    perror("shmcAccept");
#endif
    return NULL;
  }

  ShmcConn *nshmccon = (ShmcConn*)malloc(sizeof(ShmcConn));
  if (nshmccon==NULL) {
#ifdef SHMC_DEBUG_TRACE
    perror("shmcAccept");
#endif
    reset_sem_n(shmc->sem_id, AC_SEMSET_ID, 1);
    // ENOMEM
    free(nshmc);
    return NULL;
  }
  shmc->accept[conn_id] = (void *)nshmccon;
  nshmc->connect = nshmccon;
  
  // fprintf(stderr, "nshmc->connect is %x\n",nshmc->connect);
  nshmccon->conn_id = conn_id;
  int shmc_key = ftok(shmc->app_key, conn_id+1); // 
#ifdef NO_IPC_EXCL
  nshmccon->shm_id = shmget(shmc_key, SHMC_SIZE, IPC_CREAT|0666);
#else
  nshmccon->shm_id = shmget(shmc_key, SHMC_SIZE, IPC_CREAT|IPC_EXCL|0666);
#endif
  if (nshmccon->shm_id<0) {
#ifdef SHMC_DEBUG_TRACE
    perror("shmcAccept");
#endif
    reset_sem_n(shmc->sem_id, AC_SEMSET_ID, 1);
    // error  EBUSY
#ifdef SHMC_DEBUG_TRACE
    perror("shmget");
#endif
    free(nshmc);
    free(nshmccon);
    return NULL;
  }
  //
  nshmccon->send_sem_id = conn_id*2;
  nshmccon->recv_sem_id = conn_id*2+1;
  nshmccon->sendb_sem_id = conn_id*2;
  nshmccon->recvb_sem_id = conn_id*2+1;
  nshmccon->s_select_semset = conn_id*2;
  nshmccon->r_select_semset = conn_id*2+1;

  nshmccon->shmp = (char *)shmat(nshmccon->shm_id, NULL, 0);
  if (nshmccon->shmp == (void*)-1) {
#ifdef SHMC_DEBUG_TRACE
    perror("shmcAccept");
#endif
    reset_sem_n(shmc->sem_id, AC_SEMSET_ID, 1);
    shmctl(nshmccon->shm_id, IPC_RMID, NULL);
    free(nshmc);
    free(nshmccon);
    // error
    return NULL;
  }
  //
  nshmccon->status = (int *)nshmccon->shmp;
  *(nshmccon->status) = SHMC_INIT;
  // for shmcSend
  nshmccon->send = nshmccon->shmp+3*sizeof(int);
  nshmccon->send_head = (int *)(nshmccon->shmp+sizeof(int));
  nshmccon->send_length = (int *)(nshmccon->shmp+2*sizeof(int));
  *(nshmccon->send_head) = 0;
  *(nshmccon->send_length) = 0;
  
  // for shmcRecv
  nshmccon->recv = nshmccon->shmp + SEND_SIZE+6*sizeof(int);
  nshmccon->recv_head = (int *)(nshmccon->shmp+SEND_SIZE+4*sizeof(int));
  nshmccon->recv_length = (int *)(nshmccon->shmp+SEND_SIZE+5*sizeof(int));
  *(nshmccon->recv_head) = 0;
  *(nshmccon->recv_length) = 0;
  // 
  int *shmp = (int *)(shmc->connect->shmp);
  shmp[0] =conn_id;
  //
  nshmccon->s_sem_id = (int *)(nshmccon->shmp+SHMD_SIZE);
  *(nshmccon->s_sem_id) = 0;
  nshmccon->s_semset_id = (int *)(nshmccon->shmp+SHMD_SIZE+sizeof(void*));
  *(nshmccon->s_semset_id) = 0;
  nshmccon->s_flags = (int *)(nshmccon->shmp+SHMD_SIZE+sizeof(void*)*2);
  *(nshmccon->s_flags) = 0;

  nshmccon->r_sem_id = (int *)(nshmccon->shmp+SHMD_SIZE+SHMS_SIZE);
  *(nshmccon->r_sem_id) = 0;
  nshmccon->r_semset_id = (int *)(nshmccon->shmp+SHMD_SIZE+SHMS_SIZE+sizeof(void*));
  *(nshmccon->r_semset_id) = 0;
  nshmccon->r_flags = (int *)(nshmccon->shmp+SHMD_SIZE+SHMS_SIZE+sizeof(void*)*2);
  *(nshmccon->r_flags) = 0;

  // 
  *(nshmccon->status) = SHMC_ACCEPT;

  // pass ACCEPTED to the other side
  r = reset_sem_n(shmc->sem_id, AC_SEMSET_ID, 3);
  if (r<0) {
    // error
#ifdef SHMC_DEBUG_TRACE
    perror("shmcAccept");
#endif
    shmctl(nshmccon->shm_id, IPC_RMID, NULL);
    free(nshmc);
    free(nshmccon);
    return NULL;
  }
  // wait COMPLETE from the other side
  // r = timedwait_sem_n(shmc->sem_id, AC_SEMSET_ID, 4);  //
  r = wait_sem_n(shmc->sem_id, AC_SEMSET_ID, 4);  //
  if (r<0) {
    // error
#ifdef SHMC_DEBUG_TRACE
    perror("shmcAccept");
#endif
    shmctl(nshmccon->shm_id, IPC_RMID, NULL);
    free(nshmc);
    free(nshmccon);
    return NULL;
  }
  //
  shmp[0] = 0;
  // 
  nshmc->close = closeAcceptSide;
  //
  reset_sem_n(shmc->sem_id, nshmccon->send_sem_id, 1);
  reset_sem_n(shmc->sem_id, nshmccon->recv_sem_id, 1);
  reset_sem_n(shmc->sem_id_b, nshmccon->sendb_sem_id, 0);
  reset_sem_n(shmc->sem_id_b, nshmccon->recvb_sem_id, 0);

  reset_sem_n(shmc->sem_id_s, nshmccon->send_sem_id, 1);
  reset_sem_n(shmc->sem_id_s, nshmccon->recv_sem_id, 1);
  reset_sem_n(shmc->sem_id_w, nshmccon->send_sem_id, 1); // 04.28.2008
  reset_sem_n(shmc->sem_id_w, nshmccon->recv_sem_id, 1); // 04.28.2008

  // fprintf(stderr, "shmcAccept: step 5, conn_id is %d\n", conn_id);
  // pass COMPLETE to ther other side
  r = reset_sem_n(shmc->sem_id, AC_SEMSET_ID, 5);
  if (r<0) {
    // error
#ifdef SHMC_DEBUG_TRACE
    perror("shmcAccept");
#endif
    shmctl(nshmccon->shm_id, IPC_RMID, NULL);
    free(nshmc);
    free(nshmccon);
    return NULL;
  }
  //
  ShmcConn *conn = shmc->connect;
  reset_sem_n(shmc->sem_id_s, conn->s_select_semset, 1);
  reset_sem_n(shmc->sem_id_s, conn->r_select_semset, 1);
  reset_sem_n(shmc->sem_id_w, conn->s_select_semset, 1);
  reset_sem_n(shmc->sem_id_w, conn->r_select_semset, 1);

#ifdef DEBUG
  fprintf(stderr, "shmcAccept: complete\n");
#endif
  return nshmc;
}

/*
 *
 */
int shmcConnect(Shmc *shmc) 
{
  int r;
  int conn_id;
  //
  shmc->sem_id = semget(shmc->shmc_key, SEM_SIZE, 0);
  if(shmc->sem_id < 0)
    return -1;

  //
  shmc->sem_id_b = semget(ftok(shmc->app_key, MAX_CONN*2+2), SEM_SIZE, 0);
  if(shmc->sem_id_b < 0)
    return -1;

  shmc->sem_id_s = semget(ftok(shmc->app_key, MAX_CONN*2+3), SEM_SIZE, 0);
  if(shmc->sem_id_s < 0)
    return -1;

  shmc->sem_id_w = semget(ftok(shmc->app_key, MAX_CONN*2+4), SEM_SIZE, 0);
  if(shmc->sem_id_w < 0)
    return -1;

  // fprintf(stderr, "shmcConnect: step 0\n");
  //
  int shm_id = shmget(shmc->shmc_key, SHMB_SIZE, 0);
  if(shm_id < 0) {
    // error 
#ifdef SHMC_DEBUG_TRACE
    perror("shmcConnect");
#endif
    return -1;
  }
  int *shmp = (int *)shmat(shm_id, NULL, 0);
  //
  // r = timedwait_sem_n(shmc->sem_id, CONNECT_SEMSET_ID, 1);
  r = wait_sem_n(shmc->sem_id, CONNECT_SEMSET_ID, 1);
  if (r<0) {
    // error
#ifdef SHMC_DEBUG_TRACE
    perror("shmcConnect");
#endif
    return -1;
  }

  //
  shmp[0] = getpid();
  //
  {
    int *r_sem_id = shmp+2;
    int *r_semset_id = shmp+3;
    //    fprintf(stderr, "shmcConnecct: r_sem_id is %d, r_semset_id is %d\n",
    //    *r_sem_id, *r_semset_id);
    if (*r_sem_id != 0 && *r_semset_id != 0) {
      reset_sem_n(*r_sem_id, *r_semset_id, 5);
    }
  }

  //
  // r = timedwait_sem_n(shmc->sem_id, AC_SEMSET_ID, 1);
  r = wait_sem_n(shmc->sem_id, AC_SEMSET_ID, 1);
  if (r<0) {
    // error
#ifdef SHMC_DEBUG_TRACE
    perror("shmcConnect");
#endif
    reset_sem_n(shmc->sem_id, CONNECT_SEMSET_ID, 1);
    return -1;
  }

  // 
  r = reset_sem_n(shmc->sem_id, AC_SEMSET_ID, 2);
  if (r<0) {
    // error
#ifdef SHMC_DEBUG_TRACE
    perror("shmcConnect");
#endif
    reset_sem_n(shmc->sem_id, CONNECT_SEMSET_ID, 1);
    return -1;
  }

  // wait ACCEPTED from the other side
  // r = timedwait_sem_n(shmc->sem_id, AC_SEMSET_ID, 3);
  r = wait_sem_n(shmc->sem_id, AC_SEMSET_ID, 3);
  if (r<0) {
    // error
#ifdef SHMC_DEBUG_TRACE
    perror("shmcConnect");
#endif
    reset_sem_n(shmc->sem_id, CONNECT_SEMSET_ID, 1);
    return -1;
  }
#ifdef DEBUG
  fprintf(stderr, "shmcConnect: step 3\n");
#endif

  // get the connection-id from the other side
  conn_id = shmp[0];
  // fprintf(stderr, "shmcConnect: conn_id is %d\n",conn_id);

  key_t shmc_key = ftok(shmc->app_key, conn_id+1);
  ShmcConn *shmccon = (ShmcConn *)malloc(sizeof(ShmcConn));
  if (shmccon==NULL) {
    // ENOMEM
#ifdef SHMC_DEBUG_TRACE
    perror("shmcConnect");
#endif
    reset_sem_n(shmc->sem_id, AC_SEMSET_ID, 1);
    reset_sem_n(shmc->sem_id, CONNECT_SEMSET_ID, 1);
    return -1;
  }
  //
  shmc->connect = shmccon;
  shmccon->conn_id = conn_id;
  // shmccon->state = 0;
  shmccon->shm_id = shmget(shmc_key, SHMC_SIZE, 0);
  if (shmccon->shm_id<0) {
    // error
#ifdef SHMC_DEBUG_TRACE
    perror("shmcConnect");
#endif
    reset_sem_n(shmc->sem_id, AC_SEMSET_ID, 1);
    reset_sem_n(shmc->sem_id, CONNECT_SEMSET_ID, 1);
    free(shmccon);
    return -1;
  }
  shmccon->send_sem_id = conn_id*2+1;
  shmccon->recv_sem_id = conn_id*2;
  // shmccon->send_sem_id = conn_id*4+1;
  // shmccon->recv_sem_id = conn_id*4;
  shmccon->sendb_sem_id = conn_id*2+1;
  shmccon->recvb_sem_id = conn_id*2;
  // shmccon->sendb_sem_id = conn_id*4+3;
  // shmccon->recvb_sem_id = conn_id*4+2;

  shmccon->s_select_semset = conn_id*2+1;
  shmccon->r_select_semset = conn_id*2;

  shmccon->shmp = (char *)shmat(shmccon->shm_id, NULL, 0);
  if (shmccon->shmp == (void*)-1) {
    // error
#ifdef SHMC_DEBUG_TRACE
    perror("shmcConnect");
#endif
    reset_sem_n(shmc->sem_id, AC_SEMSET_ID, 1);
    reset_sem_n(shmc->sem_id, CONNECT_SEMSET_ID, 1);
    free(shmccon);
    return -1;
  }

  //
  shmccon->status = (int *)shmccon->shmp;  // it must be SHMC_ACCEPT
  // for shmcSend
  shmccon->send = shmccon->shmp+SEND_SIZE+6*sizeof(int);
  shmccon->send_head = (int *)(shmccon->shmp+SEND_SIZE+4*sizeof(int));
  shmccon->send_length = (int *)(shmccon->shmp+SEND_SIZE+5*sizeof(int));
  // for shmcRecv
  shmccon->recv = shmccon->shmp+3*sizeof(int);
  shmccon->recv_head = (int *)(shmccon->shmp+sizeof(int));
  shmccon->recv_length = (int *)(shmccon->shmp+2*sizeof(int));

  //
  shmccon->s_sem_id = (int *)(shmccon->shmp+SHMD_SIZE+SHMS_SIZE);
  shmccon->s_semset_id = (int *)(shmccon->shmp+SHMD_SIZE+SHMS_SIZE+sizeof(void*));
  shmccon->s_flags = (int *)(shmccon->shmp+SHMD_SIZE+SHMS_SIZE+sizeof(void*)*2);

  shmccon->r_sem_id = (int *)(shmccon->shmp+SHMD_SIZE);
  shmccon->r_semset_id = (int *)(shmccon->shmp+SHMD_SIZE+sizeof(void*));
  shmccon->r_flags = (int *)(shmccon->shmp+SHMD_SIZE+sizeof(void*)*2);

  //
  *(shmccon->status) = SHMC_CONNECT;   // connection has completed
  // pass COMPLETE to the other side
  r = reset_sem_n(shmc->sem_id, AC_SEMSET_ID, 4);  //
  if (r<0) {
    // error
#ifdef SHMC_DEBUG_TRACE
    perror("shmcConnect");
#endif
    reset_sem_n(shmc->sem_id, CONNECT_SEMSET_ID, 1);
    free(shmccon);
    return -1;
  }
  // then wait until the other side got the semaphore,
  // r = timedwait_sem_n(shmc->sem_id, AC_SEMSET_ID, 5);   //
  r = wait_sem_n(shmc->sem_id, AC_SEMSET_ID, 5);   //
  if (r<0) {
    // error
#ifdef SHMC_DEBUG_TRACE
    perror("shmcConnect");
#endif
    perror("wait_sem_n: 3");
    free(shmccon);
    return -1;
  }

  //
  r = reset_sem_n(shmc->sem_id, AC_SEMSET_ID, 0);
  if (r<0) {
    // error
#ifdef SHMC_DEBUG_TRACE
    perror("shmcConnect");
#endif
    free(shmccon);
    return -1;
  }

  //
  r = reset_sem_n(shmc->sem_id, CONNECT_SEMSET_ID, 1);
  if (r<0) {
    // error
#ifdef SHMC_DEBUG_TRACE
    perror("shmcConnect");
#endif
    free(shmccon);
    return -1;
  }
  // fprintf(stderr, "shmcConnect: complete\n");
  //
  shmc->close = closeConnectSide;
  shmc->type = SHMC_CONNECT;
  // completed
  return 0;
}

#if 0
//
static const int giga = 1000000000;
static const int mega = 1000000;
//
static struct timespec *sub_timespec(struct timespec *ts1, struct timespec *ts2, struct timespec *rts)
{
  rts->tv_sec = ts1->tv_sec - ts2->tv_sec;
  rts->tv_nsec= ts1->tv_nsec - ts2->tv_nsec;
  if (rts->tv_nsec < 0) {
    rts->tv_sec--;
    rts->tv_nsec += giga;
  }
  return rts;
}

//
static int cmp_timespec(struct timespec *ts1, struct timespec *ts2)
{
  if (ts1->tv_sec > ts2->tv_sec)
    return 1;
  else if (ts1->tv_sec < ts2->tv_sec)
    return -1;
  else if (abs(ts1->tv_nsec - ts2->tv_nsec) < mega)
    return 0;
  else if (ts1->tv_nsec > ts2->tv_nsec)
    return 1;
  else 
    return -1;
}


//
static int cur_timespec(struct timespec *ts)
{
  struct timeval tv;
  if (gettimeofday(&tv, NULL) < 0) {
    return -1;
  }
  ts->tv_sec = tv.tv_sec;
  ts->tv_nsec = tv.tv_usec * 1000;
  return 0; 
}
#endif

/*
 */
static int is_selected(Shmc *shmc, int flag)
{
  int r = 0;
  // if (timedwait_sem_n(shmc->sem_id_s, shmc->connect->s_select_semset,1)<0)
  if (wait_sem_n(shmc->sem_id_s, shmc->connect->s_select_semset, 1) < 0)
    return -1;

  int sem_id = *(shmc->connect->r_sem_id);
  int semset_id = *(shmc->connect->r_semset_id);
  int rflag = *(shmc->connect->r_flags) ;
  // fprintf(stderr, "shmc->connect->r_flags is 0x%x\n", (unsigned int)shmc->connect->r_flags);
  // fprintf(stderr, "is_selected: shmc is 0x%x, sem_id is %d, semset_id is %d,"
  //	  " r_flags is %d\n",
  //	  (unsigned int)shmc, sem_id, semset_id, rflag);
  if (sem_id != 0 && (rflag & flag) != 0) {
    r = 1;
    reset_sem_n(sem_id, semset_id, 5);
  }
  if (reset_sem_n(shmc->sem_id_s, shmc->connect->s_select_semset,1)<0)
    return -1;
  return r;
}

/*
 */
static int recv(Shmc *shmc, char *buf, size_t size)
{
  int rsize=0;   // size of actually received data.
  int head, length;
  int dsize;

  ShmcConn *conn = shmc->connect;

  // if (timedwait_sem_n(shmc->sem_id, conn->recv_sem_id, 1) < 0) {
  if (wait_sem_n(shmc->sem_id, conn->recv_sem_id, 1) < 0) { // hold
    // error
#ifdef SHMC_DEBUG_TRACE
    perror("shmcRecvB");
#endif
    // shmc->close(shmc);
    return -1;
  }
  // 
  head = *(conn->recv_head);
  length = *(conn->recv_length);
  if (reset_sem_n(shmc->sem_id, conn->recv_sem_id, 1) < 0) // release
    return -1;

#ifdef SHMC_DEBUG_TRACE
  fprintf(stderr, "recv: head is %d, length is %d\n", head, length);
#endif

  if (length==0) {
    // empty
    return 0;
  } else {
    if (size > length)
      size = length;
    else 
      length = size;
    // get available size at one time.
    if ((head + size) > RECV_SIZE)
      dsize = RECV_SIZE - head;
    else
      dsize = size;
    memcpy((void *)buf, (void*)(conn->recv + head), (size_t)dsize);
    rsize = dsize;
    head += dsize;
    buf += dsize;
    size -= dsize;
    // return to the head
    if (head >= RECV_SIZE) {
      head -= RECV_SIZE; // it must be 0
    }
    // when it's remains.
    if (size > 0) {
      memcpy((void *)buf, (void*)conn->recv, (size_t)size);
      rsize += size;
      head += size;
    }

    // 
    if (wait_sem_n(shmc->sem_id, conn->recv_sem_id, 1) < 0) { // hold
      // error
#ifdef SHMC_DEBUG_TRACE
      perror("shmcRecvB");
#endif
      return -1;
    }
    // update exclusively the recv_head and the recv_length.
    *(conn->recv_head) = head;
    *(conn->recv_length) -= rsize;
#ifdef SHMC_DEBUG_TRACE
    fprintf(stderr, "shmcRecv: head is %d, length is %d\n",
	    *(conn->recv_head), *(conn->recv_length));
#endif

    if (reset_sem_n(shmc->sem_id, conn->recv_sem_id, 1) < 0) // release
      return -1;

    return rsize;
  }
  return -1;
}

/*
 */
static int recvB(Shmc *shmc, char *buf, size_t size, int blocking)
{
  if (shmc==NULL || shmc->connect==NULL || buf==NULL || size<0)
    return -1;
  if (size==0)
    return 0;
  ShmcConn *conn = shmc->connect;

#ifdef SHMC_DEBUG_TRACE
  fprintf(stderr, "shmcRecvB: sem_id is %d, recv_sem_id is %d\n", shmc->sem_id, conn->recv_sem_id);
#endif

  if (get_shm_nattach(conn->shm_id) <= 1) {
    *(conn->status) = SHMC_CLOSE;
    is_selected(shmc, SEL_W);
    return -1;
  }

  while (1) {
#ifdef SHMC_DEBUG_TRACE
    fprintf(stderr, "shmcRecvB: status is %d\n", *(conn->status));
#endif
    if (*(conn->status)==SHMC_CLOSE) {
      // already closed
      reset_sem_n(shmc->sem_id_b, conn->recvb_sem_id, 5);
      is_selected(shmc, SEL_W);
      return -1;
    }
    //
    int r = recv(shmc, buf, size);

    if (r < 0) {
      *(conn->status) = SHMC_CLOSE;
      reset_sem_n(shmc->sem_id_b, conn->recvb_sem_id, 5);
      is_selected(shmc, SEL_W);
      return -1;
    } else if (r==0) {
      if (blocking) {
	if (wait_sem_n(shmc->sem_id_b, conn->recvb_sem_id, 5) < 0)
	  return -1;
      } else {
	return r;
      }
    } else {
      // fprintf(stderr, "shmcRecvB: r is %d\n", r);
      is_selected(shmc, SEL_W);
      if (reset_sem_n(shmc->sem_id_b, conn->recvb_sem_id, 5) < 0)
	return -1;
      return r;
    }
  }
}

/*
 *
 */
int shmcRecvB(Shmc *shmc, char *buf, size_t size, struct timespec *timeout)
{
  return recvB(shmc, buf, size, 1);
}

//
int shmcRecv(Shmc *shmc, char *buf, size_t size)
{
  return recvB(shmc, buf, size, 0);
}

/*
 */
static int send(Shmc *shmc, char *data, size_t size)
{
  int rsize=0;   // size of actually sent data.
  int head, length, tail, space;
  int dsize;

  ShmcConn *conn = shmc->connect;
  //
  if (wait_sem_n(shmc->sem_id, conn->send_sem_id, 1) < 0) { // hold
    // error
#ifdef SHMC_DEBUG_TRACE
    perror("shmcSend");
#endif
    return -1;
  }
  head = *(conn->send_head);
  length = *(conn->send_length);
  if (reset_sem_n(shmc->sem_id, conn->send_sem_id, 1) < 0) // release
    return -1;

  space = SEND_SIZE - length;
  if (space <= 0) {
    // buffer is FULL
    return 0;
  } else {
#ifdef SHMC_DEBUG_TRACE
    fprintf(stderr, "send: head is %d, length is %d\n",head, length);
#endif
    if (size > space)
      size = space;
    else 
      space = size;
    tail = head + length;
    if (tail >= SEND_SIZE) {
      tail -= SEND_SIZE;
      dsize = size;
    } else
      if ((tail + size) > SEND_SIZE)
	dsize = SEND_SIZE - tail ;
      else
	dsize = size;
#ifdef SHMC_DEBUG_TRACE
    fprintf(stderr, "send: tail is %d, dsize is %d\n", tail, dsize);
#endif
    memcpy((void*)(conn->send + tail), (void *)data, (size_t)dsize);

    rsize = dsize;
    data += dsize;
    size -= dsize;
    if (size > 0) {
      memcpy((void*)conn->send, (void *)data, (size_t)size);
      rsize += size;
    }

    //
    if (wait_sem_n(shmc->sem_id, conn->send_sem_id, 1) < 0) {
      // error
#ifdef SHMC_DEBUG_TRACE
      perror("shmcSend");
#endif
      return -1;
    }
    *(conn->send_length) += rsize;
    if (reset_sem_n(shmc->sem_id, conn->send_sem_id, 1) < 0)
      return -1;

    //
    return rsize;
  }
}

/*
 */
static int sendB(Shmc *shmc, char *data, size_t size, int blocking)
{
  if (shmc==NULL || shmc->connect==NULL || data==NULL || size<0)
    return -1;
  if (size==0)
    return 0;

  ShmcConn *conn = shmc->connect;

#ifdef SHMC_DEBUG_TRACE
  fprintf(stderr, "shmcSend: sem_id is %d, send_sem_id is %d\n", shmc->sem_id, conn->send_sem_id);
#endif
  //
  if (get_shm_nattach(conn->shm_id) <= 1) {
    *(conn->status) = SHMC_CLOSE;
    is_selected(shmc, SEL_R);
    return -1;
  }

  while (1) {
    //
    if (*(conn->status)==SHMC_CLOSE) {
      // already closed
      reset_sem_n(shmc->sem_id, conn->send_sem_id, 1);
      reset_sem_n(shmc->sem_id_b, conn->sendb_sem_id, 5);
      is_selected(shmc, SEL_R);
      return -1;
    }

    // int r = send(conn, data+rsize, size);
    int r = send(shmc, data, size);
    if (r < 0) {
      // error
      perror("send");
      *(conn->status) = SHMC_CLOSE;     // 04.23.2008
      reset_sem_n(shmc->sem_id_b, conn->sendb_sem_id, 5);
      is_selected(shmc, SEL_R);
      return -1;
    } else if (r==0) {
      if (blocking) {
	if (wait_sem_n(shmc->sem_id_b, conn->sendb_sem_id, 5) < 0)
	  return -1;
      } else {
	return r;
      }
    } else {
      if (reset_sem_n(shmc->sem_id_b, conn->sendb_sem_id, 5) < 0)
	return -1;
      is_selected(shmc, SEL_R);
      return r;
    }
  }
}

/*
 *
 */
int shmcSendB(Shmc *shmc, char *data, size_t size, struct timespec *timeout)
{
  return sendB(shmc, data, size, 1);
}

//
int shmcSend(Shmc *shmc, char *data, size_t size)
{
  return sendB(shmc, data, size, 0);
}

//
static int is_available2send(Shmc *shmc)
{
  ShmcConn *conn = shmc->connect;

  if (shmc->type == 0)
    return 1;
  if (shmc->type == SHMC_BIND) {
    int *shmp = (int *)(conn->shmp);
    if (shmp[0]!=0)  // anyone requiring a new connection
      return 1;
    else
      return 0;
  }
  if(*(conn->status)==SHMC_CLOSE)
    return 1;

  if (get_shm_nattach(conn->shm_id) <= 1) {
    return 1;
  }

  // fprintf(stderr, "is_available2send: \n");
  if (*(conn->send_length) < SEND_SIZE) {
    return 1;
  } else {
    return 0;
  }
}

//
static int is_available2recv(Shmc *shmc)
{
  ShmcConn *conn = shmc->connect;
  
  //
  if (shmc->type == 0)
    return 1;
  if (shmc->type == SHMC_BIND) {
    int *shmp = (int *)(conn->shmp);
    // fprintf(stderr, "is_available2recv: SHMC_BIND %d\n", shmp[0]);
    if (shmp[0]!=0)
      return 1;
    else
      return 0;
  }
  if(*(conn->status)==SHMC_CLOSE)
    return 1;

  if (get_shm_nattach(conn->shm_id) <= 1) {
    return 1;
  }
  // fprintf(stderr, "is_available2recv: \n");

  if (*(conn->recv_length)>0) {
    return 1;
  } else {
    return 0;
  }
}

//
static int check_available(int r_num, ShmcSet *set_r,
			   int w_num, ShmcSet *set_w,
			   Shmc **shmc)
{
  int i, r;
  int count = 0;

  for (i=0; i<r_num; i++) {
    if (*shmc==NULL)
      *shmc = set_r[i].shmc;
    r = is_available2recv(set_r[i].shmc);
    if (r < 0)
      return -1;
    else if (r > 0) {
      count++;
      set_r[i].set = 1;
    }
  }

  for (i=0; i<w_num; i++) {
    if (*shmc==NULL)
      *shmc = set_w[i].shmc;
    r = is_available2send(set_w[i].shmc);
    if (r < 0)
      return -1;
    else if (r > 0) {
      count++;
      set_w[i].set = 1;
    }
  }
  return count;
}

//
static int setSelectFlags(int sem_id,int semset_id, int flag, int num, ShmcSet *set)
{
  int i;
  for (i=0; i<num; i++) {
    Shmc *shmc = set[i].shmc;
    ShmcConn *conn = shmc->connect;
    // fprintf(stderr, "setSelectFlags: sem_id %d, semset_id %d, flag is %d\n",
    //	    sem_id, semset_id, flag);

    if (wait_sem_n(shmc->sem_id_s,conn->s_select_semset, 1) < 0)
      return -1;
    *(conn->s_sem_id) = sem_id;
    *(conn->s_semset_id) = semset_id;
    // fprintf(stderr, "setSelectFlags: *(conn->s_flags) %d\n",*(conn->s_flags));
    *(conn->s_flags) |= flag;
    // fprintf(stderr, "setSelectFlags: *(conn->s_flags) %d\n",*(conn->s_flags));
    if (reset_sem_n(shmc->sem_id_s, conn->s_select_semset, 1)<0)
      return -1;
  }
  return 0;
}

//
static int resetSelectFlags(int num, ShmcSet *set, int flag)
{
  int i;
  // fprintf(stderr, "resetSelectFlags: start\n");
  for (i=0; i<num; i++) {
    Shmc *shmc = set[i].shmc;
    ShmcConn *conn = shmc->connect;
    if (wait_sem_n(shmc->sem_id_s, conn->s_select_semset, 1) < 0)
      return -1;

    *(conn->s_flags) &= ~flag;
    // fprintf(stderr, "resetSelectFlags: *(conn->s_flags) %d\n",*(conn->s_flags));
    if (reset_sem_n(shmc->sem_id_s, conn->s_select_semset, 1)<0)
      return -1;
  }
  return 0;
}

/*
 *
 */
int shmcSelect(int r_num, ShmcSet *set_r, int w_num, ShmcSet *set_w,
			 struct timespec *timeout)
{
  int count;
  Shmc *sel = NULL;   // 

  count = check_available(r_num, set_r, w_num, set_w, &sel);
  // fprintf(stderr, "shmcSelect: check_available, r_num is %d, w_num is %d,"
  //  " count is %d\n",r_num, w_num, count);

  if (sel==NULL)
    return 0;
  if (count>0)
    return count;

  // if (timeout!=NULL && timeout->tv_sec==0 && timeout->tv_nsec==0)
  //  return 0;

  int sem_id;
  int semset_id;

  sem_id = sel->sem_id_w; // <---
  semset_id = sel->connect->s_select_semset;

  // fprintf(stderr, "shmcSelect: sem_id is %d, semset_id is %d\n",
  //	  sem_id, semset_id);
  if (setSelectFlags(sem_id, semset_id, SEL_R, r_num, set_r) < 0)
    return -1;

  if (setSelectFlags(sem_id, semset_id, SEL_W, w_num, set_w) < 0)
    return -1;

  //
  count = check_available(r_num, set_r, w_num, set_w, &sel);
  // fprintf(stderr, "shmcSelect: check_available, r_num is %d, w_num is %d,"
  //  " count is %d\n",r_num, w_num, count);

  if (count>0) {
    resetSelectFlags(r_num, set_r, SEL_R);
    resetSelectFlags(w_num, set_w, SEL_W);
    return count;
  }
  sem_id = sel->sem_id_w;
  semset_id = sel->connect->s_select_semset;

  //
#if 0
  struct timespec ts;
  if (timeout!=NULL && cur_timespec(&ts) < 0) {
    perror("cur_timespec");
    return -1;
  }
  struct timespec rts;
  struct timespec *cur_timeout = timeout;
#endif
  //
  while (1) {
    // fprintf(stderr, "shmcSelect: sem_id is %d, semset_id is %d\n",
    //	    sem_id, semset_id);
    // int r=wait_sem_n_wtm(sem_id, semset_id, 5, cur_timeout);
    int r=wait_sem_n(sem_id, semset_id, 5);

    // fprintf(stderr, "shmcSelect: wait_sem_n_wtm %d \n", r);
    count = check_available(r_num, set_r, w_num, set_w, &sel);
    if (count > 0) {
      // fprintf(stderr, "shmcSelect: count is %d\n", count);
      resetSelectFlags(r_num, set_r, SEL_R);
      resetSelectFlags(w_num, set_w, SEL_W);
      if (reset_sem_n(sem_id, semset_id, 1)<0) {
	perror("reset_sem_n");
	return -1;
      }
      // fprintf(stderr, "shmcSelect: return %d\n", count);
      return count;
    }
    if (r < 0) {
      if (errno==EINTR) {
#if 0
	if (timeout != NULL) {
	  struct timespec newts;
	  if (cur_timespec(&newts) < 0) {
	    perror("cur_timespec");
	    resetSelectFlags(r_num, set_r, SEL_R);
	    resetSelectFlags(w_num, set_w, SEL_W);
	    reset_sem_n(sem_id, semset_id, 1);
	    return -1;
	  }
	  struct timespec *restts;
	  restts = sub_timespec(&newts, &ts, &rts);
	  int tr = cmp_timespec(restts, timeout);
	  if (tr >= 0) {
	    // timeout 
	    resetSelectFlags(r_num, set_r, SEL_R);
	    resetSelectFlags(w_num, set_w, SEL_W);
	    if (reset_sem_n(sem_id, semset_id, 1)<0)
	      return -1;
	    return 0;
	  } else {
	    // not yet timeout
	    timeout = sub_timespec(timeout, restts, &rts);
	  }
	} else {
	  return -1;
	}
#else
	return -1;
#endif
      } else if (errno==EAGAIN) {
	// timeout 
	resetSelectFlags(r_num, set_r, SEL_R);
	resetSelectFlags(w_num, set_w, SEL_W);
	if (reset_sem_n(sem_id, semset_id, 1)<0)
	  return -1;
	return 0;
      } else {
	resetSelectFlags(r_num, set_r, SEL_R);
	resetSelectFlags(w_num, set_w, SEL_W);
	reset_sem_n(sem_id, semset_id, 1);
	return -1;
      }
    } else {
      resetSelectFlags(r_num, set_r, SEL_R);
      resetSelectFlags(w_num, set_w, SEL_W);
      if (reset_sem_n(sem_id, semset_id, 1)<0)
	return -1;
      return 0;
    }
  }
}
/*
 */
