/*
 * Copyright (C) 2006 uguu@users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.expression_computer;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Iterator;

import jp.sourceforge.expression_computer.command.AddCommand;
import jp.sourceforge.expression_computer.command.AndCommand;
import jp.sourceforge.expression_computer.command.ArithmeticRightShiftCommand;
import jp.sourceforge.expression_computer.command.BitReversingCommand;
import jp.sourceforge.expression_computer.command.ConditionCommand;
import jp.sourceforge.expression_computer.command.ConditionalAndCommand;
import jp.sourceforge.expression_computer.command.ConditionalOrCommand;
import jp.sourceforge.expression_computer.command.DivideCommand;
import jp.sourceforge.expression_computer.command.EqualCommand;
import jp.sourceforge.expression_computer.command.ExclusiveOrCommand;
import jp.sourceforge.expression_computer.command.FunctionCallCommand;
import jp.sourceforge.expression_computer.command.GreaterThanCommand;
import jp.sourceforge.expression_computer.command.GreaterThanEqualCommand;
import jp.sourceforge.expression_computer.command.InclusiveOrCommand;
import jp.sourceforge.expression_computer.command.LeftShiftCommand;
import jp.sourceforge.expression_computer.command.LessThanCommand;
import jp.sourceforge.expression_computer.command.LessThanEqualCommand;
import jp.sourceforge.expression_computer.command.LogicalRightShiftCommand;
import jp.sourceforge.expression_computer.command.MultiplyCommand;
import jp.sourceforge.expression_computer.command.NotCommand;
import jp.sourceforge.expression_computer.command.NotEqualCommand;
import jp.sourceforge.expression_computer.command.PostDecrementCommand;
import jp.sourceforge.expression_computer.command.PostIncrementCommand;
import jp.sourceforge.expression_computer.command.PreDecrementCommand;
import jp.sourceforge.expression_computer.command.PreIncrementCommand;
import jp.sourceforge.expression_computer.command.PushStackCommand;
import jp.sourceforge.expression_computer.command.SetVariableCommand;
import jp.sourceforge.expression_computer.command.SignReversingCommand;
import jp.sourceforge.expression_computer.command.SubtractCommand;
import jp.sourceforge.expression_computer.command.SurplusCommand;
import jp.sourceforge.expression_computer.type.FloatingPointLiteral;
import jp.sourceforge.expression_computer.type.IntegerLiteral;
import jp.sourceforge.expression_computer.type.Variable;

/**
 * <p>
 * コマンド リストのシリアライズ、デシリアライズを行う機能を持ちます。バイト配列はJavaのシリアライズ形式とは違い独自形式です。
 * </p>
 * 
 * @author uguu@users.sourceforge.jp
 */
public final class Serializer {

    /**
     * <p>
     * ファイルの終端を表す種類値。
     * </p>
     */
    private static final byte EOF                            = Byte.MAX_VALUE;

    /**
     * <p>
     * {@link AddCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_ADD                    = 0;

    /**
     * <p>
     * {@link AndCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_AND                    = 1;

    /**
     * <p>
     * {@link ArithmeticRightShiftCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_ARITHMETIC_RIGHT_SHIFT = 2;

    /**
     * <p>
     * {@link BitReversingCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_BIT_REVERSING          = 3;

    /**
     * <p>
     * {@link ConditionalAndCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_CONDITIONAL_AND        = 4;

    /**
     * <p>
     * {@link ConditionalOrCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_CONDITIONAL_OR         = 5;

    /**
     * <p>
     * {@link ConditionCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_CONDITION              = 6;

    /**
     * <p>
     * {@link DivideCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_DIVIDE                 = 7;

    /**
     * <p>
     * {@link EqualCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_EQUAL                  = 8;

    /**
     * <p>
     * {@link ExclusiveOrCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_EXCLUSIVE_OR           = 9;

    /**
     * <p>
     * {@link FunctionCallCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_FUNCTION_CALL          = 10;

    /**
     * <p>
     * {@link GreaterThanCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_GREATER_THAN           = 11;

    /**
     * <p>
     * {@link GreaterThanEqualCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_GREATER_THAN_EQUAL     = 12;

    /**
     * <p>
     * {@link InclusiveOrCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_INCLUSIVE_OR           = 13;

    /**
     * <p>
     * {@link LeftShiftCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_LEFT_SHIFT             = 14;

    /**
     * <p>
     * {@link LessThanCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_LESS_THAN              = 15;

    /**
     * <p>
     * {@link LessThanEqualCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_LESS_THAN_EQUAL        = 16;

    /**
     * <p>
     * {@link LogicalRightShiftCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_LOGICAL_RIGHT_SHIFT    = 17;

    /**
     * <p>
     * {@link MultiplyCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_MULTIPLY               = 18;

    /**
     * <p>
     * {@link NotCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_NOT                    = 19;

    /**
     * <p>
     * {@link NotEqualCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_NOT_EQUAL              = 20;

    /**
     * <p>
     * {@link PostDecrementCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_POST_DECREMENT         = 21;

    /**
     * <p>
     * {@link PostIncrementCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_POST_INCREMENT         = 22;

    /**
     * <p>
     * {@link PreDecrementCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_PRE_DECREMENT          = 23;

    /**
     * <p>
     * {@link PreIncrementCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_PRE_INCREMENT          = 24;

    /**
     * <p>
     * {@link PushStackCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_PUSH_STACK             = 25;

    /**
     * <p>
     * {@link SetVariableCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_SET_VARIABLE           = 26;

    /**
     * <p>
     * {@link SignReversingCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_SIGN_REVERSING         = 27;

    /**
     * <p>
     * {@link SubtractCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_SUBTRACT               = 28;

    /**
     * <p>
     * {@link SurplusCommand}クラスの種類値。
     * </p>
     */
    private static final byte COMMAND_SURPLUS                = 29;

    /**
     * <p>
     * {@link FloatingPointLiteral}クラスの種類値。
     * </p>
     */
    private static final byte TYPE_FLOATING_POINT_LITERAL    = 0;

    /**
     * <p>
     * {@link IntegerLiteral}クラスの種類値。
     * </p>
     */
    private static final byte TYPE_INTEGER_LITERAL           = 1;

    /**
     * <p>
     * {@link Variable}クラスの種類値。
     * </p>
     */
    private static final byte TYPE_VARIABLE                  = 2;

    /**
     * <p>
     * コマンド リストをシリアライズし、バイト配列に変換します。
     * </p>
     * 
     * @param commandList
     *            シリアライズするコマンド リスト。<br>
     *            nullの場合、{@link NullPointerException}例外をスローします。
     * @return シリアライズしたバイト配列。
     * @throws IOException
     *             シリアライズに失敗した場合。
     */
    public byte[] serialize(CommandList commandList) throws IOException {
        if (commandList == null) {
            throw new NullPointerException("commandListがnullです。");
        }

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                for (Iterator i = commandList.iterator(); i.hasNext();) {
                    Command c = (Command) i.next();
                    if (c instanceof AddCommand) {
                        objOut.writeByte(Serializer.COMMAND_ADD);
                    } else if (c instanceof AndCommand) {
                        objOut.writeByte(Serializer.COMMAND_AND);
                    } else if (c instanceof ArithmeticRightShiftCommand) {
                        objOut.writeByte(Serializer.COMMAND_ARITHMETIC_RIGHT_SHIFT);
                    } else if (c instanceof BitReversingCommand) {
                        objOut.writeByte(Serializer.COMMAND_BIT_REVERSING);
                    } else if (c instanceof ConditionalAndCommand) {
                        objOut.writeByte(Serializer.COMMAND_CONDITIONAL_AND);
                    } else if (c instanceof ConditionalOrCommand) {
                        objOut.writeByte(Serializer.COMMAND_CONDITIONAL_OR);
                    } else if (c instanceof ConditionCommand) {
                        objOut.writeByte(Serializer.COMMAND_CONDITION);
                    } else if (c instanceof DivideCommand) {
                        objOut.writeByte(Serializer.COMMAND_DIVIDE);
                    } else if (c instanceof EqualCommand) {
                        objOut.writeByte(Serializer.COMMAND_EQUAL);
                    } else if (c instanceof ExclusiveOrCommand) {
                        objOut.writeByte(Serializer.COMMAND_EXCLUSIVE_OR);
                    } else if (c instanceof FunctionCallCommand) {
                        objOut.writeByte(Serializer.COMMAND_FUNCTION_CALL);

                        FunctionCallCommand funcCall = (FunctionCallCommand) c;
                        objOut.writeUTF(funcCall.getName());
                        objOut.writeInt(funcCall.getArgumentNumber());
                    } else if (c instanceof GreaterThanCommand) {
                        objOut.writeByte(Serializer.COMMAND_GREATER_THAN);
                    } else if (c instanceof GreaterThanEqualCommand) {
                        objOut.writeByte(Serializer.COMMAND_GREATER_THAN_EQUAL);
                    } else if (c instanceof InclusiveOrCommand) {
                        objOut.writeByte(Serializer.COMMAND_INCLUSIVE_OR);
                    } else if (c instanceof LeftShiftCommand) {
                        objOut.writeByte(Serializer.COMMAND_LEFT_SHIFT);
                    } else if (c instanceof LessThanCommand) {
                        objOut.writeByte(Serializer.COMMAND_LESS_THAN);
                    } else if (c instanceof LessThanEqualCommand) {
                        objOut.writeByte(Serializer.COMMAND_LESS_THAN_EQUAL);
                    } else if (c instanceof LogicalRightShiftCommand) {
                        objOut.writeByte(Serializer.COMMAND_LOGICAL_RIGHT_SHIFT);
                    } else if (c instanceof MultiplyCommand) {
                        objOut.writeByte(Serializer.COMMAND_MULTIPLY);
                    } else if (c instanceof NotCommand) {
                        objOut.writeByte(Serializer.COMMAND_NOT);
                    } else if (c instanceof NotEqualCommand) {
                        objOut.writeByte(Serializer.COMMAND_NOT_EQUAL);
                    } else if (c instanceof PostDecrementCommand) {
                        objOut.writeByte(Serializer.COMMAND_POST_DECREMENT);
                    } else if (c instanceof PostIncrementCommand) {
                        objOut.writeByte(Serializer.COMMAND_POST_INCREMENT);
                    } else if (c instanceof PreDecrementCommand) {
                        objOut.writeByte(Serializer.COMMAND_PRE_DECREMENT);
                    } else if (c instanceof PreIncrementCommand) {
                        objOut.writeByte(Serializer.COMMAND_PRE_INCREMENT);
                    } else if (c instanceof PushStackCommand) {
                        objOut.writeByte(Serializer.COMMAND_PUSH_STACK);

                        PushStackCommand pushStack = (PushStackCommand) c;
                        ComputeObject compObj = pushStack.getValue();
                        if (compObj instanceof FloatingPointLiteral) {
                            objOut.writeByte(Serializer.TYPE_FLOATING_POINT_LITERAL);
                            objOut.writeDouble(compObj.getValue(null));
                        } else if (compObj instanceof IntegerLiteral) {
                            objOut.writeByte(Serializer.TYPE_INTEGER_LITERAL);
                            objOut.writeLong((long) compObj.getValue(null));
                        } else if (compObj instanceof Variable) {
                            objOut.writeByte(Serializer.TYPE_VARIABLE);
                            objOut.writeUTF(((Variable) compObj).getName());
                        } else {
                            throw new IOException("不明な計算オブジェクトです。{" + compObj + "}");
                        }
                    } else if (c instanceof SetVariableCommand) {
                        objOut.writeByte(Serializer.COMMAND_SET_VARIABLE);
                    } else if (c instanceof SignReversingCommand) {
                        objOut.writeByte(Serializer.COMMAND_SIGN_REVERSING);
                    } else if (c instanceof SubtractCommand) {
                        objOut.writeByte(Serializer.COMMAND_SUBTRACT);
                    } else if (c instanceof SurplusCommand) {
                        objOut.writeByte(Serializer.COMMAND_SURPLUS);
                    } else {
                        throw new IOException("不明なコマンドです。{" + c + "}");
                    }
                }
                objOut.writeByte(Serializer.EOF);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }

        return byteOut.toByteArray();
    }

    /**
     * <p>
     * バイト配列をデシリアライズし、コマンド リストを構築します。
     * </p>
     * 
     * @param data
     *            デシリアライズするバイト配列。<br>
     *            nullの場合、{@link NullPointerException}例外をスローします。
     * @return 構築したコマンド リスト。
     * @throws IOException
     *             デシリアライズに失敗した場合。
     */
    public CommandList deserialize(byte[] data) throws IOException {
        if (data == null) {
            throw new NullPointerException("dataがnullです。");
        }

        CommandList cl = new CommandList();

        ByteArrayInputStream byteIn = new ByteArrayInputStream(data);
        try {
            ObjectInputStream objIn = new ObjectInputStream(byteIn);
            try {
                Command c;
                byte commandType;
                while ((commandType = objIn.readByte()) != Serializer.EOF) {
                    switch (commandType) {
                        case Serializer.COMMAND_ADD:
                            c = new AddCommand();
                            break;
                        case Serializer.COMMAND_AND:
                            c = new AndCommand();
                            break;
                        case Serializer.COMMAND_ARITHMETIC_RIGHT_SHIFT:
                            c = new ArithmeticRightShiftCommand();
                            break;
                        case Serializer.COMMAND_BIT_REVERSING:
                            c = new BitReversingCommand();
                            break;
                        case Serializer.COMMAND_CONDITION:
                            c = new ConditionCommand();
                            break;
                        case Serializer.COMMAND_CONDITIONAL_AND:
                            c = new ConditionalAndCommand();
                            break;
                        case Serializer.COMMAND_CONDITIONAL_OR:
                            c = new ConditionalOrCommand();
                            break;
                        case Serializer.COMMAND_DIVIDE:
                            c = new DivideCommand();
                            break;
                        case Serializer.COMMAND_EQUAL:
                            c = new EqualCommand();
                            break;
                        case Serializer.COMMAND_EXCLUSIVE_OR:
                            c = new ExclusiveOrCommand();
                            break;
                        case Serializer.COMMAND_FUNCTION_CALL:
                            String name = objIn.readUTF();
                            int arguments = objIn.readInt();
                            c = new FunctionCallCommand(name, arguments);

                            break;
                        case Serializer.COMMAND_GREATER_THAN:
                            c = new GreaterThanCommand();
                            break;
                        case Serializer.COMMAND_GREATER_THAN_EQUAL:
                            c = new GreaterThanEqualCommand();
                            break;
                        case Serializer.COMMAND_INCLUSIVE_OR:
                            c = new InclusiveOrCommand();
                            break;
                        case Serializer.COMMAND_LEFT_SHIFT:
                            c = new LeftShiftCommand();
                            break;
                        case Serializer.COMMAND_LESS_THAN:
                            c = new LessThanCommand();
                            break;
                        case Serializer.COMMAND_LESS_THAN_EQUAL:
                            c = new LessThanEqualCommand();
                            break;
                        case Serializer.COMMAND_LOGICAL_RIGHT_SHIFT:
                            c = new LogicalRightShiftCommand();
                            break;
                        case Serializer.COMMAND_MULTIPLY:
                            c = new MultiplyCommand();
                            break;
                        case Serializer.COMMAND_NOT:
                            c = new NotCommand();
                            break;
                        case Serializer.COMMAND_NOT_EQUAL:
                            c = new NotEqualCommand();
                            break;
                        case Serializer.COMMAND_POST_DECREMENT:
                            c = new PostDecrementCommand();
                            break;
                        case Serializer.COMMAND_POST_INCREMENT:
                            c = new PostIncrementCommand();
                            break;
                        case Serializer.COMMAND_PRE_DECREMENT:
                            c = new PreDecrementCommand();
                            break;
                        case Serializer.COMMAND_PRE_INCREMENT:
                            c = new PreIncrementCommand();
                            break;
                        case Serializer.COMMAND_PUSH_STACK:
                            ComputeObject compObj;
                            byte compObjType = objIn.readByte();
                            switch (compObjType) {
                                case Serializer.TYPE_FLOATING_POINT_LITERAL:
                                    compObj = new FloatingPointLiteral(objIn.readDouble());
                                    break;
                                case Serializer.TYPE_INTEGER_LITERAL:
                                    compObj = new IntegerLiteral(objIn.readLong());
                                    break;
                                case Serializer.TYPE_VARIABLE:
                                    compObj = new Variable(objIn.readUTF());
                                    break;
                                default:
                                    throw new IOException("不明な計算オブジェクトの種類です。{" + compObjType + "}");
                            }
                            c = new PushStackCommand(compObj);
                            break;
                        case Serializer.COMMAND_SET_VARIABLE:
                            c = new SetVariableCommand();
                            break;
                        case Serializer.COMMAND_SIGN_REVERSING:
                            c = new SignReversingCommand();
                            break;
                        case Serializer.COMMAND_SUBTRACT:
                            c = new SubtractCommand();
                            break;
                        case Serializer.COMMAND_SURPLUS:
                            c = new SurplusCommand();
                            break;
                        default:
                            throw new IOException("不明なコマンドの種類です。{" + commandType + "}");
                    }
                    cl.add(c);
                }
            } finally {
                objIn.close();
            }
        } finally {
            byteIn.close();
        }

        return cl;
    }

}
