//-----------------------------------------------------------------------
//  This file is part of the Microsoft Robotics Studio Code Samples.
// 
//  Copyright (C) Microsoft Corporation.  All rights reserved.
//
//  $File: SerialIOManager.cs $ $Revision: 1 $
//-----------------------------------------------------------------------

using System;
using Microsoft.Ccr.Core;
using System.IO.Ports;
using Microsoft.Dss.Services.ConsoleOutput;

namespace Microsoft.Robotics.Services.Sensors.SickLRF
{
    internal class BadPacketException : Exception
    {
        int _count;

        public BadPacketException(int count)
            : base("Incorrect Checksum on " + count + " SickLRF packets")
        {
            _count = count;
        }

        public int Count
        {
            get { return _count; }
            set { _count = value; }
        }
    }

    internal class SerialIOManager : CcrServiceBase
    {
        public class Operations : PortSet<Open, Close, SetRate, Send>
        {
        }

        public class ResponsePort : PortSet<Packet, Exception>
        {
        }

        public class Command
        {
            public SuccessFailurePort ResponsePort = new SuccessFailurePort();
        }

        public class Open : Command
        {
        }

        public class Close : Command
        {
        }

        public class SetRate : Command
        {
            public SetRate()
            {
            }

            public SetRate(int rate)
                : base()
            {
                _rate = rate;
            }

            int _rate;
            public int Rate
            {
                get { return _rate; }
                set { _rate = value; }
            }
        }

        public class Send : Command
        {
            public Send()
            {
            }

            public Send(Packet packet)
            {
                _packet = packet;
            }

            Packet _packet;
            public Packet Packet
            {
                get { return _packet; }
                set { _packet = value; }
            }
        }

        class Recv
        {
        }

        public Operations OperationsPort = new Operations();
        public ResponsePort Responses = new ResponsePort();

        Port<Recv> DataPort = new Port<Recv>();
        SerialPort _port;
        PacketBuilder _builder = new PacketBuilder();
        string _portName;
        //int _badCount = 0;
        //string _parent;
        ConsoleOutputPort _console;

        public SerialIOManager(DispatcherQueue dispatcherQueue, string portName)
            : base(dispatcherQueue)
        {
            _portName = portName;
            CreatePort(19200);

            _builder = new PacketBuilder();
            //_builder.Parent = _parent;
            _builder.Console = _console;

            Activate(WaitForOpen());
        }

        public int BaudRate
        {
            get { return _port == null ? 19200 : _port.BaudRate; }
        }

        private void CreatePort(int rate)//̓IɃZTƂ̃|[gJĂƂ
        {
            _port = new SerialPort(_portName, rate, Parity.None, 8, StopBits.One);
            _port.DataReceived += _port_DataReceived;//炩̃f[^󂯎Ăяo\bhiQĵ
            _port.ErrorReceived += _port_ErrorReceived;//G[󂯎Ăяo\bhiPĵ
            _port.DtrEnable = true;
            _port.RtsEnable = true;
        }

        void _port_ErrorReceived(object sender, SerialErrorReceivedEventArgs e)//G[Ɋ֘At\bhEEEiPj
        {
            Responses.Post(new Exception(e.EventType.ToString()));
        }

        void _port_DataReceived(object sender, SerialDataReceivedEventArgs e)//f[^󂯎Ɋ֘At\bhEEEiQj

        {
            DataPort.Post(new Recv());
        }

        void CommandUnavailable(Command cmd)
        {
            cmd.ResponsePort.Post(new Exception("The requested command is not available in the current state"));
        }

        Interleave WaitForOpen()
        {
            return Arbiter.Interleave(
                new TeardownReceiverGroup
                (
                    Arbiter.Receive<Open>(false,OperationsPort,OpenHandler)
                ),
                new ExclusiveReceiverGroup
                (
                ),
                new ConcurrentReceiverGroup
                (
                    Arbiter.Receive(true, DataPort, IgnoreDataHandler),
                    Arbiter.Receive(true, OperationsPort.P1, CommandUnavailable),
                    Arbiter.Receive(true, OperationsPort.P2, CommandUnavailable),
                    Arbiter.Receive(true, OperationsPort.P3, CommandUnavailable)
                ));
        }


        Interleave Connected()
        {
            return Arbiter.Interleave(
                new TeardownReceiverGroup
                (
                    Arbiter.Receive<Close>(false,OperationsPort,CloseHandler)
                ),
                new ExclusiveReceiverGroup
                (
                    Arbiter.Receive<SetRate>(true, OperationsPort, SetRateHandler),
                    Arbiter.Receive<Send>(true, OperationsPort, SendHandler), //ꂪZTւ̃R}hMnh
                    Arbiter.Receive<Recv>(true, DataPort, DataHandler)
                ),
                new ConcurrentReceiverGroup
                (
                    Arbiter.Receive(true, OperationsPort.P0, CommandUnavailable)
                ));
        }

        void IgnoreDataHandler(Recv recv)
        {
            if (_port == null ||
                _port.BytesToRead <= 0)
            {
                return;
            }
            byte[] buffer = new byte[_port.BytesToRead];

            _port.Read(buffer, 0, buffer.Length);
        }

        void DataHandler(Recv recv) //DataPortɓ͂Ă΂֐
        {
            while (DataPort.Test() != null) ;

            if (_port == null ||
                _port.BytesToRead <= 0)
            {
                return;
            }
            string buffer = String.Empty;
            if(buffer.Length > 0)
                buffer.Remove(0,buffer.Length);
            
            buffer = _port.ReadExisting(); //_port̒gbufferɓǂݏo
            //System.Console.WriteLine("Response :" + buffer);

            if (_builder.Add(buffer)) //buffer̒gpacketɐ`
            {
                while (_builder.HasPacket)//L[ɂȂ܂
                {
                    Responses.Post(_builder.RemovePacket());//Responses|[g<Packet>^bZ[WPost
                }
            }
            else //buffer̃Xe[^XEiSCIPverG[jSCIP2.0R}h𑗂
            {
                System.Console.WriteLine("send SCIP2.0");
                OperationsPort.Post(new Send(new Packet("SCIP2.0")));
            }
            /*
            if (_builder.BadPackets > 0)
            {
                if (_builder.BadPackets != _badCount)
                {
                    //
                    // only report bad packets if the number has changed.
                    //
                    _badCount = _builder.BadPackets;
                    Responses.Post(new BadPacketException(_builder.BadPackets));
                }
            }
            else
            {
                _badCount = 0;
            }
             */
        }

        void CloseHandler(Close close)
        {
            _port.DataReceived -= _port_DataReceived;
            _port.ErrorReceived -= _port_ErrorReceived;
            _port.WriteLine("QT\n");
            _port.Close();
            _port = null;

            _builder = new PacketBuilder();
            _builder.Console = _console;

            Recv recv;
            while (DataPort.Test(out recv)) ;

            close.ResponsePort.Post(new SuccessResult());

            Activate(WaitForOpen());
        }

        void SendHandler(Send send)
        {
            try
            {
                //ǂōPacketɃR}h񏑂ŁCPacket̏Send^Ă̊֐ĂԂƃR}hZTɑH
                send.Packet.SendToSensor(_port);   //ŃZTɑ΂ăR}h𑗐MĂ
                send.ResponsePort.Post(new SuccessResult());
            }
            catch (Exception e)
            {
                send.ResponsePort.Post(new Exception(e.Message));
            }
        }

        void SetRateHandler(SetRate setRate)
        {
            try
            {
                _port.Close();
                CreatePort(setRate.Rate);
                _port.Open();

                setRate.ResponsePort.Post(new SuccessResult());
            }
            catch (Exception e)
            {
                setRate.ResponsePort.Post(new Exception(e.Message));
            }
        }

        void OpenHandler(Open open)
        {
            try
            {
                if (_port == null)
                {
                    CreatePort(19200);
                }

                if (_port.IsOpen)
                    _port.Close();

                _port.Open();
                open.ResponsePort.Post(new SuccessResult());
                Activate(Connected());

                DataPort.Clear();
                _port.DiscardInBuffer();
                _port.DiscardOutBuffer();
                OperationsPort.Post(new Send(new Packet("QT")));

            }
            catch (Exception e)
            {
                open.ResponsePort.Post(new Exception(e.Message));

                Activate(WaitForOpen());
            }
        }

        public ConsoleOutputPort Console
        {
            get { return _console; }
            set
            {
                _console = value;
                _builder.Console = value;
            }
        }


    }

}