
# クロスプラットフォーム対応
#$TEST = $DEBUG = ($DEBUG or $TEST)

module Event_Compiler
  
  # イベントソースフォルダ
  SRC_DIR = "src/events"
  
  # イベント強制作成フラグ
  EVENT_CREATE_FORCE = true
  
  # エレメントビジターモジュール
  module Element_Visitor
    @@visitor_method = Hash.new(:visit_default)
    def visit(element)
      method_name = self.class.visitor_method(element.name)
      if element.related? and (not self.class.visitor_method_exist?(element.name))
        method_name = self.class.visitor_method(element.value)
      end
      if Log.debug?
        Log.debug("visit:" + method_name.to_s + " " + element.name)
      end
      return method(method_name).call(element)
    end
    def visit_default(element)
      return nil
    end
    def visitor_method_exist?(name)
      return @@visitor_method.include?(name)
    end
    def visitor_method(name)
      return @@visitor_method[name]
    end
    def add_visitor_method(name,method_name)
      @@visitor_method[name] = method_name
    end
    def self.included(klass)
      klass.extend Element_Visitor
    end
  end
  
  # イベント情報読み込みモジュール
  module Event_Element_Reader
    
    # イベント情報読み込み
    def self.read_data(lines)
      element = nil
      while (not lines.empty?)
        # 一行取得
        line = lines.shift
        # 読み飛ばす行か？
        if skip_line?(line)
          next
        end
        # 最初のエレメントを作成
        element = Event_Element.new.parse_element(line)
        # エレメントに読み込ませる
        element.read_element(lines)
        break
      end
      return element
    end
    
    # 読み飛ばす行か？
    def skip_line?(line)
      if line =~ /^#{COMMENT_PREFIX}/
        return true
      end
      if line.strip == ""
        return true
      end
      return false
    end
    module_function :skip_line?
    
    COMMENT_PREFIX       = '#'   # コメント文字（行頭がこの文字ならコメント）
    EVENT_PREFIX         = '◆'  # イベント頭文字
    RELATED_EVENT_PREFIX = '：'  # 関連イベント頭文字
    EVENT_SEPARATOR      = '：'  # イベント分割文字
    INDENT_STRING        = '　'  # インデント文字
    
    # 各文字列の長さ
    EVENT_PREFIX_LEN         = EVENT_PREFIX.length
    RELATED_EVENT_PREFIX_LEN = RELATED_EVENT_PREFIX.length
    EVENT_SEPARATOR_LEN      = EVENT_SEPARATOR.length
    INDENT_STRING_LEN        = INDENT_STRING.length
    
    # イベントか関連イベントの頭文字を探す
    def find_start_index(data)
      i1 = data.index(EVENT_PREFIX)
      i2 = data.index(RELATED_EVENT_PREFIX)
      # どっちも無い
      if i1.nil? and i2.nil?
        # TODO: 構文エラー
        return 0
      end
      # 片方のみ
      if i1.nil? 
        @related = true
        return i2
      end
      if i2.nil?
        @related = false
        return i1
      end
      # 両方ある場合
      if i1 < i2
        @related = false
        return i1
      else
        @related = true
        return i2
      end
    end
    
    
    # エレメント情報パース
    def parse_element(data)
      # 生データ
      @data = data
      
      # イベントか関連イベントの頭文字を探す
      index_s = find_start_index(@data)

      # インデント数
      @indent = index_s / INDENT_STRING_LEN
      
      # エレメント名の抽出(頭文字から分割文字列まで)
      index_s += EVENT_PREFIX_LEN
      index_e = @data.index(EVENT_SEPARATOR,index_s)
      event_data_name = ""
      if index_e.nil?
        # 分割文字が無い場合は、データ最後まで
        event_data_name = @data[index_s,@data.length]
      else
        event_data_name = @data[index_s,index_e - index_s]
        # 分割されている場合は、エレメントの値も取得
        @value = @data[index_e + EVENT_SEPARATOR_LEN,@data.length]
      end
      @name = event_data_name.strip
      return self
    end
    
    # エレメント作成
    def create_element(data)
      element = Event_Element.new
      element.parse_element(data) # データをパース
      return element
    end
    
    # エレメント情報読み込み
    def read_element(lines)
      # 空なら終わり
      return self if lines.empty?
      element = nil
      # 空になるまで繰り返し
      while (not lines.empty?)
        # 一行取得
        line = lines.shift
        # 読み飛ばす行か？
        if skip_line?(line)
          next
        end
        
        # イベントエレメントを作成
        element = create_element(line)
        
        # 作成したエレメントが自分のインデントより大きい
        if element.indent > @indent
          # 子エレメントに追加
          @child_elements.push element
          # 子エレメントに読み込ませる（最後に作成したエレメントが返却される）
          element = element.read_element(lines)
        end
        # 作成したエレメントが自分のインデントより小さい
        if element.indent < @indent
          #　作成したエレメントを返却
          return element
        end
        # 作成したエレメントが関連イベント
        if element.related?
          # 関連として保存
          @related_elements.push element
        else
          # 次のイベントとして保存
          @next_element = element
          # このエレメントの読み込みは終わり
          break
        end
      end
      if element.nil?
        return nil;
      end
      # 最後に作成したエレメントに続きを読み込ませる
      return element.read_element(lines)
    end
    
  end
  
  # イベントエレメント
  class Event_Element
    include Event_Element_Reader
    
    attr_reader :data   # 生データ
    attr_accessor :name   # エレメント名
    attr_accessor :value  # エレメント値
    attr_reader :indent # インデント
    attr_reader :next_element     # 次のエレメント
    attr_reader :related_elements # 関連エレメントリスト
    attr_reader :child_elements         # 子エレメントリスト
    
    def initialize
      @data = ""
      @name = ""
      @value = ""
      @indent = 0
      @next_element = nil
      @related_elements = []
      @child_elements = []
      @related = false # 関連エレメントフラグ
    end
    
    # 関連エレメントの場合 true
    def related?
      return @related
    end
    
    # Visitorパターン用 Acceptorメソッド
    def accept( visitor )
      return visitor.visit(self)
    end
    
    # パラメーター変換
    def parameters
      return @value.split(/,|、/).collect {|e|e.strip}
    end
    
  end
  
  module_function
  
  # TODO クロスプラットフォーム対応予定
  
  # マップデータロード
  def _load_map_data(map_id)
    filename = sprintf("Data/Map%03d.rxdata", map_id)
    if Log.debug?
      Log.debug("_load_map_data #{filename}")
    end
    map_data = load_data(filename)
    # バックアップ作成
    if FileTest.directory?("Backup")
      backup_filename = filename.dup
      backup_filename[/^Data/] = "Backup"
      backup_index = 0
      loop do
        backup_index += 1
        unless FileTest.file?(backup_filename + '.' + backup_index.to_s)
          save_data(map_data,backup_filename + '.' + backup_index.to_s)
          if Log.debug?
            Log.debug(backup_filename + '.' + backup_index.to_s)
          end
          break
        end
      end
    else
      Log.error("Backup ディレクトリがありません。")
    end
    return map_data
  end
  
  # マップデータセーブ
  def _save_map_data(map_data, map_id)
    filename = sprintf("Data/Map%03d.rxdata", map_id)
    if Log.debug?
      Log.debug("_save_map_data #{filename}")
    end
    return save_data(map_data,filename)
  end
  
  # イベントソースコンパイル
  def compile
    return unless $TEST # テスト実行では無い場合は、処理しない。
    
    # マップディレクトリの各ファイルをコンパイル
    compile_directory(SRC_DIR)
  end
  
  # ディレクトリコンパイル
  def compile_directory(dir,map_id = 0)
    unless FileTest.directory?(dir)
      Log.error("compile_directory #{dir} ディレクトリがありません。")
      return
    end
    event_files = Dir.glob(dir + '/*.txt')
    # unless event_files.empty?
      # event_file_mtime = event_files.collect {|filename| File.open(filename).mtime()}.max
      # map_data_file_mtime = File.open(sprintf("Data/Map%03d.rxdata", map_id)).mtime()
      # if event_file_mtime < map_data_file_mtime
        # return
      # end
    # end
    Log.info("compile_directory #{dir}")
    
    save = false
    map_data = nil
    if map_id != 0
      # マップデータ読み込み
      map_data = _load_map_data(map_id)
    end
    
    # イベントファイルのコンパイル（初期化）
    event_files.each do |filename|
      unless map_data.nil?
        compile_file_init(map_data,filename)
      end
    end
    Dir.glob(dir + '/*') do |subdir|
      case File.extname(subdir)
      when "" # ディレクトリ
        filename = File.basename(subdir)
        if filename =~ /^[0-9]+$/
          compile_directory(subdir,filename.to_i)
        end
      when ".txt" # イベントファイル
        # ファイルのテキストデータをイベント化
        unless map_data.nil?
          result = compile_file(map_data,subdir)
          save |= result
          if Log.debug?
            Log.debug("compile_file #{result} #{save}")
          end
        end
      end
    end
    
    # マップデータ書き込み
    if save
      _save_map_data(map_data, map_id)
    end
  end
  
  def compile_file_info(filename)
    name = File.basename(filename,".txt")
    # 座標をファイル名から取得
    if name =~ /\(([0-9]+)-([0-9]+)\)/
      x = $1.to_i
      y = $2.to_i
      name[/\(([0-9]+)-([0-9]+)\)/] = ""
    end
    # IDと座標をファイル名から取得
    if name =~ /\(([0-9]+)-([0-9]+)-([0-9]+)\)/
      x = $1.to_i
      y = $2.to_i
      event_id = $3.to_i
      name[/\(([0-9]+)-([0-9]+)-([0-9]+)\)/] = ""
    end
    name = name.strip.toutf8
    event_id = 0 if event_id.nil?
    return x,y,name,event_id
  end
  
  # ファイルコンパイル初期処理
  def compile_file_init(map_data,filename)
    Log.info("compile_file_init #{filename}")
    
    x,y,name,event_id = compile_file_info(filename)
    unless x.nil? or y.nil?
      # 強制作成
      if EVENT_CREATE_FORCE
        # 同じ座標のイベントがある場合は、それを削除
        map_data.events.select {|_id,_e| _e.x == x and _e.y == y}.each do |id,e|
          map_data.events.delete(e.id)
          if Log.debug?
            Log.debug("delete #{e.name} x=#{e.x} y=#{e.y} event_id = #{e.id}")
          end
        end
        if event_id != 0
          # 同じ座標ではなく同じIDのイベントがある場合は、そのIDと交換
          id,e = map_data.events.find {|_id,_e| _e.id == event_id and not (_e.x == x and _e.y == y)}
          if not e.nil?
            id_list = map_data.events.keys
            for i in 1 .. 999
              unless id_list.include? i
                e.id = i
                map_data.events[i] = e
                break
              end
            end
          end
        end
      else
        # 同じ座標のイベントがある場合は、作成しない
        if map_data.events.any? {|_id,_e| _e.x == x and _e.y == y}
          return false
        end
        if event_id != 0
          # 同じIDのイベントがある場合は、作成しない
          if map_data.events.any? {|_id,_e| _e.id == event_id}
            return false
          end
        end
      end
      # イベントIDが決まってない場合
      if event_id == 0
        if map_data.events.empty?
          event_id = 1
        else
          id_list = map_data.events.keys
          for i in 1 .. 999
            unless id_list.include? i
              event_id = i
              break
            end
          end
        end
      end
      if Log.debug?
        Log.debug("event.name='#{name}' x=#{x} y=#{y} event_id=#{event_id}")
      end
      event = RPG::Event.new(x,y)
      # イベントID
      event.id = event_id
      # イベント名      
      event.name = name
      # 新規に作成したイベントはページを空に
      event.pages = []
      # マップにイベントを追加
      map_data.events[event.id] = event
      return true
    end
    return false
  end
  
  # ファイルコンパイル
  def compile_file(map_data,filename)
    Log.info("compile_file #{filename}")
    
    # イベント名と座標をファイル名から取得
    x,y,name,event_id = compile_file_info(filename)
    unless x.nil? or y.nil?
      # 同じ座標のイベントを検索
      id,e = map_data.events.find {|_id,_e| _e.x == x and _e.y == y}
      if e.nil?
        Log.error("イベントがありません。x=#{x} y=#{y}")
        return false
      end
      if Log.debug?
        Log.debug("event.name='#{e.name}' x=#{x} y=#{y} event_id=#{e.id}")
      end
      event = RPG::Event.new(x,y)
      # イベントID
      event.id = e.id
      if e.name.nil? or e.name.empty?
        e.name = "EV%03d" % e.id
      end
      # イベント名      
      event.name = e.name
      # イベントコンパイル
      compile_event(map_data,event,filename)
      # マップにイベントを追加
      map_data.events[event.id] = event
      return true
    end
    return false
  end
  
  # ファイルのロード
  def load_compile_event_file(filename)
    path = File.dirname(filename)
    lines = []
    IO.foreach(filename) do |line|
      line = line.toutf8().strip
      if line =~ /include\((.+?)\)/
        lines.concat(load_compile_event_file(path + '/' + $1.strip))
      end
      lines.push line
    end
    return lines
  end
  
  # イベントコンパイル
  def compile_event(map_data,event,filename)
    # 区切り文字 初期値
    boundary = "---"
    
    # ページ並び順
    pages = []
    
    page_data = {}
    data = []
    page_data['default'] = data
    
    # 読み込み
    lines = load_compile_event_file(filename)
    lines.each do |line|
     (key,value) = line.split(/\s*=\s*/)
      if key == "boundary" # 区切り文字
        boundary = value
        next
      elsif key == "pages" # ページ並び
        pages = value.split(/\s*,\s*/)
        next
      elsif key == "page.name" # ページ名
        page_data[value] = data
        next
      elsif key == "event.name" # イベント名
        event.name = value
        next
      elsif boundary == line
        data = []
        next
      end
      data.push line
    end
    
    event.pages = []
    
    # デフォルトページテンプレート
    default_page = RPG::Event::Page.new
    # デフォルトページテンプレート作成
    set_header(default_page,page_data['default'])
    
    # 各ページを並び順に作成
    pages.each do |name|
      # デフォルトページをディープコピー
      page = Marshal.load(Marshal.dump(default_page))
      # ページを作成
      if Log.debug?
        Log.debug("compile_page: #{name}")
      end
      if not page_data.include?(name)
        Log.error("compile_page: page_data not found. page.name=#{name}")
        next
      end
      compile_page(map_data,event,page,page_data[name])
      # イベントにページを追加
      event.pages.push(page)
      # 先頭に注釈の追加
      page.list.unshift RPG::EventCommand.new(108, 0, ["#{name}"])
    end
    
  end
  
  # ページ毎にコンパイル
  def compile_page(map_data,event,page,data)
    set_header(page,data)
    # TODO イベントリスト作成
    data.delete_if {|l| l == ""}
    element = Event_Element_Reader.read_data(data)
    page.list = element.accept(Event_Element_Visitor.new(map_data,event,page))
  end
  
  # ヘッダーを設定
  def set_header(page,data)
    while (not data.empty?)
      line = data.shift
      break if line == "" # 空白行までがヘッダー
      (key,value) = line.split(/\s*=\s*/,2)
      if value =~ /^"(.*)"$/
        value = $1
      end
      if key =~ /condition\.(.+)$/
        set_condition(page.condition,$1,value)
      elsif key =~ /graphic\.(.+)$/
        set_graphic(page.graphic,$1,value)
      else
        set_property(page,key,value)
      end
    end
  end
  
  STRING_PARAMETERS = {
    'trigger' => {
      '決定ボタン'        => 0,
      'プレイヤーから接触' => 1,
      'イベントから接触'   => 2,
      '自動実行'          => 3,
      '並列処理'          => 4,
    },
  }
  
  # プロパティを動的に設定
  def set_property(obj,key,value)
    # getter があるか？
    if obj.class.method_defined?(key)
      # 実際に get してみる
      case obj.method(key).call
      when Numeric # 数値の場合
        if value =~ /^[0-9]+$/
          # 数値にして setter を呼ぶ
          obj.method(key + '=').call(value.to_i)
        else
          # 数値に対して文字列が設定されている場合
          if STRING_PARAMETERS.include?(key)
            if STRING_PARAMETERS[key].include?(value)
              obj.method(key + '=').call(STRING_PARAMETERS[key][value])
            end
          end
        end
      when String # 文字列の場合
        # 文字列にして setter を呼ぶ
        obj.method(key + '=').call(value.to_s)
      when TrueClass
        # Booleanにして setter を呼ぶ
        obj.method(key + '=').call(value == "true")
      when FalseClass
        # Booleanにして setter を呼ぶ
        obj.method(key + '=').call(value == "true")
      end
    end
  end
  
  # グラフィックを設定
  def set_graphic(graphic,key,value)
    set_property(graphic,key,value)
  end
  
  # 条件を設定
  def set_condition(condition,key,value)
    case key
    when "switch1"
      if Switches.exist?(value)
        condition.switch1_valid = true
        condition.switch1_id = Switches.index(value)
        return
      else
        Log.error "スイッチ「#{value}」は、ありません"
        return
      end
    when "switch2"
      if Switches.exist?(value)
        condition.switch2_valid = true
        condition.switch2_id = Switches.index(value)
        return
      else
        Log.error "スイッチ「#{value}」は、ありません"
        return
      end
    when "variable"
      if Variables.exist?(value)
        condition.variable_valid = true
        condition.variable_id = Variables.index(value)
        return
      else
        Log.error "変数「#{value}」は、ありません"
        return
      end
    when "self_switch"
      if ['A','B','C','D'].include? value
        condition.self_switch_valid = true
        condition.self_switch_ch = value
        return
      else
        Log.error "セルフスイッチ「#{value}」は、ありません"
        return
      end
    else
      set_property(condition,key,value)
    end
  end
  
end
